# coding: utf8
from __future__ import absolute_import

import requests

from sandbox import sdk2
from sandbox.common import errors, utils
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk import environments


class AdvisorDeployToQloud(sdk2.Task):
    '''
    - Deploy docker image to Qloud (https://platform.yandex-team.ru)
    '''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        ya_package_task_id = sdk2.parameters.Task(
            'YA_PACKAGE Task',
            task_type="YA_PACKAGE",
            required=True
        )

        with sdk2.parameters.Group('Qloud Config') as qloud_block:
            qloud_service_name = sdk2.parameters.String('Qloud Service', required=True)
            qloud_project_name = sdk2.parameters.String('Qloud Project', required=True)
            qloud_environment_name = sdk2.parameters.String('Qloud Environment', required=True)
            qloud_component_name = sdk2.parameters.String('Qloud Component', required=True)
            qloud_vault_name = sdk2.parameters.String('Qloud token vault name', required=False)  # TODO: drop after switch to YAV
            qloud_token_id = sdk2.parameters.YavSecret('Qloud token Yav secret id', required=False)

    def on_execute(self):
        ya_package_task = self.Parameters.ya_package_task_id
        with self.memoize_stage.get_resource:
            if ya_package_task.status not in set(utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)):
                raise sdk2.WaitTask(ya_package_task.id, set(utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)))

        with self.memoize_stage.deploy_to_qloud:
            docker_image_url = ya_package_task.Context.output_resource_version
            if not docker_image_url:
                raise errors.TaskFailure("Resource doesn't have 'resource_version': {}".format(self.Parameters.docker_resource_id))

            environment_id = ".".join([self.Parameters.qloud_service_name, self.Parameters.qloud_project_name, self.Parameters.qloud_environment_name])
            if self.Parameters.qloud_token_id:
                token = self.Parameters.qloud_token_id.data()["QLOUD_OAUTH_TOKEN"]
            else:
                token = sdk2.Vault.data(self.owner, self.Parameters.qloud_vault_name)

            try:
                self.__update_qloud_component(token, environment_id, self.Parameters.qloud_component_name, docker_image_url)
            except (requests.ConnectionError, requests.HTTPError) as err:
                raise errors.TaskFailure("Deploy failed: {}".format(repr(err)))

    def __get_image_hash(self, docker_image_url):
        url, tag = docker_image_url.split(':')
        r = requests.get('https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={url}&tag={tag}'.format(
            url=url,
            tag=tag,
        ))
        r.raise_for_status()
        return r.json()['hash']

    def __update_qloud_component(self, token, environment_id, component, docker_image_url):
        QLOUD_API_URL = 'https://platform.yandex-team.ru/api/v1/environment'
        headers = {'authorization': 'OAuth ' + token}

        r = requests.get(
            '{}/{}/{}'.format(QLOUD_API_URL, 'current', environment_id),
            headers=headers,
        )
        r.raise_for_status()
        env_version = r.json()['version']
        image_hash = self.__get_image_hash(docker_image_url)
        postr = requests.post(
            '{}/{}/{}/{}'.format(QLOUD_API_URL, 'fast-deploy', environment_id, env_version),
            headers=headers,
            json=[
                {
                    "componentName": component,
                    "properties": {
                        "repository": docker_image_url,
                        "hash": image_hash,
                    }
                },
            ]
        )
        postr.raise_for_status()
