# coding: utf8
from __future__ import absolute_import
import os
import tempfile

import requests

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.porto.common.resource_types import PORTO_LAYER


class JafarBuildNirvanaLayer(sdk2.Task):
    '''
    - Build base Nirvana layer for Jafar tasks
    - Save layer resource id to Vault for Nirvana workflows
    '''
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_svn_url = sdk2.parameters.String(
            'Svn url for arcadia', required=True)
        parent_layer = sdk2.parameters.String(
            'Parent layer', required=True)

    def on_execute(self):
        with self.memoize_stage.get_requirements:
            if '@' in self.Parameters.arcadia_svn_url:
                arcadia_svn_path, revision = \
                    self.Parameters.arcadia_svn_url.split('@')
            else:
                arcadia_svn_path, revision = \
                    self.Parameters.arcadia_svn_url, None

            local_path = tempfile.mkdtemp()
            Arcadia.checkout(arcadia_svn_path, local_path,
                             revision=revision, depth="files")
            pip_reqs = open(os.path.join(local_path, 'requirements.txt')).\
                read()
            deb_reqs = open(os.path.join(local_path, 'deb_requirements.txt')).\
                read()
            setup = open(os.path.join(local_path, 'nirvana_setup.sh')).read().\
                format(pip_requirements=pip_reqs, deb_requirements=deb_reqs)
            self.Context.setup_url = requests.post(
                'https://paste.yandex-team.ru',
                data={'text': setup, 'syntax': 'plain'}
            ).url + '/text'

        with self.memoize_stage.build_layer:
            build_porto_layer_task = sdk2.Task['BUILD_PORTO_LAYER'](
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                kill_timeout=3600,
                description='Building layer for Nirvana Jafar operation',
                layer_name=str(Arcadia.get_revision(self.Parameters.arcadia_svn_url)),
                script=self.Context.setup_url,
                parent_layer=self.Parameters.parent_layer,
                merge_layers=True
            )

            build_porto_layer_task.enqueue()
            self.Context.build_porto_layer_task_id = build_porto_layer_task.id
            raise sdk2.WaitTask(
                [build_porto_layer_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True)

        with self.memoize_stage.set_ttl:
            build_porto_layer_task = self.find(id=self.Context.ya_package_task_id).first()
            if build_porto_layer_task.status != ctt.Status.SUCCESS:
                raise TaskFailure('Building jafar porto layer task failed')

            porto_layer = sdk2.resource.Resource.find(
                task=build_porto_layer_task,
                type=PORTO_LAYER
            ).first()

            if porto_layer.state != ctr.State.READY:
                raise TaskFailure('Jafar porto layer is not ready')

            porto_layer.ttl = 'inf'
            porto_layer.released = True
