# coding: utf-8

import os
import json
import logging
from datetime import date, timedelta  # noqa

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.yphone.launcher.parameters import YTParameters
from sandbox.projects.yphone.launcher.resource_types import LauncherAmmo


DAYS_AGO = 2
MAX_AMMO_COUNT = 12000

logger = logging.getLogger('main')
logger.setLevel(logging.DEBUG)


class MongoDBParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('MONGODB parameters'):
        user_vault = sdk2.parameters.String('mongodb user name vault', required=True, default='rt_mongodb_user_1')
        password_vault = sdk2.parameters.String('mongodb password vault name', required=True,
                                                default='rt_mongodb_password_1')
        replica_password_vault = sdk2.parameters.String('mongodb replica password vault name', required=True,
                                                        default='rt_mongodb_replica_password_1')


class LauncherGenerateAmmo(sdk2.Task):
    """ Generates ammo for Launcher loadtests """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('pymongo'),
        )
        disk_space = 2 * 1024  # 2GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60  # 1 hour
        yt_params = YTParameters
        mongodb = MongoDBParameters  # TODO: drop after switch to YAV
        mongodb_secret = sdk2.parameters.YavSecret('MongoDB credentials Yav secret id', required=False)

        container = sdk2.parameters.Container(
            "Environment container resource",  # (LXC Container: Launcher Ammo Generation Environment)
            required=True
        )

        with sdk2.parameters.Group('Ammo parameters'):
            days_ago = sdk2.parameters.Integer(
                'Number of days ago from today',
                default=DAYS_AGO
            )

            max_ammo_count = sdk2.parameters.Integer(
                'Max number of ammo',
                default=MAX_AMMO_COUNT
            )

    def on_execute(self):
        from tool import get_output_file_names, get_date_str

        logger.info('Ammo generating started.')

        date_str = (date.today() - timedelta(days=self.Parameters.days_ago)).isoformat()

        base_dir = os.path.dirname(os.path.realpath(__file__))
        if self.Parameters.yt_params.yt_token:
            yt_token = self.Parameters.yt_params.yt_token.data()["YT_TOKEN"]
        else:
            yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_params.token_vault)

        script_args = {
            'yt_token': yt_token,
            'days_ago': self.Parameters.days_ago,
            'max_ammo_count': self.Parameters.max_ammo_count,
        }
        script_args.update(get_output_file_names(get_date_str(self.Parameters.days_ago)))
        logger.info('on_execute: Current directory: %s', os.getcwd())

        try:
            result = run_process(
                ['/usr/bin/python', os.path.join(base_dir, 'tool.py')],
                environment=dict(SCRIPT_ARGS=json.dumps(script_args)),
                work_dir=os.getcwd(),
                log_prefix='make_ammo_and_uuid_files',
                shell=True
            )
            if result.returncode:
                raise Exception('Cannot make ammo. retcode %d' % result.returncode)
        except Exception as exc:
            raise SandboxTaskFailureError(exc)

        LauncherAmmo(self, 'Launcher ammo',
                     script_args['ammo_filename'],
                     for_date=date_str,
                     ammo_count=script_args['max_ammo_count'])

        from copy_uuids import copy_uuids

        if self.Parameters.mongodb_secret:
            environ = self.Parameters.mongodb_secret.data()
        else:
            vaults = self.Parameters.mongodb
            # passing sensitive data
            environ = {
                'MONGO_USER': sdk2.Vault.data(vaults.user_vault),
                'MONGO_PASSWORD': sdk2.Vault.data(vaults.password_vault),
                'MONGO_REPLICA_PASSWORD': sdk2.Vault.data(vaults.replica_password_vault),
            }

        try:
            copy_uuids(script_args['uuid_filename'], environ=environ)
            logger.error('Copy user profiles has been successful!')
        except Exception as exc:
            logger.error('Copy user profiles failed!')
            from traceback import format_exc
            logger.error(format_exc())
            raise SandboxTaskFailureError(exc)

        logger.info('Ammo generation has finished.')
