# -*- coding: utf-8 -*-

from datetime import date, timedelta
import logging
import logging.handlers
import time
import re

import requests

from sandbox import sdk2

from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt

from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

from sandbox.projects.yphone.launcher.GenerateAmmo import LauncherGenerateAmmo
from sandbox.projects.yphone.launcher.resource_types import (
    LauncherAmmo,
)

QLOUD_API_TEMPLATE = 'https://platform.yandex-team.ru/api/v1/environment/overview/{environment}'

TANK_FORMAT = '{host}:8083'
QLOUD_ENV_PATH_FORMAT = '{project}.{application}.{environment}'


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'launcher_metashooting.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


class LauncherRunShooting(sdk2.Task):
    """ Task for shooting on Launcher """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 300  # MB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 60
        description = 'Regression of Launcher project'

        with sdk2.parameters.Group('Qloud parameters'):
            qloud_oauth_toket = sdk2.parameters.String('OAuth token', default='rt_qloud_token_1', required=False)  # TODO: drop after switch to YAV
            qloud_token = sdk2.parameters.YavSecret('Qloud token Yav secret id', required=False)
            qloud_project = sdk2.parameters.String('project', default='yandex-phone', required=True)
            qloud_application = sdk2.parameters.String('application', default='launcher', required=True)
            qloud_environment = sdk2.parameters.String('environment', default='regression', required=True)
            qloud_component_tank = sdk2.parameters.String('component tank', default='tank', required=True)
            qloud_component_target = sdk2.parameters.String('component target', default='launcher', required=True)
            qloud_target_host_index = sdk2.parameters.Integer('target host index', default=0, required=True,
                                                              description='if defined several targets index helps '
                                                                          'to select right host')

        with sdk2.parameters.Group('Startrek parameters'):
            ticket = sdk2.parameters.String(
                'ticket for shooting',
                description='The shoot will be performed in this ticket. The report will be sent as a comment',
                default='ADVISOR-247',
                required=True
            )
            token = sdk2.parameters.String(
                'token for comment',
                description='Task owner must have permissions to access this token in Sandbox Vault.',
                default='rt_startrek_token_1',
                required=True
            )

        with sdk2.parameters.Group('Ammo parameters'):
            with sdk2.parameters.RadioGroup("Ammo generation") as ammo_gen_group:
                ammo_gen_group.values.use_latest = None
                ammo_gen_group.values.generate_if_2_days_ago_absent = None
                ammo_gen_group.values.force_generate = ammo_gen_group.Value(default=True)

            requests_count = sdk2.parameters.Integer(
                'Number of requests to sample',
                default=12000
            )

            generate_ammo_container = sdk2.parameters.Container(
                "Environment container resource for GENERATE_AMMO",  # (LXC Container: Launcher Ammo Generation Environment)
                required=True
            )
            mongodb_secret = sdk2.parameters.YavSecret('MongoDB credentials Yav secret id', required=False)
            yt_token = sdk2.parameters.YavSecret('YT token Yav secret id', required=False)

    class Context(sdk2.Task.Context):
        shoot_data = {
            'lunapark_link': '',
            'desc': u'Shoot on Launcher: {}',
            'config_arc_path': 'sandbox/projects/yphone/launcher/config/tank.config.1.yaml',
            'ammo': None,
            'subtask': None,
            'subtask_ids': [],
            'qloud_hosts': None,
        }
        report_tasks = []

    def generate_ammo(self):
        ammo_generation_subtask = LauncherGenerateAmmo(
            self,
            description='Generate Launcher ammo task {}'.format(self.id),
            owner=self.owner,
            container=self.Parameters.generate_ammo_container,
            mongodb_secret=self.Parameters.mongodb_secret,
            yt_token=self.Parameters.yt_token,
        ).enqueue()
        raise sdk2.WaitTask([ammo_generation_subtask.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                            wait_all=True, timeout=14400)

    def _get_qloud_hosts(self,
                         components,  # ['launcher', 'tank']
                         environment  # 'yandex-phone.launcher.regression'
                         ):
        log = logger()
        if self.Parameters.qloud_token:
            token = self.Parameters.qloud_token.data()["QLOUD_OAUTH_TOKEN"]
        else:
            token = sdk2.Vault.data(self.Parameters.qloud_oauth_toket)
        headers = {"Authorization": "OAuth %s" % token}
        url = QLOUD_API_TEMPLATE.format(environment=environment)
        log.info('_get_qloud_hosts args: url=%s, headers=%s', url, headers)
        response = requests.get(url=url, headers=headers)
        try:
            data = response.json()
        except Exception as e:
            log.error('error: cannot get hosts: %s', e)
            log.error('response: %s (%s)', response.content, response.status_code)
            raise errors.TaskError("Cannot get target & tank hosts for shooting")

        component_hosts = {
            component: [instance['host'] for instance in data['components'][component]['instances'].itervalues()]
            for component in components
        }
        for component, hosts in component_hosts.items():
            if len(hosts) < 1:
                raise errors.TaskError('component=%s must have at least one host' % component)
        return component_hosts

    def start_shooting(self, save_task_id, shoot_type, add_options=''):
        log = logger()
        params = self.Parameters
        environment_path = QLOUD_ENV_PATH_FORMAT.format(
            project=params.qloud_project,
            application=params.qloud_application,
            environment=params.qloud_environment,
        )
        components = [params.qloud_component_target, params.qloud_component_tank]

        shoot_data = self.Context.shoot_data
        if shoot_data.get('qloud_hosts') is None:
            shoot_data['qloud_hosts'] = self._get_qloud_hosts(components, environment_path)
        log.info('components=%s hosts=%s', components, shoot_data['qloud_hosts'])
        tanks = map(lambda host: TANK_FORMAT.format(host=host), shoot_data['qloud_hosts'][params.qloud_component_tank])
        target_hosts = shoot_data['qloud_hosts'][params.qloud_component_target]

        if len(target_hosts) <= params.qloud_target_host_index < 0:
            raise errors.TaskError('wrong qloud_target_host_index=%d. it must be in range [0, len(hosts)=%d)' %
                                   (params.qloud_target_host_index, len(target_hosts)))

        target_host = target_hosts[params.qloud_target_host_index]
        target_option = '-o phantom.address={} '.format(target_host)

        subtask_shoot = ShootViaTankapi(
            self,
            description=shoot_data['desc'].format(shoot_type),
            ammo_source='resource',
            ammo_resource=shoot_data['ammo'],
            config_source='arcadia',
            config_arc_path=shoot_data['config_arc_path'],
            config_add_parameters=target_option + add_options,
            use_monitoring=False,
            tanks=tanks,
            kill_timeout=3600
        ).enqueue()
        log.info('Subtask with shooting is started')
        shoot_data['subtask'] = subtask_shoot.id
        if save_task_id:
            shoot_data['subtask_ids'].append(subtask_shoot.id)
        raise sdk2.WaitTask(
            [subtask_shoot.id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=4800
        )

    def start_reporting(self, report_type):
        shoot = self.Context.shoot_data
        logger().info('Start reporting for, shoot %s (task %s)',
                      shoot['lunapark_link'], shoot['subtask'])

        if shoot['lunapark_link']:
            logger().info('Lunapark link for shoot is %s', shoot['lunapark_link'])
            shoot_id = re.findall(r'\d+', shoot['lunapark_link'])[0]
            subtask_report = LoadTestResults(
                self,
                shoot_id=shoot_id,
                report_type=report_type,
                send_comment=True,
                ticket_id=self.Parameters.ticket,
                st_token_name=self.Parameters.token,
                additional_message='{}\nhttps://sandbox.yandex-team.ru/task/{}'.format(
                    self.Parameters.description, self.id)
            ).enqueue()
            self.Context.report_tasks.append(subtask_report.id)

    def on_execute(self):
        def take_freshest_ammo():
            resource = LauncherAmmo.find(
                state=ctr.State.READY,
            ).order(-sdk2.Resource.id).first()
            if not resource:
                raise errors.TaskError('No ammo found')
            else:
                sdk2.ResourceData(resource)
                return resource

        shoot_data = self.Context.shoot_data
        if shoot_data['subtask']:
            shoot_data['lunapark_link'] = sdk2.Task[shoot_data['subtask']].Parameters.lunapark_link

        # Step 1: Generate ammo
        with self.memoize_stage.generate_ammo:
            the_2_day_ago = (date.today() - timedelta(days=2)).strftime('%Y-%m-%d')

            if self.Parameters.ammo_gen_group == 'generate_if_2_days_ago_absent':
                ammo = LauncherAmmo.find(
                    state=ctr.State.READY,
                    attrs={
                        'for_date': the_2_day_ago,
                    },
                ).order(-sdk2.Resource.id).first()
                logger().info('rt: found ammo resource for_date=%s', ammo)

                if not ammo:
                    self.generate_ammo()
                else:
                    shoot_data['ammo'] = ammo.id
            elif self.Parameters.ammo_gen_group == 'force_generate':
                self.generate_ammo()

        if not shoot_data['ammo']:
            shoot_data['ammo'] = take_freshest_ammo().id
        logger().info('Ammo with urls is %s', shoot_data['ammo'])

        # Step 2: Run shooting
        with self.memoize_stage.warmup:
            options = '-o phantom.rps_schedule="const(10,10m)" -o meta.job_name="Launcher regression (warmup)"'
            self.start_shooting(save_task_id=False, shoot_type='warmup', add_options=options)

        with self.memoize_stage.shoot_const:
            options = '-o meta.component=2578 -o meta.job_name="Launcher regression (const)"'
            self.start_shooting(save_task_id=True, shoot_type='const', add_options=options)
        with self.memoize_stage.report_const:
            self.start_reporting(report_type='const')

        with self.memoize_stage.shoot_line:
            options = ('-o meta.component=2577 -o phantom.rps_schedule="line(1,300,10m)" '
                       '-o rcassert.pass="21 22 23" -o meta.job_name="Launcher regression (line)"')
            self.start_shooting(save_task_id=True, shoot_type='line', add_options=options)
        with self.memoize_stage.report_line:
            self.start_reporting(report_type='line')

        # Step 3: Make a report
        with self.memoize_stage.reporting:
            if self.Context.report_tasks:
                raise sdk2.WaitTask(
                    self.Context.report_tasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )

        # Step 4: Save status
        log = logger()
        log.info('subtask_ids={}'.format(self.Context.shoot_data))
        for task_id in self.Context.shoot_data['subtask_ids']:
            task = sdk2.Task[task_id]
            msg = 'Subtask {} [{}] finished with status {}'.format(task.type, task.id, task.status)
            log.info(msg)
            if task.status not in ctt.Status.Group.SUCCEED:
                raise errors.TaskFailure(msg)

        log.info('Task successfully finished!')
