# -*- coding: utf-8 -*-

import logging
import os

import yaml

import sandbox.common.types.task as ctt
import sandbox.projects.yql.YQLDeployUDF as ydu
from sandbox import common
from sandbox import sdk2
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.build.YaMake2 import ArcadiaProjectBuildParameters


class YqlBatchDeployUDF(sdk2.Task):
    """
        Сборка, заливка в S3 и подгрузка (индексация) в YQL пользовательских UDF.
        Запускает в параллель несколько YQL_DEPLOY_UDF
        Позвляет передать список targets или указать all для сборки всего, что перечислено в yql/udfs/yql_deploy_udfs.yaml
    """

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = build_parameters.ArcadiaUrl()
        use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default_value=True)
        thinlto = ArcadiaProjectBuildParameters.thinlto(default=True)
        host_platform_flags = ArcadiaProjectBuildParameters.host_platform_flags(
            default="--target-platform-flag=CFLAGS='-fvisibility=hidden'  --target-platform-flag=CFLAGS='-fno-omit-frame-pointer'")
        definition_flags = ArcadiaProjectBuildParameters.definition_flags()
        targets = sdk2.parameters.String("Paths to dirs with ya.make and YQL_UDF targets or 'all' to deploy all",
                                         required=True)
        with sdk2.parameters.RadioGroup("Where to deploy") as where_to_deploy:
            for k in ydu.uploaders.keys():
                where_to_deploy.values[k] = where_to_deploy.Value(value=k)
        fail_fast = sdk2.parameters.Bool(
            "Fail immediately after first unsuccessful UDF upload. If false - try to upload the rest",
            default=False)
        req_disk_space = sdk2.parameters.Integer("Required disk space for subtasks in Mib", default=40000)
        req_ram = sdk2.parameters.Integer("Required RAM for subtasks in Mib", default=1024)
        req_cpu = sdk2.parameters.Integer("Required CPU cores for subtasks", default=0)

    def on_execute(self):
        from sandbox.projects.yql.YQLDeployUDF import YqlDeployUDF
        # get config
        conf_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'udfs_config.yaml')
        with open(conf_path, 'r') as stream:
            conf = yaml.safe_load(stream)
        assert conf

        targets = self.Parameters.targets
        if targets == 'all':
            targets = [';'.join(u for u in udf) for udf in conf['all']]
        else:
            # split targets by batches according to conf
            ts = set(targets.split(';'))
            assert ts.isdisjoint(set(conf['trusted']))
            new_targets = []
            for udf in conf['all']:
                intersect = set(udf).intersection(ts)
                if intersect:
                    ts.difference_update(intersect)
                    new_targets.append(';'.join(u for u in intersect))
            if ts and (len(ts) > 1 or '' not in ts):
                new_targets.append(';'.join(u for u in ts))
            targets = new_targets

        with self.memoize_stage.run_subtasks:
            self.Context.subtask_udfs = {}
            for udf in targets:
                description = 'Deploy {} to {} environment from {}'.format(udf, self.Parameters.where_to_deploy,
                                                                           self.Parameters.checkout_arcadia_from_url)
                st = YqlDeployUDF(self,
                                  targets=udf,
                                  description=description,
                                  checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                                  use_aapi_fuse=self.Parameters.use_aapi_fuse,
                                  thinlto=self.Parameters.thinlto,
                                  host_platform_flags=self.Parameters.host_platform_flags,
                                  definition_flags=self.Parameters.definition_flags,
                                  where_to_deploy=self.Parameters.where_to_deploy,
                                  fail_fast=self.Parameters.fail_fast,
                                  kill_timeout=self.Parameters.kill_timeout)
                st.Requirements.disk_space = self.Parameters.req_disk_space
                st.Requirements.ram = self.Parameters.req_ram
                st.Requirements.cores = self.Parameters.req_cpu
                st.save()
                st.enqueue()
                self.Context.subtask_udfs[st.id] = udf

            raise sdk2.WaitTask(self.Context.subtask_udfs.keys(), ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)

        with self.memoize_stage.check_status:
            self.Context.results = {}
            for i in self.Context.subtask_udfs.keys():
                task = sdk2.Task[i]
                self.Context.results[i] = (task.status, str(task.updated - task.created), self.Context.subtask_udfs[i])

            logging.info(
                'Subtask statuses:\n' + '\n'.join(
                    '    id: {}, status:{}'.format(k, v) for k, v in self.Context.results.items()))

            if not all([s[0] in ctt.Status.Group.SUCCEED for s in self.Context.results.values()]):
                raise common.errors.TaskFailure('Some of subtasks failed')
