# -*- coding: utf-8 -*-

import ast
import logging
import os
import re
import shutil
import traceback

import sandbox.sandboxsdk.environments as environments

from sandbox import sdk2, common
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn
from sandbox.projects.common.build.YaMake2 import YaMake2, ArcadiaProjectBuildParameters

from sandbox.projects.common.vcs.arc import Arc, ArcCommandFailed

class CodegenOutput(sdk2.Resource):
    pass

class YqlCodegenAndCommit(YaMake2):
    """
    Сборка утилиты, запуск, коммит сгенерированного файла
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            # environments.PipEnvironment('yql'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YaMake2.Parameters):

        notifications = [
            sdk2.Notification(
                [
                    ctt.Status.EXCEPTION,
                    ctt.Status.EXPIRED,
                    ctt.Status.FAILURE,
                    ctt.Status.TIMEOUT,
                    ctt.Status.NO_RES,
                    ctt.Status.NOT_RELEASED,
                    ctt.Status.STOPPED,
                ],
                ["yql-dev@yandex-team.ru"],
                ctn.Transport.EMAIL
            )
        ]

        # custom parameters
        with sdk2.parameters.Group("Code generation paraneters:") as project_params:
            executable = sdk2.parameters.String("Executable name", required=True, description="Relative to the first YMake target")
            args = sdk2.parameters.List("Program arguments")
            output = sdk2.parameters.String("Output path", required=True)

            commit_branch = sdk2.parameters.String("Commit to branch", required=True, description="Supports {task_id} via python's .format()")

            env = sdk2.parameters.Dict("Program environment")

            tokens = sdk2.parameters.YavSecret(
                "YAV secret identifier", required=True
            )
            secrets = sdk2.parameters.List("Exported secrets")

            use_validator = sdk2.parameters.Bool("Validate generated file", default=False)
            with use_validator.value[True]:
                validator_executable = sdk2.parameters.String("Validator executable", required=True, description="Relative to the Arcadia root")
                validator_args = sdk2.parameters.List("Validator arguments", description="Each string is formatted with Python's .format(<path_to_generated_file>)")
                validator_observers = sdk2.parameters.List("Validation observers", required=True)

            serialized_structs = sdk2.parameters.Bool("Pass lists and dicts as python literals (for TeamCity integration)", default=False)
            with serialized_structs.value[True]:
                args_serialized = sdk2.parameters.String("Program arguments (serialized list)")
                env_serialized = sdk2.parameters.String("Program environment (serialized dict)")
                secrets_serialized = sdk2.parameters.String("Exported secrets (serialized list)")
                validator_args_serialized = sdk2.parameters.String("Validator arguments (serialized list)")
                validator_observers_serialized = sdk2.parameters.String("Validator observers (serialized list)")

    def pre_build(self, source_dir):
        # self.suspend()
        pass

    def _first_target(self):
        return self.Parameters.targets.split(';')[0]

    def mount_arcadia(self):
        return Arc(arc_oauth_token=self.Parameters.tokens.data()["arc_token"]).mount_path(
            '',
            changeset='HEAD',
            fetch_all=False
        )

    def get_field_maybe_serialized(self, normal_field, serialized_field):
        s = str(serialized_field).strip()
        if len(s) > 0:
            return ast.literal_eval(s)
        else:
            return normal_field

    def get_args(self):
        return self.get_field_maybe_serialized(self.Parameters.args, self.Parameters.args_serialized)

    def get_env(self):
        return self.get_field_maybe_serialized(self.Parameters.env, self.Parameters.env_serialized)

    def get_secrets(self):
        return self.get_field_maybe_serialized(self.Parameters.secrets, self.Parameters.secrets_serialized)

    def get_validator_args(self):
        return self.get_field_maybe_serialized(self.Parameters.validator_args, self.Parameters.validator_args_serialized)

    def get_validator_observers(self):
        return self.get_field_maybe_serialized(self.Parameters.validator_observers, self.Parameters.validator_observers_serialized)

    def post_build(self, source_dir, output_dir, pack_dir):
        exec_path = sdk2.path.Path(output_dir).joinpath(self._first_target()).joinpath(self.Parameters.executable)
        logging.info("Executable: " + str(exec_path))

        output = CodegenOutput(self, "Codegen output", "codegen_output.txt")

        with output.path.open('w') as file_out:
            with sdk2.helpers.ProcessLog(self, logger="shell") as pl:
                env = dict(self.get_env())
                for secret_name in self.get_secrets():
                    env[secret_name] = self.Parameters.tokens.data()[secret_name]

                sdk2.helpers.subprocess.check_call(
                    [str(exec_path)] + list(self.get_args()),
                    stdout=file_out, stderr=pl.stderr, env=env,
                )

        if self.Parameters.use_validator:
            val_exec_path = sdk2.path.Path(output_dir).joinpath(self.Parameters.validator_executable)
            val_args = [
                arg.format(str(output.path)) for arg in self.get_validator_args()
            ]
            logging.info("Validating: " + str(val_exec_path) + ' '.join(val_args))

            with output.path.open() as file_in:
                with sdk2.helpers.ProcessLog(self, logger="shell") as pl:
                    try:
                        sdk2.helpers.subprocess.check_call(
                            [str(val_exec_path)] + val_args,
                            stdin=file_in, stdout=pl.stdout, stderr=pl.stderr,
                        )
                    except sdk2.helpers.subprocess.CalledProcessError as ex:
                        self.server.notification(
                            subject="YQL_CODEGEN_AND_COMMIT {} validation failure".format(self.id),
                            body="https://sandbox.yandex-team.ru/task/{}/view\n{}".format(self.id, str(ex)),
                            recipients=list(self.get_validator_observers()),
                            transport=common.types.notification.Transport.EMAIL
                        )
                        raise

        with self.mount_arcadia() as mount_point:
            path_in_arcadia = os.path.join(mount_point, self.Parameters.output)
            branch = str(self.Parameters.commit_branch).format(task_id=self.id)
            message = 'YQL_CODEGEN_AND_COMMIT {}'.format(self.id)

            arc = Arc(arc_oauth_token=self.Parameters.tokens.data()["arc_token"])
            arc.checkout(mount_point, branch=branch, create_branch=True)
            shutil.copy(str(output.path), path_in_arcadia)
            arc.add(mount_point, path_in_arcadia)

            status = arc.status(mount_point, as_dict=True)

            if "staged" not in status["status"]:
                self.set_info("Nothing to commit, exiting")
                return

            arc.commit(mount_point, message=message)
            arc.push(mount_point, upstream=branch, force=True)

            try:
                out = arc.pr_create(mount_point, message=message, publish=True, auto=True)
            except ArcCommandFailed:
                out = 'Review updated:\n' + arc.pr_status(mount_point)

            self.set_info(self.highlight_links(out), do_escape=False)

    def highlight_links(self, msg):
        expr = re.compile(r'(\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|!:,.;]*[-A-Z0-9+&@#\/%=~_|])', re.IGNORECASE)

        msg_parts = list()
        last_split_idx = 0

        for match in expr.finditer(msg):
            link_begin, link_end = match.span()
            msg_parts.append(msg[last_split_idx:link_begin])
            link = msg[link_begin:link_end]
            last_split_idx = link_end
            msg_parts.append('<a href="{0}" target="_blank">{0}</a>'.format(link))

        msg_parts.append(msg[last_split_idx:])

        return ''.join(msg_parts)
