# -*- coding: utf-8 -*-
import glob
import json
import logging
import mimetypes
import os

import sandbox.sandboxsdk.environments as environments

from sandbox import sdk2, common
from sandbox.projects.common.build.YaMake2 import YaMake2, ArcadiaProjectBuildParameters


# S3 part took from sandbox.projects.sandbox_ci.managers.S3Manager
class YqlDeployUdfUploader:

    def __init__(self, s3_bucket, secret_id, api_host, api_port, yql_token_id):
        logging.getLogger('botocore').setLevel(logging.INFO)
        logging.getLogger('s3transfer').setLevel(logging.INFO)
        logging.getLogger('yql.client.request').setLevel(logging.DEBUG)

        self.s3_domain = 's3.mds.yandex.net'
        self.s3_bucket = s3_bucket
        self.s3_endpoint = 'https://{}'.format(self.s3_domain)

        self.secret_id = secret_id
        self.api_host = api_host
        self.api_port = api_port
        self.yql_token_id = yql_token_id

    def prepare(self, s3_secret, yql_token):
        import boto3
        from botocore.config import Config
        from yql.api.v1.client import YqlClient

        s3_json = json.loads(s3_secret)
        aws_access_key_id = s3_json['AccessKeyId']
        aws_secret_access_key = s3_json['AccessSecretKey']

        assert aws_access_key_id != '' and aws_secret_access_key != '', 'Access key or secret key to S3 is empty'
        session = boto3.session.Session(
            aws_access_key_id=aws_access_key_id,
            aws_secret_access_key=aws_secret_access_key,
        )
        self.s3_client = session.client(
            service_name='s3',
            endpoint_url=self.s3_endpoint,
            config=Config(
                signature_version='s3',
                parameter_validation=False,
                # https://botocore.readthedocs.io/en/latest/reference/config.html
                # speed up (default 10)
                max_pool_connections=512
            )
        )

        assert yql_token != '', 'yql token required but empty'
        self.yql_client = YqlClient(server=self.api_host, port=self.api_port, token=yql_token)

    def upload_to_s3(self, file_path, key):
        try:
            self.s3_client.upload_file(
                file_path,
                self.s3_bucket,
                key,
                ExtraArgs={'ContentType': mimetypes.guess_type(file_path)[0]})

            return 'https://{}.{}/{}'.format(self.s3_bucket, self.s3_domain, key)
        except Exception as e:
            logging.error(e)

        return ''

    def trigger_yql_api(self, udf_name, udf_url):
        try:
            upload_request = self.yql_client.upload_udf_legacy(udf_name, udf_url)
            upload_request.run()
            assert upload_request.is_ok, upload_request.text
            upload_request.wait_for_resolve(600)

            return True
        except Exception as e:
            logging.error(e)
        return False


uploaders = {
    'test': YqlDeployUdfUploader(s3_bucket='yql-udfs-test',
                                 secret_id='sec-01dz6g6py55febq6rdgqqracsn',  # robot-yql-deploy-tst
                                 api_host='yql-test.yandex.net',
                                 api_port=443,
                                 yql_token_id='yql_token'),

    'qa': YqlDeployUdfUploader(s3_bucket='yql-udfs-qa',
                               secret_id='sec-01dz6g6py55febq6rdgqqracsn',  # robot-yql-deploy-tst
                               api_host='yql-qa.search.yandex.net',
                               api_port=32390,
                               yql_token_id='yql_token_qa'),

    'stress': YqlDeployUdfUploader(s3_bucket='yql-udfs-stress',
                                   secret_id='sec-01dz6g6py55febq6rdgqqracsn',  # robot-yql-deploy-tst
                                   api_host='yql-stress.search.yandex.net',
                                   api_port=32390,
                                   yql_token_id='yql_token_stress'),

    'prod': YqlDeployUdfUploader(s3_bucket='yql-udfs',
                                 secret_id='sec-01dz6g6m4djykwvsgmxs25ht3d',  # robot-yql-deploy
                                 api_host='yql.yandex.net',
                                 api_port=443,
                                 yql_token_id='yql_token'),
}


class YqlDeployUDF(YaMake2):
    """
        Сборка, заливка в S3 и подгрузка (индексация) в YQL пользовательских UDF.
        Выкатка возможно на тест и проод - нужны соответсвующие права.
        Сборка идет через YAM_MAKE_2 - доступны все те же параметры для конфигурироваания.
        Список targets должен указывать на конечную папку сборки UDF и после себя порождать один *.so файл
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yql'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YaMake2.Parameters):
        host_platform_flags = ArcadiaProjectBuildParameters.host_platform_flags(
            default="--target-platform-flag=CFLAGS='-fvisibility=hidden'  --target-platform-flag=CFLAGS='-fno-omit-frame-pointer'")

        # custom parameters
        with sdk2.parameters.Group("Project configuration parameters:") as project_params:
            targets = sdk2.parameters.String("Paths to dir with ya.make and YQL_UDF targets", required=True)
            with sdk2.parameters.RadioGroup("Where to deploy") as where_to_deploy:
                for k in uploaders.keys():
                    where_to_deploy.values[k] = where_to_deploy.Value(value=k)
            fail_fast = sdk2.parameters.Bool(
                "Fail immediately after first unsuccessful UDF upload. If false - try to upload the rest",
                default=False)

    def pre_build(self, source_dir):
        self.uploader = uploaders[self.Parameters.where_to_deploy]
        # yav must be called from inside Task class
        secret = sdk2.yav.Secret(self.uploader.secret_id).data()
        self.uploader.prepare(secret['uploader_access_key'], secret[self.uploader.yql_token_id])

        # validate targets
        for t in self.get_targets():
            makefile = os.path.join(source_dir, t, 'ya.make')
            assert os.path.isfile(makefile), '{} file does not exist'.format(makefile)
            # simple check content is not possible. There is no warranty that smth like YQL_UDF will be in ya.make

    def post_build(self, source_dir, output_dir, pack_dir):
        task_id = self.id

        self.Context.results = {}
        for udf_name in self.get_targets():
            # find .so file = get first
            file_path = glob.glob('{}/{}/*.so'.format(output_dir, udf_name))
            if len(file_path) != 1:
                logging.error('Expect exact 1 .so file per UDF, got {}. Skip {} UDF'.format(file_path, udf_name))
                continue

            s3_key = '{}/{}'.format(udf_name, task_id)
            logging.info('uploading {} to key {} from {}'.format(udf_name, s3_key, file_path[0]))

            udf_url = self.uploader.upload_to_s3(file_path[0], s3_key)

            uploaded = False
            if udf_url != '':
                logging.info('triggering api to upload {} from {}'.format(udf_name, udf_url))
                uploaded = self.uploader.trigger_yql_api(udf_name, udf_url)
            else:
                logging.error('{} was not uploaded. Skip triggering api'.format(udf_name))
            self.Context.results[udf_name] = (uploaded, udf_url)

            if self.Parameters.fail_fast and not uploaded:
                break

        logging.info('Summary UDFs status:\n' + '\n'.join(
            '    UDF: {}, from: {}, uploaded: {}'.format(k, v[1], v[0]) for k, v in self.Context.results.items()))

        super(YqlDeployUDF, self).post_build(source_dir, output_dir, pack_dir)

        if not all([s[0] for s in self.Context.results.values()]):
            raise common.errors.TaskError("Some of UDFs was not uploaded successfully - fail the task")
