# -*- coding: utf-8 -*-

import logging
import os

from sandbox.projects.common import utils
from sandbox.projects.nanny.common.build_helpers import git_clone_and_checkout
from sandbox.projects.nanny.common.gobabygo_parameters import GitRefIdParameter, GitRefShaParameter, ReleaseParameter
from sandbox.sandboxsdk.process import run_process


class cached_property(object):
    def __init__(self, func):
        self.__doc__ = getattr(func, '__doc__')
        self.func = func

    def __get__(self, obj, cls=None):
        if obj is None:
            return self
        val = obj.__dict__[self.func.__name__] = self.func(obj)
        return val


class YtBuildBundleTask(object):
    GIT_URL = 'https://github.yandex-team.ru/yt/{repository}'

    @cached_property
    def _do_release(self):
        return utils.get_or_default(self.ctx, ReleaseParameter)

    @cached_property
    def _git_ref(self):
        ref_id = utils.get_or_default(self.ctx, GitRefIdParameter)
        ref_sha = utils.get_or_default(self.ctx, GitRefShaParameter)
        return ref_sha or ref_id

    @cached_property
    def _git_repo(self):
        return getattr(self, 'GIT_REPO', None) or self.BUNDLE_NAME

    @cached_property
    def _git_repo_url(self):
        return self.GIT_URL.format(repository=self._git_repo)

    @cached_property
    def _work_dir(self):
        return self.BUNDLE_NAME

    def _checkout(self, ref=None):
        ref = ref or self._git_ref
        repo_url = self._git_repo_url
        logging.info('Checking out ref "{}" of {} to {}'
                     .format(ref, repo_url, self._work_dir))
        return git_clone_and_checkout(repo_url, self._work_dir, ref)

    def _runner(self, *args):
        log_prefix = 'runner.sh {}'.format(args[0])
        return run_process(['./runner.sh'] + list(args),
                            work_dir=self._work_dir,
                            log_prefix=log_prefix,
                            shell=True)

    @cached_property
    def _build_prepared(self):
        return False

    def _prepare_build(self):
        if self._build_prepared:
            return
        logging.info('Initializing build environment')
        res = self._runner('go')
        if res:
            self._build_prepared = True
        return res

    def _build(self):
        self._prepare_build()
        logging.info('Building')
        return self._runner('build')

    @cached_property
    def _bundle_prepared(self):
        return False

    @cached_property
    def _bundle_root(self):
        return '.{}_bundle_root'.format(self.BUNDLE_NAME)

    def _prepare_bundle(self, dirs=None, files=None):
        if self._bundle_prepared:
            return

        dirs = dirs or {}
        files = files or {}
        logging.info('Preparing the bundle')

        bundle_dirs = set()
        targets = dirs.values() + files.values()
        for target in targets:
            dst = os.path.dirname(target)
            if dst:
                bundle_dirs.add(dst)

        logging.info('Pre-creating bundle directories')
        dirs_to_make = [self._bundle_root] + [os.path.join(self._bundle_root, target) for target in sorted(bundle_dirs)]
        run_process(['mkdir', '-vp'] + dirs_to_make, log_prefix='prepare_bundle_root', shell=True)

        logging.info('Populating bundle contents')
        for src in sorted(dirs.keys()):
            run_process(['mv', '-v',
                         os.path.join(self._work_dir, src),
                         os.path.join(self._bundle_root, dirs[src])],
                         log_prefix='prepare_bundle_dirs',
                         shell=True)
        for src in sorted(files.keys()):
            run_process(['mv', '-v',
                         os.path.join(self._work_dir, src),
                         os.path.join(self._bundle_root, files[src])],
                         log_prefix='prepare_bundle_files',
                         shell=True)

        self._bundle_prepared = True

    @cached_property
    def _bundle_version(self):
        return self._get_bundle_version()

    def _get_bundle_version(self):
        logging.info('Parsing bundle version')
        version = run_process(["head -n 1 debian/changelog | cut -f2 -d'(' | cut -f1 -d')'"],
                                work_dir=self._work_dir,
                                log_prefix='get_bundle_version',
                                outs_to_pipe=True,
                                shell=True,
                                wait=False).stdout.read().strip()
        logging.info('Parsed version: {!r}'.format(version))
        return version or None

    @cached_property
    def _bundle_filename(self):
        if self._bundle_version:
            return '{}-{}.tar.gz'.format(self.BUNDLE_NAME, self._bundle_version)
        else:
            return '{}.tar.gz'.format(self.BUNDLE_NAME)

    def _package_bundle(self, dirs=None, files=None):
        dirs = dirs or {}
        files = files or {}

        self._prepare_bundle(dirs, files)

        logging.info('Packaging the bundle')
        output = run_process(['find',
                              '-mindepth', '1',
                              '-maxdepth', '1',
                              '-printf', '%P\n'],
                              work_dir=self._bundle_root,
                              outs_to_pipe=True,
                              shell=True).stdout
        bundle_contents = sorted(output.read().split('\n'))
        logging.info('Going to write: {}'.format(self._bundle_filename))
        run_process(['tar', '-czv',
                     '--owner', 'root',
                     '--group', 'root',
                     '-f', self._bundle_filename,
                     '-C', self._bundle_root]
                        + bundle_contents,
                    log_prefix='package_bundle',
                    shell=True)

    def _release_bundle(self, **kwargs):
        logging.info('Uploading the bundle')
        if 'resource_path' not in kwargs:
            kwargs['resource_path'] = self._bundle_filename
        if 'description' not in kwargs:
            kwargs['description'] = '{} bundle'.format(self.BUNDLE_NAME)
        return self.create_resource(**kwargs)
