# -*- coding: utf-8 -*-

import json
import logging
from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.yt.common.tasks.YtBuildConfigTask import YtBuildConfigTask
from sandbox.projects.yt.common import YtClusters, ArcadiaCheckoutTask
from sandbox.projects.yt.common.resources import YtConfigResource


class YtBuildOdinDaemonConfigBase(YtBuildConfigTask):
    class Parameters(sdk2.Task.Parameters):
        config_template_name = sdk2.parameters.String(
            "Config template name",
            default="odin.config.template.json",
            required=True
        )
        revision = sdk2.parameters.String("Revision", default=None, required=False)
        cluster_selector = sdk2.parameters.List("Cluster selector", default=None, required=False)
        cluster_filter = sdk2.parameters.List("Cluster filter", default=None, required=False)

    class Requirements(sdk2.Task.Requirements):
        max_restarts = 3
        disk_space = 10 * 1024
        environments = (
            PipEnvironment("functools32", version="3.2.3-2"),
            PipEnvironment("pyrsistent", version="0.15.5"),
            PipEnvironment("jsonschema", version="3.0.1", custom_parameters=["--no-deps"])
        )

    def set_variables(self):
        self.ctx = {
            "checkout_data": {
                "odin": ArcadiaCheckoutTask("/yt/odin/configs", "configs", self.Parameters.revision)
            },
            "config_template": "configs/{}".format(self.Parameters.config_template_name),
            "validation_schema": "configs/odin.schema.json",
            "output_file": "odin.config.json",
            "skip_clusters": [],
            "output_resources": [(YtConfigResource, "odin.config.json")]
        }

    def prepare(self):
        self.ctx["task_revisions"] = []
        for name, task in self.ctx["checkout_data"].items():
            revision = self._arcadia_checkout(task.url, task.dstdir, task.revision)
            self.ctx["task_revisions"].append("Url {} revision {}".format(task.url, revision))

    @staticmethod
    def _filter_clusters(cluster_proxy_pairs, selector, filter_):
        if selector:
            cluster_proxy_pairs = filter(lambda c: c[0] in selector, cluster_proxy_pairs)
        if filter_:
            cluster_proxy_pairs = filter(lambda c: c[0] not in filter_, cluster_proxy_pairs)
        return cluster_proxy_pairs

    def build_context(self):
        self.ctx["yt_clusters"] = self._filter_clusters(
            YtClusters.get_clusters(with_proxy=True),
            self.Parameters.cluster_selector,
            self.Parameters.cluster_filter
        )

    def build_configs(self):
        template = self._load_json(self.ctx["config_template"])
        yt_token = template["secrets"]["yt_token"]
        for cluster, proxy in self.ctx["yt_clusters"]:
            if cluster in self.ctx["skip_clusters"]:
                continue
            new_section = {
                "{}".format(cluster): {
                    "yt_config": {
                        "proxy": proxy
                    },
                    "db_config": {
                        "options": {
                            "cluster": cluster,
                            "proxy": "locke.yt.yandex.net",
                            "table": "//sys/odin/checks",
                            "token": yt_token,
                        }
                    },
                    "juggler_host": "odin.{}".format(proxy)
                }
            }
            old_section = template["clusters"].get(cluster, None)
            if old_section:
                new_section[cluster].update(old_section)
            template["clusters"].update(new_section)
        self.ctx["config"] = template

    def validate(self):
        # NB: this is a hack needed to enforce correct backports initialization.
        # https://ml.yandex-team.ru/thread/sandbox/171418260816852509/#message171418260816852551
        import backports
        reload(backports)
        import jsonschema
        schema = self._load_json(self.ctx["validation_schema"])
        logging.debug(json.dumps(schema, indent=4, sort_keys=True))
        logging.debug(json.dumps(self.ctx["config"], indent=4, sort_keys=True))
        jsonschema.validate(self.ctx["config"], schema)

    def save_configs(self):
        self._save_json(self.ctx["config"], self.ctx["output_file"])

    def create_resources(self):
        for resource, path in self.ctx["output_resources"]:
            resource(self, ", ".join(self.ctx["task_revisions"]), path)
