# -*- coding: utf-8

import json
import logging
from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.yt.common.tasks.YtBuildConfigTask import YtBuildConfigTask
from sandbox.projects.yt.common import YtClusters, ArcadiaCheckoutTask
from sandbox.projects.yt.common.resources import YtConfigResource

PRODUCTION_TEMPLATE = "transfer_manager.conf.template.json"


class YtBuildTransferManagerConfig(YtBuildConfigTask):
    class Parameters(sdk2.Task.Parameters):
        template = sdk2.parameters.String("Config template", default=PRODUCTION_TEMPLATE, required=True)
        revision = sdk2.parameters.String("Revision", default=None, required=False)

    class Requirements(sdk2.Task.Requirements):
        max_restarts = 3
        disk_space = 10 * 1024
        environments = (
            PipEnvironment("functools32", version="3.2.3-2"),
            PipEnvironment("pyrsistent", version="0.15.5"),
            PipEnvironment("jsonschema", version="3.0.1", custom_parameters=["--no-deps"])
        )

    def set_variables(self):
        self.ctx = {
            "checkout_data": {
                "transfer_manager": ArcadiaCheckoutTask("/transfer_manager/server/config", "config", self.Parameters.revision)
            },
            "config_template": "config/{}".format(self.Parameters.template),
            "validation_schema": "config/transfer_manager.schema.json",
            "output_file": "transfer_manager.conf.json",
            "skip_clusters": ["yp-sas"],
            "ofd_clusters": ["ofd", "ofd-xdc"],
            "ofd_destinations": ["hahn"],
            "auth_clusters": ["flux"],
            "output_resources": [(YtConfigResource, "transfer_manager.conf.json")]
        }

    def prepare(self):
        self.ctx["task_revisions"] = []
        for name, task in self.ctx["checkout_data"].items():
            revision = self._arcadia_checkout(task.url, task.dstdir, task.revision)
            self.ctx["task_revisions"].append("Url {} revision {}".format(task.url, revision))

    def build_context(self):
        self.ctx["yt_clusters"] = YtClusters.get_clusters(with_proxy=True)

    def build_configs(self):
        template = self._load_json(self.ctx["config_template"])
        auth_clusters = set(self.ctx["auth_clusters"])
        kiwi_transmitter = template.get("kiwi_transmitter")
        if kiwi_transmitter is not None:
            auth_clusters.add(kiwi_transmitter)
        for cluster, proxy in self.ctx["yt_clusters"]:
            if cluster in self.ctx["skip_clusters"]:
                continue
            new_section = {
                "{}".format(cluster): {
                    "type": "yt",
                    "options": {
                        "proxy": proxy,
                    },
                    "fastbone": True,
                    "oauth_application": "{}_api".format(cluster),
                    "table_url_pattern": "https://yt.yandex-team.ru/{}/#page=navigation&path=%table%".format(cluster),
                }
            }
            if cluster in auth_clusters:
                new_section[cluster]["options"]["token"] = "{TM_TOKEN}"
            old_section = template["clusters"].get(cluster, None)
            if old_section:
                new_section[cluster].update(old_section)
            template["clusters"].update(new_section)

            graph = template["availability_graph"].setdefault(cluster, list())
            if cluster in self.ctx["ofd_clusters"]:
                graph.extend(edge for edge in self.ctx["ofd_clusters"] if edge != cluster)
                graph.extend(edge for edge in self.ctx["ofd_destinations"])
            else:
                graph.extend(edge for edge, _ in self.ctx["yt_clusters"] if not (edge == cluster or edge in self.ctx["skip_clusters"]))
        for cluster, edges in template["availability_graph"].iteritems():
            for edge in edges:
                assert edge in template["clusters"], "Source '{}': unknown destination '{}'".format(cluster, edge)
        self.ctx["config"] = template

    def validate(self):
        import jsonschema
        schema = self._load_json(self.ctx["validation_schema"])
        logging.debug(json.dumps(schema, indent=4, sort_keys=True))
        logging.debug(json.dumps(self.ctx["config"], indent=4, sort_keys=True))
        jsonschema.validate(self.ctx["config"], schema)

    def save_configs(self):
        self._save_json(self.ctx["config"], self.ctx["output_file"])

    def create_resources(self):
        for resource, path in self.ctx["output_resources"]:
            resource(self, ", ".join(self.ctx["task_revisions"]), path)
