# -*- coding: utf-8

import json
import shutil
import os
import re
import six

import sandbox.common as common
import sandbox.common.types.task as ctt
from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as arcadiasdk


class YtCronRunnerResource(sdk2.Resource):
    releasable = True


class YtCronScriptBundleResource(sdk2.Resource):
    releasable = True
    bundled_scripts = sdk2.Attributes.String("scripts", required=True)


class YtCronLegacyScriptBundleResource(YtCronScriptBundleResource):
    releasable = True
    pass


class PackageJson(object):
    __slots__ = [
        "targets",
        "data",
        "provides",
    ]

    def __init__(self, json_data=None, package=None):
        self.targets = set()
        self.data = dict()
        self.provides = set()
        if json_data is not None:
            self.targets = set(json_data.get("build", dict()).get("targets", list()))
            data = json_data.get("data", list())
            self.data = {
                item["destination"]["path"]: item
                for item in data
            }
            self.provides = set(json_data.get("meta", dict()).get("provides", [package]))

    def to_dict(self):
        ret = {
            "meta": {
                "name": "yandex-yt-cron-binaries",
                "maintainer": "Rebenko Yaroslav <rebenkoy@yandex-team.ru>",
                "description": "YT Cron scripts package",
                "version": "{revision}",
            },
            "build": {
                "targets": list(self.targets),
            },
            "data": list(self.data.values()),
        }
        return ret

    def __iadd__(self, other):
        self.targets.update(other.targets)
        self.provides.update(other.provides)
        for dst, element in six.iteritems(other.data):
            if dst not in self.data:
                self.data[dst] = element
            elif self.data[dst] == element:
                continue
            else:
                raise RuntimeError("destination conflict for path {}".format(dst))
        return self


class YtBuildCronPackage(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 1 * 1024

    class CustomParameters(object):
        package_type = "tarball"
        compress_package_archive = True
        use_aapi_fuse = False

        use_new_format = True

        ttl = 30
        package_path = ""
        package_name = ""
        comment_to_description = ""
        description = ""
        revision = None
        output_resource = "YT_PACKAGE_RESOURCE"

        force_dupload = False
        publish_package = False
        publish_to = ""
        key_user = ""

        checkout_mode = "manual"

    class Context(sdk2.Task.Context):
        target_list = []

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Runner") as runner_block:
            build_runner = sdk2.parameters.Bool("Build runner package", default=True)

        with sdk2.parameters.Group("Scripts") as scripts_block:
            build_all = sdk2.parameters.Bool("Build all scripts.", default=False)
            exception_caption = None
            with build_all.value[False]:
                package_inclusion_list = sdk2.parameters.List("Included packages:")
            with build_all.value[True]:
                package_exclusion_list = sdk2.parameters.List("Excluded packages:")

        with sdk2.parameters.Group("Legacy") as legacy:
            use_legacy_package = sdk2.parameters.Bool("Use yt/cron/package.json", default=False)

        with sdk2.parameters.Group("Misc") as misc:
            arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia url", required=True)
            do_not_remove = sdk2.parameters.Bool("Set 'do not remove' for resources", default_value=False)
            startrack_ticket_ids = sdk2.parameters.List("Startrek tickets", required=False)
            arcadia_patch = sdk2.parameters.String(
                "Apply patch (doc: https://nda.ya.ru/3QTTV4)", required=False, default="", multiline=True)

    @staticmethod
    def normalize_name(name):
        name = name.strip()

        name = name.replace("-", "-")
        name = name.replace("_", "-")
        json_extension = ".json"
        if name.endswith(json_extension):
            name = name[:-len(json_extension)]
        package_prefix = "yandex-yt-cron"
        if name.startswith(package_prefix):
            name = name[len(package_prefix):]
        name = name.strip("-")
        return name

    def on_execute(self):
        with self.memoize_stage.create_subtasks:
            file_to_package_map = {}
            revision = sdk2.svn.Arcadia.info(self.Parameters.arcadia_url)["commit_revision"]
            url = "{}@{}".format(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, revision)
            with arcadiasdk.mount_arc_path(url, use_arc_instead_of_aapi=True) as arcadia_path:
                patch = self.Parameters.arcadia_patch
                if patch:
                    sdk2.svn.Arcadia.apply_patch(arcadia_path, patch, self.path())

                packages = {}
                packages_dir = os.path.join(arcadia_path, "yt/cron/packages")
                for file_name in os.listdir(str(packages_dir)):
                    if file_name.endswith(".json"):
                        with open(os.path.join(str(packages_dir), file_name), "r") as fd:
                            normalized_file_name = YtBuildCronPackage.normalize_name(file_name)
                            text = fd.read()
                            text = re.sub(r",([ \t]*\n[ \t]*[\]\}])", r"\1", text)
                            package_json = json.loads(text)
                            name = YtBuildCronPackage.normalize_name(package_json["meta"]["name"])
                            if name in packages:
                                raise RuntimeError("Conflict for package name {}".format(name))
                            packages[name] = PackageJson(package_json, file_name[:-len(".json")])
                            file_to_package_map[normalized_file_name] = name

                if self.Parameters.build_all:
                    targets = set(packages.keys())
                    for exclusion in self.Parameters.package_exclusion_list:
                        targets.discard(YtBuildCronPackage.normalize_name(exclusion))
                else:
                    targets = set()
                    for inclusion in self.Parameters.package_inclusion_list:
                        targets.add(YtBuildCronPackage.normalize_name(inclusion))

                package = PackageJson()
                for target in targets:
                    package += packages[target]
                self.Context.target_list = list(package.provides)

            sub_tasks = []
            selective_checkout = True
            if self.Parameters.arcadia_patch:
                selective_checkout = False

            if self.Parameters.build_runner:
                runner_task = sdk2.Task["YA_PACKAGE"](
                    self,
                    description="Build runner package",
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    arcadia_patch=self.Parameters.arcadia_patch,
                    checkout=selective_checkout,

                    packages="yt/cron/runner/package.json",
                    resource_type=YtCronRunnerResource.name,

                    checkout_mode=self.CustomParameters.checkout_mode,
                    compress_package_archive=self.CustomParameters.compress_package_archive,
                    force_dupload=self.CustomParameters.force_dupload,
                    key_user=self.CustomParameters.key_user,
                    package_type=self.CustomParameters.package_type,
                    publish_package=self.CustomParameters.publish_package,
                    publish_to=self.CustomParameters.publish_to,
                    use_aapi_fuse=self.CustomParameters.use_aapi_fuse,
                    use_new_format=self.CustomParameters.use_new_format,
                )
                runner_task.Requirements.disk_space = self.Requirements.disk_space
                runner_task.save().enqueue()
                sub_tasks.append(runner_task)
            if self.Parameters.use_legacy_package:
                legacy_task = sdk2.Task["YA_PACKAGE"](
                    self,
                    description="Build legacy scripts package",
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    arcadia_patch=self.Parameters.arcadia_patch,
                    checkout=selective_checkout,

                    packages="yt/cron/package.json",
                    resource_type=YtCronLegacyScriptBundleResource.name,

                    checkout_mode=self.CustomParameters.checkout_mode,
                    compress_package_archive=self.CustomParameters.compress_package_archive,
                    force_dupload=self.CustomParameters.force_dupload,
                    key_user=self.CustomParameters.key_user,
                    package_type=self.CustomParameters.package_type,
                    publish_package=self.CustomParameters.publish_package,
                    publish_to=self.CustomParameters.publish_to,
                    use_aapi_fuse=self.CustomParameters.use_aapi_fuse,
                    use_new_format=self.CustomParameters.use_new_format,
                )
                legacy_task.Requirements.disk_space = self.Requirements.disk_space
                legacy_task.save().enqueue()
                sub_tasks.append(legacy_task)
            if package.data:
                scripts_task = sdk2.Task["YA_PACKAGE"](
                    self,
                    description="Build cron scripts package",
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    arcadia_patch=self.Parameters.arcadia_patch,
                    checkout=selective_checkout,

                    adhoc_packages=[json.dumps(package.to_dict())],
                    resource_type=YtCronScriptBundleResource.name,

                    checkout_mode=self.CustomParameters.checkout_mode,
                    compress_package_archive=self.CustomParameters.compress_package_archive,
                    force_dupload=self.CustomParameters.force_dupload,
                    key_user=self.CustomParameters.key_user,
                    package_type=self.CustomParameters.package_type,
                    publish_package=self.CustomParameters.publish_package,
                    publish_to=self.CustomParameters.publish_to,
                    use_aapi_fuse=self.CustomParameters.use_aapi_fuse,
                    use_new_format=self.CustomParameters.use_new_format,
                )
                scripts_task.Requirements.disk_space = self.Requirements.disk_space
                scripts_task.save().enqueue()
                sub_tasks.append(scripts_task)

            raise sdk2.WaitTask(
                sub_tasks,
                set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)),
                wait_all=True,
            )

        with self.memoize_stage.collect_resources:
            sub_tasks = self.find()
            for sub_task in sub_tasks:
                r = self._copy_resource(YtCronRunnerResource, sub_task)
                if r is not None:
                    runner_resource = r
                r = self._copy_resource(YtCronLegacyScriptBundleResource, sub_task)
                if r is not None:
                    legacy_resource = r
                r = self._copy_resource(YtCronScriptBundleResource, sub_task)
                if r is not None:
                    scripts_resource = r
                    scripts_resource.bundled_scripts = ", ".join(self.Context.target_list)

    def _copy_resource(self, cls, sub_task):
        old_resource = sdk2.Resource.find(
            cls,
            task=sub_task,
        ).first()
        if old_resource is None:
            return
        old_resource_data = sdk2.ResourceData(old_resource)
        old_resource_path = str(old_resource_data.path)

        new_resource_data_path = str(self.path(os.path.basename(old_resource_path)))
        if os.path.isfile(old_resource_path):
            shutil.copy(old_resource_path, new_resource_data_path)
        else:
            shutil.copytree(old_resource_path, new_resource_data_path)

        return cls(
            self,
            old_resource.description,
            new_resource_data_path,
            ttl=old_resource.ttl,
        )
