import collections
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
import sandbox.common as common

dynconfig_message_template = """
<html>
There are uncommitted changes of dynamic config on some clusters.
<pre>
{}
</pre>
Please commit or revert these changes.


Full diff below:
<pre>
{}
</pre>
</html>
"""

dynconfig_load_error_message_template = """
<html>
Could not load dynamic config for some services.
<pre>
{}
</pre>
</html>
"""

cron_config_message_template = """
<html>
There are uncommitted changes of dynamic cron config.
<pre>
{}
</pre>
Please commit or revert these changes.
</html>
"""

cron_warn_message_template = """
<html>
{}
{}
</html>
"""


class YtDynamicConfigWatcher(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_token = sdk2.parameters.YavSecret(
            "YT token",
            required=True,
        )
        recipients = sdk2.parameters.List(
            "Recipients",
            required=True,
            default=["yt-dev-root"]
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "YT_DYNAMIC_CONFIG_WATCHER", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def check_dynconfig(self):
        import yt.yson as yson
        from yt.common import YtError
        from yt.admin.ytcfgen.dynamic.src import parsers, connections
        from deepdiff import DeepDiff
        import yt.wrapper as yt

        yt_client = yt.YtClient(proxy="locke", token=self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key])
        cron_config = yt_client.get("//sys/cron/config/clusters")
        offline_clusters = set(cluster for cluster in cron_config if cron_config[cluster]["group"] == "offline")

        # Current ARC config
        configs, filtered_out = parsers.build_configs([], [])

        diffs = collections.defaultdict(dict)
        load_errors = collections.defaultdict(list)
        diffed_services = collections.defaultdict(list)
        for cluster, components in configs.items():
            if cluster in offline_clusters:
                continue
            for component, config in components.items():
                try:
                    old_config = connections.fetch(cluster, component, yt_token=self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key])
                    new_config = config
                    ddiff = DeepDiff(yson.yson_to_json(old_config), yson.yson_to_json(new_config), ignore_order=True, verbose_level=2)
                    if ddiff:
                        diffs[cluster][component] = ddiff
                        diffed_services[cluster].append(component)
                except YtError:
                    load_errors[cluster].append(component)

        if diffs:
            component_diff_text = yson.dumps(diffed_services, yson_format="pretty")
            diff_text = yson.dumps(diffs, yson_format="pretty")
            message = dynconfig_message_template.format(
                component_diff_text.decode(),
                diff_text.decode(),
            )
            logging.info(message)
            self.server.notification(
                subject="Uncommitted dynamic config changes",
                body=message,
                headers=["Content-Type: text/html; charset=UTF-8\r\n"],
                recipients=self.Parameters.recipients,
                transport=common.types.notification.Transport.EMAIL,
            )
        else:
            logging.info("No diff detected.")

        if load_errors:
            component_diff_text = yson.dumps(load_errors, yson_format="pretty")
            message = dynconfig_load_error_message_template.format(
                component_diff_text.decode(),
            )
            logging.info(message)
            self.server.notification(
                subject="Error loading dynconfig from YT",
                body=message,
                headers=["Content-Type: text/html; charset=UTF-8\r\n"],
                recipients=self.Parameters.recipients,
                transport=common.types.notification.Transport.EMAIL,
            )
        else:
            logging.info("No load errors detected.")

    def get_sb_warnings(self, label, config):
        from yt.admin.luigi.lib.cron import publish as publish_parts

        resource_warnings = publish_parts.get_resource_warnings(config)

        warnings = []
        for resource, script_name in resource_warnings["ttl"]:
            warnings.append("{}: {} for sandbox resource [{}] in {} is not set to infinity, resource will expire.".format(
                "WARNING",
                "ttl",
                resource,
                script_name,
            ))
        for resource, bundled_scripts, script_name in resource_warnings["missing_from_bundle"]:
            warnings.append("{}: sandbox resource [{}] provides [{}] scripts, not recommended to use in {}.".format(
                "WARNING",
                resource,
                bundled_scripts,
                script_name,
            ))
        for resource, script_name in resource_warnings["no_bundle"]:
            warnings.append("{}: {} for sandbox resource [{}] is not set, can not validate {}.".format(
                "WARNING",
                "bundled_scripts",
                resource,
                script_name,
            ))

        if len(warnings):
            return "There are warnings for config present in {}:\n<pre>{}</pre>".format(
                label,
                "\n".join(" " * 4 + x for x in warnings),
            )
        return ""

    def check_cron_config(self):
        from google.protobuf.json_format import ParseDict, MessageToDict

        from yt.admin.luigi.lib.cron import publish as publish_parts
        from yt.cron.runner.cron_scheduler.proto import schema_pb2
        import yt.yson as yson
        import yt.wrapper as yt
        from deepdiff import DeepDiff

        yt_client = yt.YtClient(proxy="locke", token=self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key])

        config_in_arc = publish_parts.read_config(use_arc=True)
        config_in_yt = ParseDict(yson.yson_to_json(yt_client.get("//sys/cron/config")), schema_pb2.Config())

        config_in_arc_warn_text = self.get_sb_warnings("Arcanum", config_in_arc)
        config_in_yt_warn_text = self.get_sb_warnings("YT", config_in_yt)

        ddiff = DeepDiff(MessageToDict(config_in_yt), MessageToDict(config_in_arc), ignore_order=True, verbose_level=2)
        ddiff_text = ""
        if ddiff:
            ddiff_text = yson.dumps(ddiff, yson_format="pretty").decode()

        if ddiff_text:
            message = cron_config_message_template.format(
                ddiff_text,
            )
            logging.info(message)
            self.server.notification(
                subject="Uncommitted cron config changes",
                body=message,
                headers=["Content-Type: text/html; charset=UTF-8\r\n"],
                recipients=self.Parameters.recipients,
                transport=common.types.notification.Transport.EMAIL,
            )
        else:
            logging.info("No diff detected.")

        if config_in_yt_warn_text or config_in_arc_warn_text:
            message = cron_warn_message_template.format(
                config_in_yt_warn_text,
                config_in_arc_warn_text,
            )
            logging.info(message)
            self.server.notification(
                subject="Warnings for cron config",
                body=message,
                headers=["Content-Type: text/html; charset=UTF-8\r\n"],
                recipients=self.Parameters.recipients,
                transport=common.types.notification.Transport.EMAIL,
            )
        else:
            logging.info("No diff detected.")

    def on_execute(self):
        super(YtDynamicConfigWatcher, self).on_execute()
        self.check_dynconfig()
        self.check_cron_config()
