# -*- coding: utf-8

import os

from sandbox import sdk2
from sandbox.projects.common import binary_task


class YtMergeTablesTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Required parameters") as req:
            proxy = sdk2.parameters.String(
                "YT proxy",
                required=True,
            )
            root = sdk2.parameters.String(
                "search root",
                required=True,
            )
            pool = sdk2.parameters.String(
                "yt pool",
                required=True,
            )
            queue = sdk2.parameters.String(
                "yt queue path",
                required=True,
            )
            secret = sdk2.parameters.YavSecret(
                "YT token",
                required=True,
            )
            secret_key = sdk2.parameters.String(
                "secret key",
                required=True,
            )
        with sdk2.parameters.Group("Optional parameters") as optional:
            fast_pool = sdk2.parameters.String(
                "yt fast pool",
                default="",
            )
            tmp = sdk2.parameters.String(
                "yt path for temporary",
                default="",
            )
            minimum_number_of_chunks = sdk2.parameters.Integer(
                "minimum number of chunks",
                default=100
            )
            maximum_chunk_size = sdk2.parameters.Integer(
                "maximum chunk size",
                default=100 * 1024 * 1024
            )
            thread_count = sdk2.parameters.Integer(
                "thread count",
                default=8,
            )
            minimal_age = sdk2.parameters.Integer(
                "minimal age (in hours)",
                default=0,
            )

        with sdk2.parameters.Group("Defaulted parameters") as default:
            ignore_acl = sdk2.parameters.Bool(
                "ignore acl",
                default=True,
            )
        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(YtMergeTablesTask, self).on_execute()

        token = self.Parameters.secret.data()[self.Parameters.secret_key]
        os.environ['YT_TOKEN'] = token

        from yt.cron.library.token_variables import TokenEnvironmentVariables, replace_envar
        import yt.cron.merge.perform_merge.lib as perform_merge
        import yt.cron.merge.find_tables_to_merge.lib as find_tables_to_merge

        find_tables_to_merge.configure_logger()
        find_tables_to_merge.configure_client()
        table_finder = find_tables_to_merge.TableFinder(
            self.Parameters.proxy,
            self.Parameters.queue,
            root=self.Parameters.root,
            minimum_number_of_chunks=self.Parameters.minimum_number_of_chunks,
            maximum_chunk_size=self.Parameters.maximum_chunk_size,
            minimal_age=self.Parameters.minimal_age
        )
        table_finder.find_tables()
        table_finder.save_to_queue()

        replace_envar(TokenEnvironmentVariables.YT_TOKEN.name, TokenEnvironmentVariables.YT_MERGE_TOKEN.name)
        perform_merge.start(
            self.Parameters.proxy,
            self.Parameters.queue,
            pool=self.Parameters.pool,
            tmp_path=self.Parameters.tmp or None,
            fast_pool=self.Parameters.fast_pool or self.Parameters.pool,
            ignore_acl=self.Parameters.ignore_acl,
            thread_count=self.Parameters.thread_count
        )

    def on_before_timeout(self, seconds):
        if seconds <= 10:
            import yt.cron.merge.perform_merge.lib as perform_merge
            perform_merge.stop_merging()
            self.set_info('<span style="color: red">Task finished by timeout</span>', do_escape=False)
        super(YtMergeTablesTask, self).on_before_timeout(seconds)
