# coding: utf-8

import os
import logging

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
import sandbox.common.types.task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types.resource import State
from sandbox.projects.porto import BuildPortoLayer
from sandbox.projects.common.nanny import nanny
from sandbox.common.share import skynet_get


class YtBuildDefaultUserLayer(nanny.ReleaseToNannyTask2, sdk2.Task):
    # type = 'YT_BUILD_DEFAULT_USER_LAYER'
    class Parameters(sdk2.Task.Parameters):
        def resource_types_with_prefixes(*args):
            return [r for r in sdk2.Resource if any(r.name.startswith(arg) for arg in args)]

        layer_type = sdk2.parameters.String('Layer type', required=True, default='PORTO_LAYER_YT')
        layer_name = sdk2.parameters.String('Layer name', required=True, default='yt_default_user_jobs_rootfs')
        with sdk2.parameters.String("Compress", multiline=True, required=True) as compress:
            for choice in ['tar.gz', 'tar.xz', 'tar']:
                if choice == 'tar.gz':
                    compress.values[choice] = compress.Value(default=True)
                else:
                    compress.values[choice] = None

        parent_layer = sdk2.parameters.Resource(
            "Parent layer", required=True, resource_type=resource_types_with_prefixes('PORTO_LAYER'))
        script_url = sdk2.parameters.ArcadiaUrl('Setup script URL')
        space_limit = sdk2.parameters.Integer('Max disk usage (MB)', default=8192)
        memory_limit = sdk2.parameters.Integer('Max memory usage (MB)', default=4096)
        debug_build = sdk2.parameters.Bool("Debug build", default=False)
        cluster = sdk2.parameters.String('Target YT cluster', required=True)
        yt_path = sdk2.parameters.String(
            'YT cypress path',
            required=True,
            default="//porto_layers/yt_default_user_jobs_rootfs.tar.gz")
        yt_token_vault_name = sdk2.parameters.String('YT token vault name', required=True)
        release_task = sdk2.parameters.Bool("Release build task", default=False)
        skip_if_synced = sdk2.parameters.Bool("Skip if synced", default=False)
        merge_all_layers = sdk2.parameters.Bool("Merge all layers", default=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment("yandex-yt"),)
        client_tags = ctc.Tag.GENERIC

    description = "Default user layer: YTADMIN-8669"
    META_CLUSTER = 'locke'
    GEOBASE_META_PATH = '//home/geobase'
    PACKAGES_META_PATH = '//sys/admin/user_packages/clusters'

    def _get_yt_client(self, cluster):
        from yt.wrapper import YtClient
        return YtClient(cluster, token=sdk2.Vault.data(self.Parameters.yt_token_vault_name))

    def fetch_resources_from_meta(self):
        if not self.Context.packages_from_meta:
            self.Context.packages_from_meta = self.meta_client.get("{}/{}".format(self.PACKAGES_META_PATH, self.Parameters.cluster))
        if not self.Context.skip_geobase and not self.Context.geobases_from_meta:
            self.Context.geobases_from_meta = [r[1] for r in self.meta_client.get(self.GEOBASE_META_PATH).items()
                                                   if r[0].split("-stable")[0] in ["geodata5", "geodata4", "geodata-treeling", "tzdata"]]

    def resources_are_synced_with_meta(self):
        cluster_client = self._get_yt_client(self.Parameters.cluster)
        yt_path = self.Parameters.yt_path
        packages_path = "{}/@packages".format(yt_path)
        geobases_path = "{}/@geobases".format(yt_path)
        if cluster_client.exists(packages_path) and cluster_client.exists(geobases_path):
            packages_armed = cluster_client.get(packages_path)
            logging.debug("packages_armed: {}".format(dict(packages_armed)))
            logging.debug("packages_from_meta: {}".format(dict(self.Context.packages_from_meta)))
            logging.debug("packages_in_sync: {}".format(dict(packages_armed) == dict(self.Context.packages_from_meta)))
            if not self.Context.skip_geobase:
                geobases_armed = cluster_client.get(geobases_path)
                logging.debug("geobases_armed: {}".format(set(self.Context.geobases_from_meta)))
                logging.debug("geobases_from_meta: {}".format(set(geobases_armed)))
                logging.debug("geobases_in_sync: {}".format(set(geobases_armed) == set(self.Context.geobases_from_meta)))
            if dict(packages_armed) == dict(self.Context.packages_from_meta) and \
                    (self.Context.skip_geobase or set(geobases_armed) == set(self.Context.geobases_from_meta)):
                return True

    def on_execute(self):
        if self.Parameters.skip_if_synced:
            self.meta_client = self._get_yt_client(self.META_CLUSTER)
            if not self.meta_client.exists("{}/{}/p2p-geodata".format(self.PACKAGES_META_PATH, self.Parameters.cluster)):
                self.Context.skip_geobase = True
            self.fetch_resources_from_meta()
            if self.resources_are_synced_with_meta():
                return

        if not self.Context.build_layer_task_id:
            layer_build_params = {
                BuildPortoLayer.ParentLayer.name: self.Parameters.parent_layer,
                BuildPortoLayer.LayerType.name: self.Parameters.layer_type,
                BuildPortoLayer.LayerName.name: self.Parameters.layer_name,
                BuildPortoLayer.Compress.name: self.Parameters.compress,
                BuildPortoLayer.ScriptUrl.name: self.Parameters.script_url,
                BuildPortoLayer.ScriptEnv.name: {'YT_CLUSTER': self.Parameters.cluster},
                BuildPortoLayer.MergeLayers.name: self.Parameters.merge_all_layers,
                BuildPortoLayer.DebugBuild.name: self.Parameters.debug_build,
                BuildPortoLayer.SpaceLimit.name: self.Parameters.space_limit,
                BuildPortoLayer.MemoryLimit.name: self.Parameters.memory_limit,
            }

            logging.debug("BUILD_PORTO_LAYER build params: %s", str(layer_build_params))
            task_class = sdk2.Task['BUILD_PORTO_LAYER']
            layer_build_task = task_class(
                self,
                description=self.Parameters.description,
                **{
                    key: value.id if isinstance(value, sdk2.Resource) else value
                    for key, value in layer_build_params.iteritems()
                }
            ).enqueue()
            self.Context.build_layer_task_id = layer_build_task.id

            logging.debug("BUILD_PORTO_LAYER build task: %d", layer_build_task.id)

            raise sdk2.WaitTask([self.Context.build_layer_task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            if not all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.build_layer_task_id)):
                raise common.errors.TaskFailure("Build porto layer failed")

            if self.Parameters.release_task:
                parent_layer = sdk2.Resource.find(id=self.Parameters.parent_layer.id).first()
                if not parent_layer.state == State.READY:
                    raise common.errors.TaskFailure("Parent layer: {} is not ready.".format(parent_layer.id))
                if parent_layer.released not in [ctt.ReleaseStatus.STABLE]:
                    raise common.errors.TaskFailure("Parent layer: {} is not released stable.".format(parent_layer.id))
                with self.memoize_stage.release_task(commit_on_entrance=False):
                    self.server.release(task_id=self.Context.build_layer_task_id, type=ctt.ReleaseStatus.STABLE, subject="[{}] Porto Layer YT Default User RootFS.".format(self.Parameters.cluster))

            layer_resource = sdk2.Resource.find(task_id=self.Context.build_layer_task_id, type=self.Parameters.layer_type).first()
            skynet_get(layer_resource.skynet_id, '.')

            yt_client = self._get_yt_client(self.Parameters.cluster)
            with open(os.path.join(".", os.path.basename("{}.{}".format(self.Parameters.layer_name, self.Parameters.compress)))) as f:
                yt_client.write_file(self.Parameters.yt_path, f)
            yt_client.set("{}/@sandbox_resource_id".format(self.Parameters.yt_path), layer_resource.id)

            if self.Parameters.skip_if_synced:
                yt_client.set("{}/@packages".format(self.Parameters.yt_path), self.Context.packages_from_meta)
                if not self.Context.skip_geobase:
                    yt_client.set("{}/@geobases".format(self.Parameters.yt_path), self.Context.geobases_from_meta)
