# -*- coding: utf-8 -*-

import os
import shutil
import logging

from sandbox import sdk2
import sandbox.common as common
import sandbox.common.types.task as ctt
from sandbox.projects.common.nanny import nanny
from sandbox.projects.yt.common.tasks.YtBasicTask import YtBasicTask


class YtBuildYaPackage(nanny.ReleaseToNannyTask2, YtBasicTask):
    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia url", required=True)
        do_not_remove = sdk2.parameters.Bool("Set 'do not remove' for resources", default_value=False)
        startrack_ticket_ids = sdk2.parameters.List("Startrek tickets", required=False)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (doc: https://nda.ya.ru/3QTTV4)", required=False, default="", multiline=True)

    class Requirements(sdk2.Requirements):
        disk_space = 25 * 1024  # 25GB by default

    class CustomParameters(object):
        package_type = "tarball"
        use_aapi_fuse = False

        ttl = 30
        package_path = ""
        package_name = ""
        comment_to_description = ""
        description = ""
        revision = None
        output_resource = "YT_PACKAGE_RESOURCE"

        force_dupload = False
        publish_package = False
        publish_to = ""
        key_user = ""

        checkout_mode = "manual"

    def on_prepare(self):
        self.CustomParameters.revision = self._parse_arcadia_url(self.Parameters.arcadia_url).revision

    def on_execute(self):
        self._read_package_json()

        with self.memoize_stage.create_children:
            sub_tasks = [self._run_sub_task(), ]
            raise sdk2.WaitTask(
                sub_tasks,
                set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)),
                wait_all=True)

        ya_package_sub_task = self.find().first()

        if ya_package_sub_task.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure("Subtask unsuccessfully finished with status {}".format(ya_package_sub_task.status))

        self._copy_resource_from_sub_task(ya_package_sub_task)

    def _copy_resource_attributes(self, resource):
        attributes = [
            "branch", "build_type", "platform",
            "resource_name", "resource_version",
            "svn_path", "svn_revision"
        ]
        logging.debug("MARK0")
        logging.debug(resource.svn_revision)
        copied_attributes = {}
        for attribute in attributes:
            if hasattr(resource, attribute):
                logging.debug("-- {}:{} --".format(attribute, getattr(resource, attribute)))
                copied_attributes.update({attribute: getattr(resource, attribute)})
        logging.debug("MARK")
        logging.debug(copied_attributes)
        logging.debug(resource.to_dict())
        logging.debug(dir(resource))
        return copied_attributes

    def _copy_resource_from_sub_task(self, sub_task):
        sub_task_resource = sdk2.Resource[self.CustomParameters.output_resource].find(task=sub_task).first()
        sub_task_resource_data = sdk2.ResourceData(sub_task_resource)
        sub_task_data_path = str(sub_task_resource_data.path)

        new_resource_data_path = self._get_path(os.path.basename(sub_task_data_path))
        if os.path.isfile(sub_task_data_path):
            shutil.copy(sub_task_data_path, new_resource_data_path)
        else:
            shutil.copytree(sub_task_data_path, new_resource_data_path)

        sdk2.Resource[self.CustomParameters.output_resource](
            self,
            self.CustomParameters.description,
            new_resource_data_path,
            ttl='inf' if self.Parameters.do_not_remove else self.CustomParameters.ttl,
            **self._copy_resource_attributes(sub_task_resource)
        )

    def _read_package_json(self):
        self._arcadia_export(self.CustomParameters.package_path, "package.json", self.CustomParameters.revision)
        package_json = self._load_json("package.json")

        package_name = None
        package_description = None
        if "meta" in package_json:
            package_name = package_json["meta"].get("name", None)
            package_description = package_json["meta"].get("description", None)

        description = ""
        if package_description:
            description = "{}".format(package_description)
        if package_name:
            description = "{} ({})".format(description, package_name)
        if self.CustomParameters.comment_to_description:
            description = "{}. {}".format(description, self.CustomParameters.comment_to_description)
        self.CustomParameters.description = description
        self.CustomParameters.package_name = package_name

    def _run_sub_task(self):
        checkout = not self.Parameters.arcadia_patch
        ya_package_task = sdk2.Task["YA_PACKAGE"](
            self,
            description=self.CustomParameters.description,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            packages=self.CustomParameters.package_path,
            use_new_format=True,
            package_type=self.CustomParameters.package_type,
            compress_package_archive=True,
            resource_type=self.CustomParameters.output_resource,
            force_dupload=self.CustomParameters.force_dupload,
            publish_package=self.CustomParameters.publish_package,
            publish_to=self.CustomParameters.publish_to,
            key_user=self.CustomParameters.key_user,
            arcadia_patch=self.Parameters.arcadia_patch,
            checkout=checkout,
            checkout_mode=self.CustomParameters.checkout_mode,
            use_aapi_fuse=self.CustomParameters.use_aapi_fuse,
        )
        ya_package_task.Requirements.disk_space = self.Requirements.disk_space
        return ya_package_task.save().enqueue()
