import logging
import os
import copy

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.build.YaMake2 import YaMake2

logger = logging.getLogger('yt_upload')


class YtYaMakeUpload(YaMake2):
    class Parameters(YaMake2.Parameters):
        with sdk2.parameters.Group("YT parameters:") as yt_upload_params:
            yt_proxy_list = sdk2.parameters.List("List of upload clutser proxies", required=True)
            yt_destination_path = sdk2.parameters.String(
                "Destination directory (not full path!) on cluster",
                required=True,
            )
            yt_token = sdk2.parameters.YavSecret("Yav secret with YT token")
            with sdk2.parameters.Output:
                yt_outputs = sdk2.parameters.JSON(
                    "YT-related information about uploaded artifacts", required=True
                )

    class Requirements(YaMake2.Requirements):
        environments = [environments.PipEnvironment("yandex-yt")]

    def post_build(self, source_dir, output_dir, pack_dir):
        super(YtYaMakeUpload, self).post_build(source_dir, output_dir, pack_dir)
        import yt.logger
        yt.logger.LOGGER = logging.getLogger('yt')
        self.Context.yt_outputs = []

        try:
            for proxy in self.Parameters.yt_proxy_list:
                self._upload(output_dir, proxy)
        except Exception:
            logger.exception("Caught exception while uploading:")
            raise

        self.Parameters.yt_outputs = self.Context.yt_outputs

    def get_destination_file_name(self, file_path):
        return os.path.basename(file_path)

    def get_extra_attributes(self, file_path):
        return {}

    def _upload(self, output_dir, proxy):
        import yt.wrapper as yt

        logger.debug("Uploading to %s", proxy)

        token = self.Parameters.yt_token.data()["yt.token"]
        logger.debug("Token = %s, type = %s", token, type(token))
        client = yt.YtClient(proxy=proxy, token=token)
        sandbox_attributes = self.get_base_resource_attrs()
        sandbox_attributes.update({
            "task_id": self.id,
        })
        common_attributes = {
            "sandbox_attributes": sandbox_attributes,
        }

        for art in self.get_arts():
            path = os.path.abspath(os.path.join(output_dir, art["path"]))

            filename = self.get_destination_file_name(path)
            extra_attributes = self.get_extra_attributes(path)

            attributes = copy.deepcopy(common_attributes)
            attributes.update(extra_attributes)

            dst_path = self.Parameters.yt_destination_path + "/" + filename

            logger.debug("Upload artifact %s to %s", path, dst_path)

            file_id = client.create("file", dst_path, attributes=attributes, force=True)
            with open(path, "rb") as f:
                client.write_file(
                    dst_path,
                    f,
                    filename_hint=filename)

            info = {
                "artifact": art["path"],
                "yt_cluster": proxy,
                "yt_filename": filename,
                "yt_path": path,
                "yt_object_id": file_id,
            }
            self.Context.yt_outputs.append(info)
