import logging
import tarfile
import os
import tempfile
import re
import subprocess
import json

from sandbox.sdk2.vcs.svn import Arcadia


class YtOperationExecutor:
    def __init__(self, cluster, token):
        self.client = self._get_yt_client(cluster, token)

    @staticmethod
    def _get_yt_client(cluster, token):
        from yt.wrapper import YtClient
        return YtClient(cluster, token=token)

    def write_to_yt_from_local_file(self, yt_path, local_path):
        logging.debug("Start writing in {} file from {}".format(yt_path, local_path))
        with open(local_path, 'rb') as f:
            self.client.write_file(yt_path, f)
        logging.debug("End of writing")

    def get_ordered_list_of_base_layers(self, directory, start_of_layer_name):
        creation_time_attr = 'creation_time'
        type_attr = 'type'
        all_releases = self.client.list(directory, attributes=[creation_time_attr, type_attr])
        all_releases.sort(key=lambda r: r.attributes[creation_time_attr])
        all_releases_with_name = [r for r in all_releases
                                  if r.startswith(start_of_layer_name) and r.attributes[type_attr] == "file"]

        return all_releases_with_name[::-1]

    def get_last_release(self, directory, start_of_layer_name):
        files = self.get_ordered_list_of_base_layers(directory, start_of_layer_name)
        if len(files) == 0:
            return None

        return "{}/{}".format(directory, files[0])

    def remove_files_from_directory(self, directory, files_for_remove):
        logging.debug("Start removing {} files from {} directory".format(len(files_for_remove), directory))
        for file_name in files_for_remove:
            self.client.remove(directory + "/" + file_name)
        logging.debug("End of removing")

    def remove_file(self, file_for_remove):
        self.client.remove(file_for_remove)

    def get_file_attributes_info(self, file_path, attributes):
        attribute_dict = dict()

        for attr in attributes:
            attr_path = "{}/@{}".format(file_path, attr)
            if self.client.exists(attr_path):
                attribute_dict[attr] = self.client.get(attr_path)

        return attribute_dict

    def set_attributes_info(self, file_path, attribute_dict):
        logging.debug("Start setting attributes for file {}".format(file_path))
        for attribute_name in attribute_dict:
            self.client.set("{}/@{}".format(file_path, attribute_name), attribute_dict[attribute_name])

    def update_link(self, link_path, target_path):
        self.client.link(target_path, link_path, force=True)


class InfoAboutUnpackedFiles:
    def __init__(self, abspath, arcname_path):
        self.arcname_path = arcname_path
        self.abspath = abspath


class FileLocationInfo:
    def __init__(self, source_file_path, additional_dirs):
        self.additional_dirs = additional_dirs
        self.source_file_path = source_file_path


class TarHelper:
    @staticmethod
    def unpack_tar_data(tar_path):
        unpacked_path = tempfile.mkdtemp()
        with tarfile.open(tar_path) as archive:
            archive.extractall(unpacked_path)
        return unpacked_path

    @staticmethod
    def get_all_files_from_directory(dir_name, current_dir=""):
        list_of_file = os.listdir(dir_name)
        all_files = list()
        for entry in list_of_file:
            full_path = os.path.join(dir_name, entry)
            if os.path.isdir(full_path):
                all_files = all_files + TarHelper.get_all_files_from_directory(full_path,
                                                                               os.path.join(current_dir, entry))
            else:
                all_files.append(InfoAboutUnpackedFiles(full_path, os.path.join(current_dir, entry)))

        return all_files

    @staticmethod
    def make_tarfile(tar_file, source_files):
        with tarfile.open(tar_file, "w:gz") as tar:
            for source_file_info in source_files:
                inside_dirs = source_file_info.additional_dirs
                source_file = source_file_info.source_file_path

                if os.path.isfile(source_file) and tarfile.is_tarfile(source_file):
                    unpacked_dir = TarHelper.unpack_tar_data(source_file)
                    all_files_from_dir = TarHelper.get_all_files_from_directory(unpacked_dir)
                    for unpacked_file_info in all_files_from_dir:
                        file_path = os.path.join(*(inside_dirs + [unpacked_file_info.arcname_path]))
                        tar.add(unpacked_file_info.abspath, arcname=file_path)
                else:
                    file_path = os.path.join(*(inside_dirs + [os.path.basename(source_file)]))
                    tar.add(source_file, arcname=file_path)


class DynamicLibraryMappingBuilder(object):

    def __init__(self, root):
        self.mapping = {}
        self.root = root
        self.directory = os.path.join(self.root, "opt/yt-libs")
        self.mapping_directory = os.path.join(self.directory, "configs")

    def _is_library(self, filename):
        return re.match(r"""lib[\w\-]+\.so""", filename) is not None

    def _get_version(self, path):
        try:
            filename = os.path.basename(path)
            version_string = filename[filename.index(".so.") + 4:]
            return tuple(map(int, version_string.split(".")))
        except ValueError:
            return None

    def _get_soname(self, path):
        out = subprocess.check_output(["objdump", "-p", path])
        soname_regex = re.compile(r"""^\s*SONAME\s*(?P<soname>[\w.\-]+)\s*""", re.MULTILINE)
        match = soname_regex.search(out)
        return match.group("soname") if match else None

    def _drop_version(self, name):
        version_regex = re.compile(r"""(\.\d+)*$""")
        return version_regex.sub("", name)

    def _update_entry(self, soname, path):
        try:
            final_path = "/" + os.path.relpath(path, self.root)
            if soname not in self.mapping or self._get_version(self.mapping[soname]) < self._get_version(final_path):
                self.mapping[soname] = final_path
                self.mapping[self._drop_version(soname)] = final_path
        except ValueError:
            pass

    def add_directory(self, path):
        for filename in os.listdir(path):
            file_path = os.path.join(path, filename)
            if os.path.isdir(file_path):
                self.add_directory(file_path)
            elif os.path.isfile(file_path) and self._is_library(filename):
                soname = self._get_soname(file_path)
                if soname is not None:
                    self._update_entry(soname, file_path)

    def generate_library_mapping(self, mapping_file_name):
        mapping_file_path = os.path.join(self.mapping_directory, mapping_file_name)
        if not os.path.isdir(self.mapping_directory):
            os.makedirs(self.mapping_directory)
        with open(mapping_file_path, "w+") as f:
            json.dump(self.mapping, f, indent=4)

    def put_patcher_script(self, script_arcadia_url):
        script_path = os.path.join(self.directory, "patch_ldcache.py")
        script = Arcadia.cat(script_arcadia_url)
        with open(script_path, "w+") as f:
            f.write(script)
        os.chmod(script_path, 0o755)
