import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt


class BaseLayerDistributedCreatorTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            'Target YT cluster',
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            'YT token vault name',
            required=True,
        )
        yt_path = sdk2.parameters.String(
            'YT cypress path',
            default="//home/arapova/porto_layers",
            required=True,
        )

        default_resource_types = [
            "PORTO_LAYER_SEARCH_UBUNTU_PRECISE_APP",
            "PORTO_LAYER_SEARCH_UBUNTU_TRUSTY_APP",
            "PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP",
            "PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP"
        ]
        parent_layers_resource_types = sdk2.parameters.List(
            'Types of parent layers resources',
            sdk2.parameters.String,
            default=default_resource_types,
            required=True,
        )
        script_url = sdk2.parameters.ArcadiaUrl(
            'Setup script URL',
            required=True,
        )

        releases_count = sdk2.parameters.Integer(
            "Limit count of releases, which were saved in yt",
            default=10,
            required=True,
        )
        layer_type = sdk2.parameters.String(
            'Base layer with extra packages type',
            default='PORTO_LAYER_YT',
            required=True,
        )

    @staticmethod
    def get_linux_type(resource_type):
        linux_types = ["trusty", "precise", "xenial", "bionic"]
        for linux_type in linux_types:
            if linux_type in resource_type.lower():
                return linux_type

    def _get_default_params_for_task(self, resource_type):
        return {
            "cluster": self.Parameters.cluster,
            "yt_token_vault_name": self.Parameters.yt_token_vault_name,
            "yt_path": "{}/{}".format(self.Parameters.yt_path, self.get_linux_type(resource_type)),
            "releases_count": self.Parameters.releases_count,
            "layer_type": self.Parameters.layer_type,
            "script_url": self.Parameters.script_url
        }

    @staticmethod
    def _find_resource(resource_type):
        return sdk2.Resource\
            .find(type=resource_type, state='READY', attrs={"released": "stable"})\
            .order(-sdk2.Resource.id)\
            .first()

    def on_execute(self):
        if not self.Context.start_child_task:
            layer_build_tasks_id = []
            for resource_type in self.Parameters.parent_layers_resource_types:
                resource = self._find_resource(resource_type)

                if resource is None:
                    logging.error("For type {} did not find resource".format(resource_type))
                    continue
                logging.info("For type {} find resource with id {}".format(str(resource_type), resource.id))

                layer_build_task = sdk2.Task['BASE_LAYER_CREATOR_TASK'](
                    self,
                    description="Task for resource_type {}".format(resource_type),
                    parent_layer=resource.id,
                    layer_name="{}".format(resource_type.lower()),
                    **self._get_default_params_for_task(resource_type.lower())

                ).enqueue()

                logging.debug("BASE_LAYER_CREATOR_TASK build task: {}".format(str(layer_build_task.id)))
                layer_build_tasks_id.append(layer_build_task.id)

            self.Context.start_child_task = True
            raise sdk2.WaitTask(layer_build_tasks_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
