import logging
import datetime

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.porto import BuildPortoLayer
from sandbox.common.types.resource import State
from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import YtOperationExecutor


class BaseLayerAttribute:
    ParentLayerAttributeName = "sandbox_parent_layer_id"
    ScriptAttributeName = "svn_script_commit_number"
    BuildPortoLayerTaskId = "build_layer_task_id"

    def __init__(self, parent_layer_attribute_value, script_attribute_value):
        self.additional_attributes = dict()

        self.script_attribute_value = str(script_attribute_value).strip()
        self.parent_layer_attribute_value = str(parent_layer_attribute_value).strip()

    def __eq__(self, other):
        return \
            self.parent_layer_attribute_value == other.parent_layer_attribute_value and \
            self.script_attribute_value == other.script_attribute_value

    def add_additional_attribute(self, attribute_name, attribute_value):
        self.additional_attributes[attribute_name] = attribute_value

    def log_attribute_values(self):
        d = self.as_dictionary_of_attributes()
        for attr_name, attr_value in d.iteritems():
            logging.info("{} is {}".format(attr_name, str(attr_value)))

    def as_dictionary_of_attributes(self):
        current_attr = {
            BaseLayerAttribute.ParentLayerAttributeName: self.parent_layer_attribute_value,
            BaseLayerAttribute.ScriptAttributeName: self.script_attribute_value
        }
        current_attr.update(self.additional_attributes)
        return current_attr

    @staticmethod
    def get_all_attributes_name():
        return [
            BaseLayerAttribute.ParentLayerAttributeName,
            BaseLayerAttribute.ScriptAttributeName,
        ]

    @staticmethod
    def from_dictionary_of_attributes(attr_dict):
        return BaseLayerAttribute(
            attr_dict[BaseLayerAttribute.ParentLayerAttributeName],
            attr_dict[BaseLayerAttribute.ScriptAttributeName],
        )


class BaseLayerCreatorTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        def resource_types_with_prefixes(*args):
            return [resource for resource in sdk2.Resource if any(resource.name.startswith(arg) for arg in args)]

        layer_compress = 'tar.gz'

        cluster = sdk2.parameters.String(
            'Target YT cluster',
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            'YT token vault name',
            required=True,
        )
        yt_path = sdk2.parameters.String(
            'YT cypress path',
            default="//home/arapova/porto_layers",
            required=True,
        )
        releases_count = sdk2.parameters.Integer(
            "Limit count of releases, which were saved in yt",
            default=10,
            required=True,
        )

        parent_layer = sdk2.parameters.Resource(
            "Parent layer",
            resource_type=resource_types_with_prefixes('PORTO_LAYER'),
            state=State.READY,
            required=True,
        )

        layer_type = sdk2.parameters.String(
            'Base layer with extra packages type',
            default='PORTO_LAYER_YT',
            required=True,
        )
        layer_name = sdk2.parameters.String(
            'Base layer with extra packages name',
            default='base_layer_with_delta',
            required=True,
        )
        script_url = sdk2.parameters.ArcadiaUrl(
            'Setup script URL',
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def _get_name_in_yt_with_current_datetime(self):
        today = datetime.datetime.today().strftime("%Y-%m-%d-%H.%M.%S")
        return "{}-{}".format(self.Parameters.layer_name, today)

    def _get_last_release_with_same_parent_layer(self, yt_operations_executor):
        directory = self.Parameters.yt_path
        base_layers = yt_operations_executor.get_ordered_list_of_base_layers(directory, self.Parameters.layer_name)
        if len(base_layers) != 0:
            return "{}/{}".format(directory, base_layers[0])
        return None

    def _get_actual_resources_info(self):
        script_svn_commit_number = dict(sdk2.svn.Arcadia.info(self.Parameters.script_url))['commit_revision']
        parent_layer_resource_id = self.Parameters.parent_layer.id

        return BaseLayerAttribute(parent_layer_resource_id, script_svn_commit_number)

    def _get_resource_info_for_last_release(self, yt_operations_executor):
        last_released_layer_path = self._get_last_release_with_same_parent_layer(yt_operations_executor)
        if last_released_layer_path is None:
            return BaseLayerAttribute("", "")

        attr_dict = yt_operations_executor.get_file_attributes_info(
            last_released_layer_path,
            BaseLayerAttribute.get_all_attributes_name())

        return BaseLayerAttribute.from_dictionary_of_attributes(attr_dict)

    def _remove_old_releases_for_same_parent_layer(self, yt_operations_executor):
        directory = self.Parameters.yt_path
        releases_for_remove = yt_operations_executor.get_ordered_list_of_base_layers(
            directory, self.Parameters.layer_name)[int(self.Parameters.releases_count) + 1:]
        yt_operations_executor.remove_files_from_directory(directory, releases_for_remove)

    def create_build_porto_layer_task(self, layer_name_with_datetime):
        layer_build_params = {
            BuildPortoLayer.ParentLayer.name: self.Parameters.parent_layer.id,
            BuildPortoLayer.LayerType.name: self.Parameters.layer_type,
            BuildPortoLayer.LayerName.name: layer_name_with_datetime,
            BuildPortoLayer.Compress.name: self.Parameters.layer_compress,
            BuildPortoLayer.ScriptUrl.name: self.Parameters.script_url,
            BuildPortoLayer.MergeLayers.name: True,
            BuildPortoLayer.DebugBuild.name: False,
            BuildPortoLayer.OutputResourceAttr.name: {
                "name_without_datetime": self.Parameters.layer_name,
            },
        }
        logging.debug("BUILD_PORTO_LAYER build params: {}".format(str(layer_build_params)))

        build_porto_layer_task = sdk2.Task['BUILD_PORTO_LAYER']
        layer_build_task = build_porto_layer_task(
            self,
            description=self.Parameters.description,
            **layer_build_params).enqueue()
        logging.debug("BUILD_PORTO_LAYER build task: {}".format(str(layer_build_task.id)))

        return layer_build_task.id

    def on_execute(self):
        yt_operations_executor = YtOperationExecutor(
            self.Parameters.cluster,
            sdk2.Vault.data(self.Parameters.yt_token_vault_name))

        logging.debug(self.Parameters.parent_layer.name)

        actual_resource_info = self._get_actual_resources_info()
        resource_info_for_last_release = self._get_resource_info_for_last_release(yt_operations_executor)

        if actual_resource_info == resource_info_for_last_release:
            logging.info("Resources for new release are the same to previous release layer")
            actual_resource_info.log_attribute_values()
            return

        if not self.Context.build_layer_task_id:
            local_layer_path = self._get_name_in_yt_with_current_datetime()
            self.Context.build_layer_task_id = self.create_build_porto_layer_task(local_layer_path)
            self.Context.layer_name_with_datetime = "{}.{}".format(
                local_layer_path, self.Parameters.layer_compress)

            raise sdk2.WaitTask([self.Context.build_layer_task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
        else:
            actual_resource_info.add_additional_attribute(
                BaseLayerAttribute.BuildPortoLayerTaskId,
                self.Context.build_layer_task_id)

            built_porto_layer_resource = sdk2.Resource \
                .find(task_id=self.Context.build_layer_task_id, type=self.Parameters.layer_type, state=State.READY) \
                .first()
            self.server.release(task_id=self.Context.build_layer_task_id, type=ctt.ReleaseStatus.STABLE,
                                subject=self.Parameters.description)
            logging.info("id of built porto layer is {}".format(str(built_porto_layer_resource.id)))

            local_layer_path = str(sdk2.ResourceData(built_porto_layer_resource).path)
            yt_layer_path = "{}/{}".format(self.Parameters.yt_path, self.Context.layer_name_with_datetime)

            with yt_operations_executor.client.Transaction() as tx:
                logging.info('Transaction {} for write layer to yt with attributes'.format(str(tx.transaction_id)))

                yt_operations_executor.write_to_yt_from_local_file(yt_layer_path, local_layer_path)
                yt_operations_executor.set_attributes_info(yt_layer_path,
                                                           actual_resource_info.as_dictionary_of_attributes())
                yt_operations_executor.update_link(
                    "{}/{}_lastest.{}".format(
                        self.Parameters.yt_path, self.Parameters.layer_name, self.Parameters.layer_compress),
                    yt_layer_path
                )

            self._remove_old_releases_for_same_parent_layer(yt_operations_executor)
