import logging
import json
import os
import re
from shutil import copy
import subprocess
import tempfile

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments

from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import \
    YtOperationExecutor, TarHelper, FileLocationInfo, DynamicLibraryMappingBuilder


class YtBuildCudaLayer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            "Target YT cluster",
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            required=True,
        )
        yt_directory_path = sdk2.parameters.String(
            "YT path layer location",
            default="//porto_layers/gpu_drivers",
            required=True,
        )
        toolkit_version = sdk2.parameters.String(
            "CUDA toolkit version (aka layer name)",
            required=True,
        )
        toolkit_resource = sdk2.parameters.Resource(
            "CUDA toolkit archive resource",
            required=True,
        )
        patcher_script_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL of ld.so.cache patcher script",
            default_value="arcadia:/arc/trunk/arcadia/yt/scripts/patch_ldcache/patch_ldcache.py"
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def _get_toolkit_archive_path(self):
        return str(sdk2.ResourceData(self.Parameters.toolkit_resource).path)

    def _unpack_cuda_toolkit(self):
        # We don't use TarHelper here since it does not support .tar.xz format because of Python 2.7 limitations
        unpacked_path = tempfile.mkdtemp()
        subprocess.check_call(["tar", "-C", unpacked_path, "-xf", self._get_toolkit_archive_path()])
        return unpacked_path

    def _get_actual_toolkit_version(self, toolkit_files_path):
        version = None

        version_txt = os.path.join(toolkit_files_path, "version.txt")
        if os.path.exists(version_txt):
            with open(version_txt) as f:
                version_regex = re.compile(r"""^CUDA Version\s*(?P<version>(\d+\.)+\d+).*$""", re.MULTILINE)
                m = version_regex.search(f.readline())
                version = m.group("version")

        version_json = os.path.join(toolkit_files_path, "version.json")
        if os.path.exists(version_json):
            with open(version_json) as f:
                version = json.load(f)["cuda"]["version"]

        if version is None:
            raise RuntimeError("Version files are missing (tried: [\"version.txt\", \"version.json\"])")

        major, minor = tuple(version.split(".")[:2])
        return "{}.{}".format(major, minor)

    def _check_toolkit_version(self, toolkit_files_path):
        actual_version = self._get_actual_toolkit_version(toolkit_files_path)
        if self.Parameters.toolkit_version != actual_version:
            raise common.errors.TaskFailure("Toolkit version mismatch: expected {}, actual {}".format(
                self.Parameters.toolkit_version,
                actual_version
            ))

    def _copy_library_files(self, toolkit_files_path, layer_root_path):
        libs_dir_path = os.path.join(layer_root_path, "opt", "yt-nvidia", "lib", "cuda")
        ld_so_conf_d_path = os.path.join(layer_root_path, "etc", "ld.so.conf.d")
        os.makedirs(libs_dir_path)
        os.makedirs(ld_so_conf_d_path)

        lib_file_re = re.compile(r""".*\.so[\d.]*""")
        lib_files_path = os.path.join(toolkit_files_path, "lib64")
        # os.listdir is used because we don't need recursive traversal of toolkit files
        # since in this case we would also pick stub files
        for filename in os.listdir(lib_files_path):
            if lib_file_re.match(filename):
                full_path = os.path.join(lib_files_path, filename)
                if os.path.islink(full_path):
                    continue
                copy(full_path, os.path.join(libs_dir_path, filename))

        with open(os.path.join(ld_so_conf_d_path, "yt-nvidia-cuda.conf"), "w+") as ld_conf:
            ld_conf.write("/opt/yt-nvidia/lib/cuda")

    def _create_library_mapping(self, layer_root_path):
        builder = DynamicLibraryMappingBuilder(layer_root_path)
        library_path = os.path.join(layer_root_path, "opt", "yt-nvidia", "lib", "cuda")
        builder.add_directory(library_path)
        builder.generate_library_mapping("yt-nvidia-cuda.json")
        builder.put_patcher_script(self.Parameters.patcher_script_url)

    def _create_layer_archive(self, layer_root_path):
        source_files = []

        for filename in os.listdir(layer_root_path):
            path = os.path.join(layer_root_path, filename)
            source_files.append(FileLocationInfo(path, []))

        layer_filename = self.Parameters.toolkit_version
        logging.info("Creating archive {}".format(layer_filename))
        TarHelper.make_tarfile(layer_filename, source_files)

    def _upload_layer(self):
        yt_operation_executor = YtOperationExecutor(
            cluster=self.Parameters.cluster,
            token=sdk2.Vault.data("YT_ROBOT", self.Parameters.yt_token_vault_name),
        )

        layer_filename = self.Parameters.toolkit_version
        yt_path = os.path.join(self.Parameters.yt_directory_path, layer_filename)

        logging.info("Uploading layer to YT")
        yt_operation_executor.write_to_yt_from_local_file(yt_path, layer_filename)

    def on_execute(self):
        layer_root_path = "layer_root"
        toolkit_files_path = self._unpack_cuda_toolkit()

        self._check_toolkit_version(toolkit_files_path)

        self._copy_library_files(toolkit_files_path, layer_root_path)
        self._create_library_mapping(layer_root_path)
        self._create_layer_archive(layer_root_path)
        self._upload_layer()
