import os
import os.path
import logging
import datetime
import hashlib

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox import common

from sandbox.projects.geobase import Geodata4BinStable, Geodata5BinStable, Geodata6BinStable
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import YtOperationExecutor, FileLocationInfo, TarHelper

TASK_VERSION = 2


def calculate_hash(file_path):
    BLOCKSIZE = 65536
    md5 = hashlib.md5()
    with open(file_path, 'rb') as file:
        file_buffer = file.read(BLOCKSIZE)
        while len(file_buffer) > 0:
            md5.update(file_buffer)
            file_buffer = file.read(BLOCKSIZE)
    return md5.hexdigest()


class GeobaseLayerCreatorTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        prefix_layer_name = "layer_with_geodata"
        layer_compress = 'tar.gz'

        cluster = sdk2.parameters.String(
            'Target YT cluster',
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            'YT token vault name',
            required=True,
        )
        yt_path = sdk2.parameters.String(
            'YT path layer location',
            default='//home/arapova/geobase_layers',
            required=True,
        )
        releases_count = sdk2.parameters.Integer(
            "Limit count of releases, which were saved in yt",
            default=10,
            required=True,
        )
        geobase_test_arcadia_location = sdk2.parameters.String(
            "geobase test arcadia location",
            default="junk/arapova/geobase_layer_test/geobase_layer_test",
            required=True,
        )
        base_layer_path = sdk2.parameters.String(
            "base layer path, which need in geobase test",
            default="//home/arapova/ubuntu-xenial-base.tar.xz",
            required=True
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def on_prepare(self):
        self.geobase4 = self._find_last_released_resource(Geodata4BinStable.GEODATA4BIN_STABLE)
        self.geobase5 = self._find_last_released_resource(Geodata5BinStable.GEODATA5BIN_STABLE)
        self.geobase6 = self._find_last_released_resource(Geodata6BinStable.GEODATA6BIN_STABLE)
        self.tzdata = self._find_last_released_resource(GEODATATZDATA_STABLE)
        self.geobase_ling = self._find_last_released_resource(GEODATA_TREE_LING_STABLE)

    @staticmethod
    def _get_resource_data_(geobase_parameter):
        return str(sdk2.ResourceData(geobase_parameter).path)

    def _get_layer_filename_with_current_date(self):
        today = datetime.datetime.today().strftime("%Y-%m-%d-%H.%M.%S")
        return "{}-{}.{}".format(self.Parameters.prefix_layer_name, today, self.Parameters.layer_compress)

    def _get_info_about_resources(self):
        prefix = '/var/cache/geobase/'
        info_about_archive_resources = {
            'geobase4_archive_md5': self.geobase4.md5,
            'tzdata_archive_md5': self.tzdata.md5
        }

        geodata4_params = {
            'resource': prefix + 'geodata4.bin',
            'md5': self.geodate4_md5,
            "id": self.geobase4.id
        }
        geodata5_params = {
            'resource': prefix + 'geodata5.bin',
            'md5': self.geobase5.md5,
            "id": self.geobase5.id
        }
        geodata6_params = {
            'resource': prefix + 'geodata6.bin',
            'md5': self.geobase6.md5,
            "id": self.geobase6.id
        }
        geodata4_ling_params = {
            'resource': prefix + 'geodata4-tree+ling.bin',
            'md5': self.geobase_ling.md5,
            "id": self.geobase_ling.id
        }

        return {
            "info_about_files_in_layer": {
                "geodata4": geodata4_params,
                "geodata5": geodata5_params,
                "geodata6": geodata6_params,
                "geodata_ling": geodata4_ling_params
            },
            "info_about_archive_resources": info_about_archive_resources,
            "task_version": TASK_VERSION,
        }

    @staticmethod
    def _find_last_released_resource(resource_type):
        return sdk2.Resource \
            .find(type=resource_type, state='READY', attrs={"released": "stable"}) \
            .order(-sdk2.Resource.id) \
            .first()

    def _create_archive_with_geobase_layer(self, layer_filename):
        default_additional_dirs = ["var", "cache", "geobase"]

        geodata4_archive_path = self._get_resource_data_(self.geobase4)
        geodata5_path = self._get_resource_data_(self.geobase5)
        geodata6_path = self._get_resource_data_(self.geobase6)
        geodata_ling_path = self._get_resource_data_(self.geobase_ling)
        geodata_tzdata = self._get_resource_data_(self.tzdata)

        geodata4_path = os.path.join(TarHelper.unpack_tar_data(geodata4_archive_path), 'geodata4.bin')
        self.geodate4_md5 = calculate_hash(geodata4_path)

        # YTADMINREQ-18766.
        if not os.path.exists("usr/share"):
            os.makedirs("usr/share")
        os.symlink("../../var/cache/geobase/tzdata", "usr/share/geobase")

        list_of_files = [
            FileLocationInfo(geodata4_path, default_additional_dirs),
            FileLocationInfo(geodata5_path, default_additional_dirs),
            FileLocationInfo(geodata6_path, default_additional_dirs),
            FileLocationInfo(geodata_ling_path, default_additional_dirs),
            FileLocationInfo(geodata_tzdata, default_additional_dirs + ["tzdata"]),
            FileLocationInfo("usr/share/geobase", ["usr", "share"])]

        TarHelper.make_tarfile(layer_filename, list_of_files)

    def _remove_old_releases_of_geobase_layer(self, yt_operations_executor):
        directory = self.Parameters.yt_path
        releases_for_remove = yt_operations_executor.get_ordered_list_of_base_layers(
            directory, self.Parameters.prefix_layer_name)[int(self.Parameters.releases_count) + 1:]
        yt_operations_executor.remove_files_from_directory(directory, releases_for_remove)

    def _upload_geobase_layer_in_yt(self, layer_filename, yt_file_path, yt_operation_executor):
        logging.info('Create transaction')
        with yt_operation_executor.client.Transaction() as tx:
            logging.info('Transaction: {}'.format(str(tx.transaction_id)))

            logging.info("Write geobase layer in yt. Path: {}".format(yt_file_path))
            yt_operation_executor.write_to_yt_from_local_file(yt_file_path, layer_filename)

            attributes = self._get_info_about_resources()
            attributes['test_task_id'] = self.Context.test_task_id
            yt_operation_executor.client.set_attribute(yt_file_path, "geobase_info", attributes)

    def _is_current_geobase_data_equal_to_last_release(self, yt_operation_executor):
        last_released_file = yt_operation_executor\
            .get_last_release(self.Parameters.yt_path, self.Parameters.prefix_layer_name)
        if last_released_file is None:
            return False

        geobase_info = yt_operation_executor.client.get_attribute(last_released_file, attribute="geobase_info")
        if geobase_info is None:
            return False

        task_version = geobase_info.get("task_version", 0)
        geobase4_md5 = geobase_info["info_about_archive_resources"]["geobase4_archive_md5"]
        geobase5_md5 = geobase_info["info_about_files_in_layer"]["geodata5"]["md5"]
        geobase6_md5 = geobase_info["info_about_files_in_layer"]["geodata6"]["md5"]
        geodata4_long_md5 = geobase_info["info_about_files_in_layer"]["geodata_ling"]["md5"]
        tzdata_md5 = geobase_info["info_about_archive_resources"]["tzdata_archive_md5"]

        return \
            task_version == TASK_VERSION \
            and geobase4_md5 == self.geobase4.md5 \
            and geobase5_md5 == self.geobase5.md5 \
            and geobase6_md5 == self.geobase6.md5 \
            and geodata4_long_md5 == self.geobase_ling.md5 \
            and tzdata_md5 == self.tzdata.md5

    def _is_test_for_geobase_success(self):
        return all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.test_task_id))

    def create_test_task_for_geobase(self, layer_filename, yt_file_path):
        token_placeholder = '$(vault:value:{}:{})'.format(sdk2.Task.current.owner, self.Parameters.yt_token_vault_name)
        env_vars = "YT_TOKEN='{}' GEOBASE_LAYER_PATH='{}' CLUSTER='{}' BASE_LAYER='{}'" \
            .format(token_placeholder, yt_file_path, self.Parameters.cluster, self.Parameters.base_layer_path)

        test_task_for_geobase = sdk2.Task["YA_EXEC"](
            self,
            description="Test task for geobase layer {}".format(layer_filename),
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia",
            program=self.Parameters.geobase_test_arcadia_location,
            env_vars=env_vars,
        ).enqueue()

        return test_task_for_geobase.id

    def on_execute(self):
        yt_operation_executor = YtOperationExecutor(
            self.Parameters.cluster,
            sdk2.Vault.data("YT_ROBOT", self.Parameters.yt_token_vault_name),
        )

        if not self.Context.test_task_id:
            if self._is_current_geobase_data_equal_to_last_release(yt_operation_executor):
                logging.info("last release has the same resources")
                return

            layer_filename = self._get_layer_filename_with_current_date()
            yt_file_path = "{}/{}".format(self.Parameters.yt_path, layer_filename)

            self._create_archive_with_geobase_layer(layer_filename)
            logging.info("local create file with geobase layer in {}".format(layer_filename))

            test_geobase_task_id = self.create_test_task_for_geobase(layer_filename, yt_file_path)
            logging.debug("YA_EXEC build task: {}".format(str(test_geobase_task_id)))

            self.Context.test_task_id = test_geobase_task_id
            self.Context.yt_file_path = yt_file_path

            self._upload_geobase_layer_in_yt(layer_filename, yt_file_path, yt_operation_executor)

            raise sdk2.WaitTask([test_geobase_task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            if not self._is_test_for_geobase_success():
                yt_operation_executor.remove_file(self.Context.yt_file_path)
                raise common.errors.TaskFailure("Built porto layer with geobase is not valid")
            else:
                link_path = "{}/{}_lastest.{}" \
                    .format(self.Parameters.yt_path, self.Parameters.prefix_layer_name, self.Parameters.layer_compress)
                yt_operation_executor.update_link(link_path, self.Context.yt_file_path)
                self._remove_old_releases_of_geobase_layer(yt_operation_executor)
