import logging
import os
import re
import stat
from shutil import copy
import subprocess
import tarfile

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments

from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import \
    YtOperationExecutor, TarHelper, FileLocationInfo, DynamicLibraryMappingBuilder


class YtBuildGpuLayer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            "Target YT cluster",
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            required=True,
        )
        yt_directory_path = sdk2.parameters.String(
            "YT path layer location",
            default="//porto_layers/gpu_drivers",
            required=True,
        )
        driver_version = sdk2.parameters.String(
            "GPU driver version (aka layer name)",
            required=True,
        )
        driver_installer = sdk2.parameters.Resource(
            "Nvidia driver installer resource",
            required=True,
        )
        infiniband_libraries = sdk2.parameters.Resource(
            "Resource with infiniband libraries",
            default=None,
            required=True,
        )
        patcher_script_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL of ld.so.cache patcher script",
            default_value="arcadia:/arc/trunk/arcadia/yt/scripts/patch_ldcache/patch_ldcache.py"
        )
        binaries_to_copy = sdk2.parameters.List(
            "Driver binaries to copy from resource",
            default=[]
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def _get_driver_installer_path(self):
        installer_path = str(sdk2.ResourceData(self.Parameters.driver_installer).path)
        st = os.stat(installer_path)
        os.chmod(installer_path, st.st_mode | stat.S_IEXEC)
        return installer_path

    def _get_actual_driver_version(self):
        driver_installer_path = self._get_driver_installer_path()
        out = subprocess.check_output([driver_installer_path, "-v"])

        version_regex = re.compile(r"""^nvidia-installer:\s*version\s*(?P<version>(\d+\.)+\d+).*$""", re.MULTILINE)

        m = version_regex.search(out)
        return m.group("version")

    def _check_driver_version(self):
        actual_version = self._get_actual_driver_version()
        if self.Parameters.driver_version != actual_version:
            raise common.errors.TaskFailure("Driver version mismatch: expected {}, actual {}".format(
                self.Parameters.driver_version,
                actual_version
            ))

    def _extract_driver_files(self, driver_files_path):
        driver_installer_path = self._get_driver_installer_path()
        subprocess.check_call([driver_installer_path, "-x", "--target", driver_files_path])

    def _copy_library_files(self, driver_files_path, layer_root_path):
        libs_dir_path = os.path.join(layer_root_path, "opt", "yt-nvidia", "lib", "driver")
        ld_so_conf_d_path = os.path.join(layer_root_path, "etc", "ld.so.conf.d")
        os.makedirs(libs_dir_path)
        os.makedirs(ld_so_conf_d_path)

        lib_file_re = re.compile(r""".*\.so[\d.]*""")
        # os.listdir is used because we don't need recursive traversal of driver files
        # since in this case we would also pick up 32-bit libraries
        for filename in os.listdir(driver_files_path):
            if lib_file_re.match(filename):
                copy(os.path.join(driver_files_path, filename), os.path.join(libs_dir_path, filename))

        with open(os.path.join(ld_so_conf_d_path, "yt-nvidia-driver.conf"), "w+") as ld_conf:
            ld_conf.write("/opt/yt-nvidia/lib/driver")

    def _copy_binary_files(self, driver_files_path, layer_root_path):
        bin_dir_relative_path = os.path.join("opt", "yt-nvidia", "bin", "driver")
        bin_dir_path = os.path.join(layer_root_path, bin_dir_relative_path)
        bin_dir_root_path = os.path.join("/", bin_dir_relative_path)
        usr_bin_path = os.path.join(layer_root_path, "usr", "bin")

        os.makedirs(bin_dir_path)
        os.makedirs(usr_bin_path)

        for filename in self.Parameters.binaries_to_copy:
            copy(os.path.join(driver_files_path, filename), os.path.join(bin_dir_path, filename))
            os.symlink(os.path.join(bin_dir_root_path, filename), os.path.join(usr_bin_path, filename))

    def _copy_infiniband_library_files(self, layer_root_path):
        if self.Parameters.infiniband_libraries is None:
            return

        infiniband_archive = str(sdk2.ResourceData(self.Parameters.infiniband_libraries).path)

        with tarfile.open(infiniband_archive) as tar:
            tar.extractall(layer_root_path)

    def _create_library_mapping(self, layer_root_path):
        builder = DynamicLibraryMappingBuilder(layer_root_path)
        library_path = os.path.join(layer_root_path, "opt", "yt-nvidia", "lib", "driver")
        builder.add_directory(library_path)
        builder.generate_library_mapping("yt-nvidia-driver.json")
        builder.put_patcher_script(self.Parameters.patcher_script_url)

    def _create_layer_archive(self, layer_root_path):
        source_files = []

        for filename in os.listdir(layer_root_path):
            path = os.path.join(layer_root_path, filename)
            source_files.append(FileLocationInfo(path, []))

        layer_filename = self.Parameters.driver_version
        logging.info("Creating archive, name: {}".format(layer_filename))
        TarHelper.make_tarfile(layer_filename, source_files)

    def _upload_layer(self):
        yt_operation_executor = YtOperationExecutor(
            cluster=self.Parameters.cluster,
            token=sdk2.Vault.data("YT_ROBOT", self.Parameters.yt_token_vault_name),
        )

        layer_filename = self.Parameters.driver_version
        yt_path = os.path.join(self.Parameters.yt_directory_path, layer_filename)

        logging.info("Uploading layer to YT")
        yt_operation_executor.write_to_yt_from_local_file(yt_path, layer_filename)

    def on_execute(self):
        self._check_driver_version()

        driver_files_path = "driver_files"
        layer_root_path = "layer_root"

        self._extract_driver_files(driver_files_path)
        self._copy_binary_files(driver_files_path, layer_root_path)
        self._copy_library_files(driver_files_path, layer_root_path)
        self._copy_infiniband_library_files(layer_root_path)
        self._create_library_mapping(layer_root_path)
        self._create_layer_archive(layer_root_path)
        self._upload_layer()
