import logging
import datetime

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
import sandbox.common.types.task as ctt

from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import YtOperationExecutor, FileLocationInfo, TarHelper


class JdkLayerCreatorTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        prefix_layer_name = "layer_with_jdk"
        layer_compress = 'tar.gz'
        resource_jdk_type = "ARCADIA_PROJECT_BINARY"

        cluster = sdk2.parameters.String(
            'Target YT cluster',
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            'YT token vault name',
            required=True,
        )
        yt_path = sdk2.parameters.String(
            'YT path layer location',
            default='//home/arapova/jdk_layers',
            required=True,
        )
        releases_count = sdk2.parameters.Integer(
            "Limit count of releases, which were saved in yt",
            default=10,
            required=True,
        )

        jdk_versions = sdk2.parameters.List(
            'List of jdk versions',
            sdk2.parameters.String,
            default=['11'],
            required=True,
        )

        jdk_test_arcadia_location = sdk2.parameters.String(
            "jdk test arcadia location",
            default="junk/arapova/java_layer_test/run.sh",
            required=True,
        )
        jdk_test_program_params = sdk2.parameters.String(
            "jdk test program params",
            default="--java junk/arapova/java_layer_test/jdk/bin/java  Main",
            required=True,
        )
        base_layer_path = sdk2.parameters.String(
            "base layer path, which need in jdk test",
            default="//home/arapova/ubuntu-xenial-base.tar.xz",
            required=True
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    @staticmethod
    def _get_resource_path_(resource):
        return str(sdk2.ResourceData(resource).path)

    def _get_layer_name_with_current_datetime(self):
        today = datetime.datetime.today().strftime("%Y-%m-%d-%H.%M.%S")
        return "{}-{}.{}".format(self.Parameters.prefix_layer_name, today, self.Parameters.layer_compress)

    def _get_id_of_build_jdk_task(self, jdk_version):
        jdk_build_task = sdk2.Task["YA_MAKE"](
            self,
            description="ya make jdk version={}".format(jdk_version),
            definition_flags="-DJDK_VERSION={}".format(jdk_version),
            targets="jdk",
            arts="jdk/jdk{}.tar".format(jdk_version),
            result_single_file=True,
            result_rt=self.Parameters.resource_jdk_type
        ).enqueue()
        return jdk_build_task.id

    def _get_id_of_test_jdk_layer_task(self, layer_filename, yt_file_path, jdk_version):
        token_placeholder = '$(vault:value:{}:{})'.format(sdk2.Task.current.owner, self.Parameters.yt_token_vault_name)
        env_vars = "YT_TOKEN='{}' JDK_LAYER_PATH='{}' CLUSTER='{}' BASE_LAYER='{}' JAVA_PATH='{}'".format(
            token_placeholder,
            yt_file_path,
            self.Parameters.cluster,
            self.Parameters.base_layer_path,
            "/opt/jdk{}/bin/java".format(jdk_version),
        )

        test_task_for_jdk = sdk2.Task["YA_EXEC"](
            self,
            description="Test task for jdk layer {}".format(layer_filename),
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia",
            program=self.Parameters.jdk_test_arcadia_location,
            program_args=self.Parameters.jdk_test_program_params,
            env_vars=env_vars,
        ).enqueue()

        return test_task_for_jdk.id

    @staticmethod
    def _wait_tasks(tasks_id):
        return sdk2.WaitTask(
            tasks_id,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
        )

    def _is_test_for_jdk_success(self):
        return all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.jdk_test_tasks_id))

    def _get_jdk_resources(self, jdk_task_id):
        return sdk2.Resource \
            .find(type=self.Parameters.resource_jdk_type, state='READY', task_id=jdk_task_id) \
            .order(-sdk2.Resource.id) \
            .first()

    def _get_ready_jdk_resources(self):
        jdk_resources = []
        jdk_versions_with_existing_resource = []
        for jdk_version, jdk_task_id in self.Context.jdk_build_tasks_info.iteritems():
            jdk_resource = self._get_jdk_resources(jdk_task_id)
            if jdk_resource is None:
                logging.error("there is not resource for jdk version = {}".format(jdk_version))
                continue

            jdk_versions_with_existing_resource.append(jdk_version)
            additional_dirs = ["opt", "jdk{}".format(jdk_version)]
            jdk_resources.append(FileLocationInfo(self._get_resource_path_(jdk_resource), additional_dirs))

        return jdk_resources, jdk_versions_with_existing_resource

    @staticmethod
    def _upload_jdk_layer_in_yt(layer_filename, yt_file_path, yt_operation_executor, jdk_versions):
        logging.info('Create transaction')
        with yt_operation_executor.client.Transaction() as tx:
            logging.info('Transaction: {}'.format(str(tx.transaction_id)))

            logging.info("Write jdk layer in yt. Path: {}".format(yt_file_path))
            yt_operation_executor.write_to_yt_from_local_file(yt_file_path, layer_filename)

            yt_operation_executor.client.set_attribute(
                yt_file_path,
                "jdk_layer_info",
                {"jdk_versions": jdk_versions}
            )

    def _remove_old_releases_of_jdk_layer(self, yt_operations_executor):
        directory = self.Parameters.yt_path
        releases_for_remove = yt_operations_executor.get_ordered_list_of_base_layers(
            directory, self.Parameters.prefix_layer_name)[int(self.Parameters.releases_count) + 1:]
        yt_operations_executor.remove_files_from_directory(directory, releases_for_remove)

    def on_execute(self):
        yt_operation_executor = YtOperationExecutor(
            self.Parameters.cluster,
            sdk2.Vault.data("YT_ROBOT", self.Parameters.yt_token_vault_name))

        if not self.Context.jdk_build_tasks_info:
            jdk_build_tasks = dict()

            for jdk_version in self.Parameters.jdk_versions:
                jdk_build_task_id = self._get_id_of_build_jdk_task(jdk_version)
                jdk_build_tasks[jdk_version] = jdk_build_task_id

            self.Context.jdk_build_tasks_info = jdk_build_tasks

            raise self._wait_tasks(jdk_build_tasks.values())
        else:
            if not self.Context.jdk_test_tasks_id:
                layer_name = self._get_layer_name_with_current_datetime()
                yt_layer_name = "{}/{}".format(self.Parameters.yt_path, layer_name)

                jdk_resources, jdk_versions_with_resource = self._get_ready_jdk_resources()
                TarHelper.make_tarfile(layer_name, jdk_resources)
                self._upload_jdk_layer_in_yt(
                    layer_name,
                    yt_layer_name,
                    yt_operation_executor,
                    jdk_versions_with_resource
                )

                test_tasks_id = []
                for jdk_version in jdk_versions_with_resource:
                    test_tasks_id.append(self._get_id_of_test_jdk_layer_task(layer_name, yt_layer_name, jdk_version))

                self.Context.jdk_test_tasks_id = test_tasks_id
                self.Context.yt_file_path = yt_layer_name
                raise self._wait_tasks(test_tasks_id)
            else:
                if not self._is_test_for_jdk_success():
                    yt_operation_executor.remove_file(self.Context.yt_file_path)
                    raise common.errors.TaskFailure("Built porto layer with jdk is not valid")
                else:
                    link_path = "{}/{}_lastest.{}".format(
                        self.Parameters.yt_path,
                        self.Parameters.prefix_layer_name,
                        self.Parameters.layer_compress)
                    yt_operation_executor.update_link(link_path, self.Context.yt_file_path)
                    self._remove_old_releases_of_jdk_layer(yt_operation_executor)
