import os
import datetime

import sandbox.common.types.task as ctt
import sandbox.projects.resource_types as rt

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.yt.layers_tasks.BuildLayerHelpers import YtOperationExecutor, TarHelper, FileLocationInfo


TEAM = ["sprav"]
ROBOTS_LIST = ["robot-thoth"]


class SpravLayerCreatorResource(rt.AbstractResource):
    releasable = True
    releasers = TEAM + ROBOTS_LIST
    any_arch = False
    auto_backup = True


class SpravLayerCreatorTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        prefix_layer_name = "layer_with_spravdict"
        layer_compress = ".tar.gz"

        cluster = sdk2.parameters.String(
            "Target YT cluster",
            default="hahn",
            required=True,
        )

        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            required=True,
        )

        yt_layer_path = sdk2.parameters.String(
            "Path to layer folder on yt",
             required=True,
        )

        release_on_success = sdk2.parameters.Bool('Release if the task complete success', default=False, required=True)
        with release_on_success.value[True]:
            release_status = sdk2.parameters.Bool('Release STABLE if true else TESTING', default=False, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def _get_layer_filename_with_current_date(self):
        today = datetime.datetime.today().strftime("%Y-%m-%d-%H.%M.%S")
        return "{}-{}.{}".format(self.Parameters.prefix_layer_name, today, self.Parameters.layer_compress)

    def _get_info_about_resources(self):
        return ""

    def on_execute(self):
        yt_operation_executor = YtOperationExecutor(
            self.Parameters.cluster,
            sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault_name),
        )

        with self.memoize_stage.build:
            sub_task = sdk2.Task["YA_MAKE"](
                self,
                description="Build ya_packages",
                test=False,
                build_bundle=False,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                print_ya_make_trace_on_error=True,
                build_system="ya",
                build_type="release",
                check_return_code=True,
                failed_tests_cause_error=True,
                checkout=False,
                checkout_mode="manual",
                clear_build=False,
                force_build_depends=False,
                force_vcs_info_update=False,
                ya_yt_store=True,
                ya_yt_put=False,
                arts="sprav/tools/sprav_dict/table_downloader/table_downloader",
                targets="sprav/tools/sprav_dict/table_downloader",
                result_rd="Arcadia Project",
                result_rt="ARCADIA_PROJECT",
                result_single_file=False,
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, ctt.Status.Group.FINISH, wait_all=True)

        bin_res = sdk2.Resource.find(type=rt.ARCADIA_PROJECT, task_id=self.Context.sub_task_id).first()
        bin_dir_path = str(sdk2.ResourceData(bin_res).path)
        work_dir = os.getcwd()
        bin_path = os.path.join(bin_dir_path, "table_downloader")
        build_cmd = [bin_path]
        environment = os.environ.copy()
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault_name)
        environment["YT_TOKEN"] = yt_token
        run_process(build_cmd, environment=environment, work_dir=work_dir)

        layer_name = self.Parameters.prefix_layer_name + self.Parameters.layer_compress
        yt_file_path = self.Parameters.yt_layer_path + layer_name
        list_of_files = []
        for item in os.listdir(work_dir):
            if item in ["feed.dict",
                        "feature.dict",
                        "feature_enum_value.dict",
                        "provider.dict",
                        "rubric.dict",
                        "unit.dict",
                        "unit_group.dict",
                        "aspect.dict",
                        "black_list.dict",
                        "black_list_record.dict"
                        ]:
                list_of_files += [FileLocationInfo(os.path.join(work_dir, item), ["var", "cache", "sprav"])]
        layer_filename = os.path.join(work_dir, layer_name)
        TarHelper.make_tarfile(layer_filename, list_of_files)
        with yt_operation_executor.client.Transaction() as tx:
            yt_operation_executor.write_to_yt_from_local_file(yt_file_path, layer_filename)
        SpravLayerCreatorResource(self, "spravdict delta layer", layer_filename)

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.release_on_success:
            status = ctt.ReleaseStatus.STABLE if self.Parameters.release_status else ctt.ReleaseStatus.TESTING
            self.on_release(dict(
                releaser=self.author,
                release_status=status,
                release_subject='Preparing file with spravdicts',
                email_notifications=dict(to=['nouret'], cc=[]),
                release_comments='preparing spravdict file',
            ))

    def on_release(self, params):
        super(SpravLayerCreatorTask, self).on_release(params)
        self.mark_released_resources(params["release_status"])
