import logging

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
from sandbox.projects.porto import BuildPortoLayer
from sandbox.common.types.resource import State


class BuildUserLayerTask(sdk2.Task):
    TaskIdAttributeName = "build_layer_task_id"
    ParentLayerType = "PORTO_LAYER_YT"

    class Parameters(sdk2.Task.Parameters):
        layer_compress = 'tar.gz'

        cluster = sdk2.parameters.String(
            'Target YT cluster',
            default="hume",
            required=True,
        )
        yt_token_vault_name = sdk2.parameters.String(
            'YT token vault name',
            required=True,
        )
        yt_target_directory_path = sdk2.parameters.String(
            'YT cypress directory, in which layers will be saved',
            default="//home/arapova/my_own_layers",
            required=True,
        )
        with_clearing_target_directory = sdk2.parameters.Bool(
            'If you want clear directory before load layer. This parameter should be TRUE, if you create SCHEDULER',
            default=False,
            required=True,
        )

        parent_layer_yt_path = sdk2.parameters.String(
            "Parent layer yt path",
            default="//home/arapova/porto_layers/porto_layer_search_ubuntu_xenial_subagent_yt_lastest.tar.gz",
            required=True,
        )

        delta_layer_type = sdk2.parameters.String(
            'Base layer with extra packages type',
            default='PORTO_LAYER_YT',
            required=True,
        )
        delta_layer_name = sdk2.parameters.String(
            'Base layer with extra packages name',
            default='delta_layer',
            required=True,
        )

        additional_delta_layers = sdk2.parameters.List(
            'Path to YT delta layers. For example, geobase or jdk delta layer',
            sdk2.parameters.String,
            default=[]
        )
        delta_script_url = sdk2.parameters.ArcadiaUrl(
            'Setup script for delta layer URL',
            required=True,
        )

        test_script_url = sdk2.parameters.String(
            'Test for delta layer URL.',
        )
        program_args = sdk2.parameters.String(
            'Args to parameters. After this args will be written yt path for layers',
        )
        env_vars = sdk2.parameters.String(
            "Environment variables for test (e.g. VAR1=val1 VAR2='v a l 2')",
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    @staticmethod
    def _get_yt_client(cluster, token):
        from yt.wrapper import YtClient
        return YtClient(cluster, token=token)

    @staticmethod
    def write_to_yt_from_local_file(client, yt_path, local_path):
        logging.debug("Start writing in {} file from {}".format(yt_path, local_path))
        with open(local_path, 'rb') as f:
            client.write_file(yt_path, f)
        logging.debug("End of writing")

    @staticmethod
    def _set_serial_number_to_layer_in_attribute(yt_client, yt_path, serial_number):
        yt_client.set_attribute(yt_path, "serial_layer_number", serial_number)

    @staticmethod
    def get_source_path_for_file(yt_client, yt_path):
        if yt_client.get_type(yt_path) == "link":
            return yt_client.get_attribute(yt_path, "target_path")
        return yt_path

    @staticmethod
    def _wait_task(task_id):
        raise sdk2.WaitTask([task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def create_build_delta_layer_task(self, parent_layer_id):
        layer_build_params = {
            BuildPortoLayer.ParentLayer.name: parent_layer_id,
            BuildPortoLayer.LayerType.name: self.ParentLayerType,
            BuildPortoLayer.LayerName.name: self.Parameters.delta_layer_name,
            BuildPortoLayer.Compress.name: self.Parameters.layer_compress,
            BuildPortoLayer.ScriptUrl.name: self.Parameters.delta_script_url,
            BuildPortoLayer.MergeLayers.name: False,
            BuildPortoLayer.DebugBuild.name: False
        }
        logging.debug("BUILD_PORTO_LAYER build params: {}".format(str(layer_build_params)))

        build_porto_layer_task = sdk2.Task['BUILD_PORTO_LAYER']
        layer_build_task = build_porto_layer_task(
            self,
            description=self.Parameters.description,
            **layer_build_params).enqueue()
        logging.debug("BUILD_PORTO_LAYER build task: {}".format(str(layer_build_task.id)))

        return layer_build_task.id

    def _create_test_task(self, program_args):
        test_task = sdk2.Task["YA_EXEC"](
            self,
            description="Test task for user's layer",
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia",
            program=self.Parameters.test_script_url,
            env_vars=self.Parameters.env_vars,
            program_args=program_args
        ).enqueue()

        return test_task.id

    def _get_parent_layer_resource_id(self, yt_client, yt_parent_layer_path):
        if yt_client.get_type(yt_parent_layer_path) == "link":
            yt_parent_layer_path = yt_parent_layer_path.get_attribute(yt_parent_layer_path, "target_path")

        build_task_id = yt_client.get_attribute(yt_parent_layer_path, self.TaskIdAttributeName)
        parent_layer_resource = sdk2.Resource \
            .find(task_id=build_task_id, type=self.ParentLayerType, state=State.READY) \
            .first()

        if parent_layer_resource is None:
            return None

        return parent_layer_resource.id

    def _get_user_delta_layer_local_path(self):
        built_porto_layer_resource = sdk2.Resource.find(
            task_id=self.Context.build_layer_task_id,
            type=self.Parameters.delta_layer_type,
            state=State.READY).first()
        logging.info("id of built user's porto layer is {}".format(str(built_porto_layer_resource.id)))
        return str(sdk2.ResourceData(built_porto_layer_resource).path)

    def _save_layers_in_directory(self, yt_client):
        layers_for_copy_path = \
            list(self.Parameters.additional_delta_layers) + [self.Parameters.parent_layer_yt_path]
        for i in range(len(layers_for_copy_path)):
            layer = layers_for_copy_path[i]
            if layer == "":
                continue

            source_path = self.get_source_path_for_file(yt_client, layer)
            layer_name = layer.split("/")[-1]
            destination_path = "{}/{}".format(self.Parameters.yt_target_directory_path, layer_name)

            with yt_client.Transaction() as tx:
                logging.info('Transaction: {}'.format(str(tx.transaction_id)))
                yt_client.copy(source_path, destination_path)
                self._set_serial_number_to_layer_in_attribute(yt_client, destination_path, i + 1)

    def _with_testing(self):
        return self.Parameters.test_script_url != ""

    def _is_test_success(self, test_task_id):
        return all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=test_task_id))

    def highlight_target_directory(self, yt_client):
        if self.Parameters.with_clearing_target_directory:
            if yt_client.exists(self.Parameters.yt_target_directory_path):
                yt_client.remove(self.Parameters.yt_target_directory_path)

        if not yt_client.exists(self.Parameters.yt_target_directory_path):
            yt_client.mkdir(self.Parameters.yt_target_directory_path)

        yt_client.set_attribute(self.Parameters.yt_target_directory_path, "directory_with_layers", True)

    def on_execute(self):
        oauth_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault_name)
        yt_client = self._get_yt_client(self.Parameters.cluster, oauth_token)

        self.highlight_target_directory(yt_client)

        if not self.Context.build_layer_task_id:
            parent_layer_id = self._get_parent_layer_resource_id(yt_client, self.Parameters.parent_layer_yt_path)
            if parent_layer_id is None:
                raise common.errors.TaskFailure("Parent layer does not exist")

            self.Context.build_layer_task_id = self.create_build_delta_layer_task(parent_layer_id)
            self._wait_task(self.Context.build_layer_task_id)
        else:
            local_layer_path = self._get_user_delta_layer_local_path()
            yt_layer_path = "{}/{}.{}".format(
                self.Parameters.yt_target_directory_path,
                self.Parameters.delta_layer_name,
                self.Parameters.layer_compress,
            )

            with yt_client.Transaction() as tx:
                logging.info('Transaction: {}'.format(str(tx.transaction_id)))
                self.write_to_yt_from_local_file(yt_client, yt_layer_path, local_layer_path)
                self._set_serial_number_to_layer_in_attribute(yt_client, yt_layer_path, 0)

            if self._with_testing():
                if not self.Context.test_task_id:
                    program_args = "{} {} {} {}".format(
                        self.Parameters.program_args,
                        yt_layer_path,
                        " ".join(self.Parameters.additional_delta_layers),
                        self.Parameters.parent_layer_yt_path,
                    )

                    self.Context.test_task_id = self._create_test_task(program_args)
                    self._wait_task(self.Context.test_task_id)
                else:
                    if not self._is_test_success(self.Context.test_task_id):
                        yt_client.remove(yt_layer_path)
                        raise common.errors.TaskFailure("User's layer is not valid")
                    else:
                        self._save_layers_in_directory(yt_client)
            else:
                self._save_layers_in_directory(yt_client)
