# coding=utf-8
import logging
import os
import urlparse

import shutil

import sandbox.common.types.client as ctc

from sandbox.projects.yt.legacy import resource_types
from sandbox.sandboxsdk import copy, errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task


class ResourceUrls(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'upload_urls'
    description = 'Resource upload urls'
    required = True


class GitCommit(parameters.SandboxStringParameter):

    name = 'git_commit'
    description = 'git commit'
    required = False
    default_value = 'none'


class BuildNumber(parameters.SandboxStringParameter):

    name = 'build_number'
    description = 'Teamcity build number'
    required = False
    default_value = 'none'


class GitBranch(parameters.SandboxStringParameter):

    name = 'git_branch'
    description = 'YT git branch'
    required = False
    default_value = 'none'


class YtChefResources(task.SandboxTask):
    """
    Upload prebuilt yt resources from teamcity
    """

    type = "YT_CHEF_RESOURCES"

    execution_space = 20 * 1024  # MB
    input_parameters = (ResourceUrls, GitCommit, GitBranch, BuildNumber)
    client_tags = ctc.Tag.LINUX_PRECISE

    def _download_item(self, uri, download_folder):
        import api.copier.errors

        try:
            url_scheme = urlparse.urlparse(uri)

            kws = {}
            if url_scheme.scheme == 'rbtorrent':
                kws.update({
                    'fallback_to_bb': True,
                })
            else:
                raise errors.SandboxSvnInvalidParameters('Unsupported url scheme: {}'.format(url_scheme.scheme))

            remote = copy.RemoteCopy(uri, download_folder)
            logging.info("Using %r to fetch remote data.", remote)
            remote(**kws)

        except api.copier.errors.ResourceNotAvailable as ex:
            raise errors.SandboxTaskFailureError(ex)

        copied_files = os.listdir(download_folder)
        if not copied_files:
            raise errors.SandboxTaskFailureError(
                "Error while copying from SkyNet: {}".format(copied_files)
            )
        if len(copied_files) != 1:
            msg = 'Too many files in rbtorrent {}: {}'.format(uri, ','.join(copied_files))
            raise errors.SandboxTaskFailureError(msg)

        result_item = copied_files[0]
        return result_item

    def on_execute(self):
        download_folder = paths.make_folder('download')
        result_folder = paths.make_folder('result')

        urls = self.ctx[ResourceUrls.name]
        for resource_name, uri in urls.iteritems():
            result_item = self._download_item(uri, download_folder)
            resource_type = resource_types.YT_PACKAGE if result_item.endswith("deb") else resource_types.YT_BINARY

            result_item_path = os.path.join(download_folder, result_item)
            if not os.path.isfile(result_item_path):
                raise errors.SandboxTaskFailureError('File type expected for {} resource'.format(resource_name))

            resource_path = os.path.join(result_folder, '{}'.format(result_item))
            shutil.move(result_item_path, resource_path)

            self.create_resource(
                'Imported {} : {}'.format(resource_name, self._format_version()),
                resource_path,
                resource_type,
                arch='linux')

    def _format_version(self):
        return "{}@{}-{}".format(
            self.ctx[GitBranch.name],
            self.ctx[GitCommit.name],
            self.ctx[BuildNumber.name],
        )

    def arcadia_info(self):
        """
        Получение информации о задаче при релизе
        Может быть переопределён в наследниках для уточнения возвращаемых данных

        :return список из трёх значений revision, tag, branch
        """
        return self.ctx[GitCommit.name], self.ctx[BuildNumber.name], self.ctx[GitBranch.name]


__TASK__ = YtChefResources
