# coding=utf-8

import logging
import os
import urlparse
import datetime

import shutil

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.projects.yt.legacy import resource_types
from sandbox.sandboxsdk import copy, errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import environments
import sandbox.projects.common.nanny.nanny as nanny


_RESOURCE_TYPES = {
    'ytserver-core-forwarder': resource_types.YT_CORE_FORWARDER,
    'ytserver-exec': resource_types.YT_EXEC,
    'ytserver-job-proxy': resource_types.YT_JOB_PROXY,
    'ytserver-master': resource_types.YT_MASTER,
    'ytserver-node': resource_types.YT_NODE,
    'ytserver-proxy': resource_types.YT_PROXY,
    'ytserver-http-proxy': resource_types.YT_HTTP_PROXY,
    'ytserver-scheduler': resource_types.YT_SCHEDULER,
    'ytserver-clickhouse': resource_types.YT_CLICKHOUSE,
    'ytserver-tools': resource_types.YT_TOOLS,
    'ytserver-all': resource_types.YT_ALL,
    'ypserver-master': resource_types.YP_MASTER,
    'ytserver-skynet-manager': resource_types.YT_SKYNET_MANAGER,
    'yt_local': resource_types.YT_LOCAL_BIN,
    'yt_binaries': resource_types.YT_BINARIES,
}

_UPLOADABLE_RESOURCES = {
    'yt_binaries',
}


class ResourceUrls(parameters.DictRepeater, parameters.SandboxStringParameter):

    name = 'upload_urls'
    description = 'Resource upload urls. Keys: {}'.format(', '.join(sorted(_UPLOADABLE_RESOURCES)))
    required = True

    @classmethod
    def cast(cls, value):
        dict_value = super(ResourceUrls, cls).cast(value)

        for url in dict_value.itervalues():
            url_scheme = urlparse.urlparse(url)
            if url_scheme.scheme != 'rbtorrent':
                raise ValueError('Invalid url scheme {}'.format(url_scheme.scheme))

        # per-item resource case
        provided_keys = set(dict_value.keys())
        if provided_keys - _UPLOADABLE_RESOURCES:
            raise KeyError('Extra keys: {}'.format(provided_keys - _UPLOADABLE_RESOURCES))
        if _UPLOADABLE_RESOURCES - provided_keys:
            raise KeyError('Missing keys: {}'.format(_UPLOADABLE_RESOURCES - provided_keys))

        return dict_value


class GitCommit(parameters.SandboxStringParameter):

    name = 'git_commit'
    description = 'git commit'
    required = False
    default_value = 'none'


class BuildNumber(parameters.SandboxStringParameter):

    name = 'build_number'
    description = 'Teamcity build number'
    required = False
    default_value = 'none'


class GitBranch(parameters.SandboxStringParameter):

    name = 'git_branch'
    description = 'YT git branch'
    required = False
    default_value = 'none'


class BuildProject(parameters.SandboxStringParameter):

    name = 'build_project'
    description = 'Build project name'
    required = False
    default_value = 'none'


class FullBuildType(parameters.SandboxStringParameter):

    name = 'full_build_type'
    description = 'Full build type'
    required = False
    default_value = None


class YtUploadResources(task.SandboxTask, nanny.ReleaseToNannyTask):
    """
    Upload prebuilt yt resources from teamcity
    """

    type = "YT_UPLOAD_RESOURCES"
    environment = [environments.PipEnvironment("startrek_client", use_wheel=True)]

    execution_space = 50 * 1024  # MB
    input_parameters = (ResourceUrls, GitCommit, GitBranch, BuildNumber, BuildProject)
    client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE

    def _process_yt_binaries(self, result_item, download_folder, result_folder, resource_name):
        result_item_path = os.path.join(download_folder, result_item)
        if not os.path.isdir(result_item_path):
            raise errors.SandboxTaskFailureError('Directory type expected for {} resource'.format(resource_name))

        resource_path = os.path.join(result_folder, '{}'.format(resource_name))
        shutil.move(result_item_path, resource_path)
        paths.chmod(resource_path, 0755)

        self.create_resource(
            'Imported {} : {}'.format(resource_name, self._format_version()),
            resource_path,
            resource_types.YT_BINARIES,
            arch='linux',
            attributes={'ttl': 'inf', 'auto_backup': True},
        )

        for filename in _RESOURCE_TYPES:
            candidate_path = os.path.join(resource_path, filename)
            if not os.path.exists(candidate_path) or filename == 'yt_binaries':
                continue
            new_path = os.path.join(result_folder, filename)
            shutil.copy(candidate_path, new_path)
            paths.chmod(new_path, 0755)

            self.create_resource(
                'Imported {} : {}'.format(filename, self._format_version()),
                new_path,
                _RESOURCE_TYPES[filename],
                arch='linux',
                attributes={'ttl': 'inf', 'auto_backup': True},
            )

    def _process_single_binary(self, result_item, download_folder, result_folder, resource_name):
        result_item_path = os.path.join(download_folder, result_item)
        if not os.path.isfile(result_item_path):
            raise errors.SandboxTaskFailureError('File type expected for {} resource'.format(resource_name))

        resource_path = os.path.join(result_folder, '{}'.format(resource_name))
        shutil.move(result_item_path, resource_path)
        paths.chmod(resource_path, 0755)

        self.create_resource(
            'Imported {} : {}'.format(resource_name, self._format_version()),
            resource_path,
            _RESOURCE_TYPES[resource_name],
            arch='linux',
            attributes={'ttl': 'inf', 'auto_backup': True},
        )

    def _download_item(self, uri, download_folder):
        import api.copier.errors

        try:
            url_scheme = urlparse.urlparse(uri)

            kws = {}
            if url_scheme.scheme == 'rbtorrent':
                kws.update({
                    'fallback_to_bb': True,
                })
            else:
                raise errors.SandboxSvnInvalidParameters('Unsupported url scheme: {}'.format(url_scheme.scheme))

            remote = copy.RemoteCopy(uri, download_folder)
            logging.info("Using %r to fetch remote data.", remote)
            remote(**kws)

        except api.copier.errors.ResourceNotAvailable as ex:
            raise errors.SandboxTaskFailureError(ex)

        copied_files = os.listdir(download_folder)
        if not copied_files:
            raise errors.SandboxTaskFailureError(
                "Error while copying from SkyNet: {}".format(copied_files)
            )
        if len(copied_files) != 1:
            msg = 'Too many files in rbtorrent {}: {}'.format(uri, ','.join(copied_files))
            raise errors.SandboxTaskFailureError(msg)

        result_item = copied_files[0]
        return result_item

    def on_execute(self):
        download_folder = paths.make_folder('download')
        result_folder = paths.make_folder('result')

        post_process_dispatcher = {
            resource_types.YT_BINARIES: self._process_yt_binaries,
            # resource_types.YT_CORE_FORWARDER: self._process_single_binary,
            # resource_types.YT_EXEC: self._process_single_binary,
            # resource_types.YT_JOB_PROXY: self._process_single_binary,
            # resource_types.YT_MASTER: self._process_single_binary,
            # resource_types.YT_NODE: self._process_single_binary,
            # resource_types.YT_PROXY: self._process_single_binary,
            # resource_types.YT_SCHEDULER: self._process_single_binary,
            # resource_types.YT_TOOLS: self._process_single_binary,
        }

        urls = self.ctx[ResourceUrls.name]
        for resource_name, uri in urls.iteritems():
            result_item = self._download_item(uri, download_folder)
            post_process_dispatcher[_RESOURCE_TYPES[resource_name]](
                result_item, download_folder, result_folder, resource_name
            )

    def _format_version(self):
        return "{}@{}-{}".format(
            self.ctx[GitBranch.name],
            self.ctx[GitCommit.name],
            self.ctx[BuildNumber.name],
        )

    @property
    def release_template(self):
        subject = "Yt prebuilt version {}".format(self._format_version())
        # More details in YPADMIN-134.
        if self.ctx.get(BuildProject.name) == "yp":
            subject = "[yp] " + subject

        result = self.ReleaseTemplate(
            subject=subject,
            cc=['alximik@yandex-team.ru'],
            message=self.descr,
        )
        return result

    def on_release(self, additional_parameters):
        if additional_parameters["release_status"].lower() == "prestable" and self.ctx.get(BuildProject.name) == "yp":
            for env in self.environment:
                env.prepare()

            from startrek_client import Startrek
            now = datetime.datetime.now()

            revision, tag, branch = self.arcadia_info()
            full_build_type = self.ctx.get(FullBuildType.name)
            if full_build_type is None:
                full_build_type = "Yt_PreciseYpOnlyYaRelease"

            startrek_token = sdk2.Vault.data("YT_ROBOT", "startrek_token")
            st_client = Startrek(useragent=self.__class__.__name__, token=startrek_token)
            issue = st_client.issues.create(
                queue="YP",
                summary="Release {0} {1} {2} {3}".format(now.date().isoformat(), tag, branch, revision),
                description="SOX: {0}\n"
                            "https://teamcity.yandex-team.ru/viewLog.html?buildTypeId={1}&buildNumber={2}\n"
                            "https://sandbox.yandex-team.ru/task/{3}\n"
                            .format(revision, full_build_type, tag, self.id),
                tags=["hermes-sox", "release"],
            )
            self.set_info("Release ticket created " + issue.key)
            self.ctx[nanny.STARTREK_TICKET_IDS_KEY] = [issue.key]

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    def arcadia_info(self):
        """
        Retrieves information about task on release.
        Can be redefined in inherited classes.

        :return revision, tag, branch
        """
        return self.ctx[GitCommit.name], self.ctx[BuildNumber.name], self.ctx[GitBranch.name]


__TASK__ = YtUploadResources
