# coding=utf-8
import os
import subprocess

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

PYTHON3_EXECUTABLE = "/usr/bin/python3.6"


class ZelibobaFormattingCheck(sdk2.Task):
    """Run formatting check on the arcadia project"""

    class Requirements(sdk2.Task.Requirements):
        """
        Используется кастомный LXC контейнер с установленным python3.6.

        Инструкция: https://wiki.yandex-team.ru/sandbox/cookbook/#use-custom-lxc-image
        Ещё одна инструкция: https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/sandbox_lxc_image/__init__.py
        Таска: https://sandbox.yandex-team.ru/task/801557372/view


        container_resource  - ID ресурса с контейнером. Если задолбаешься обновлять это число,
        можно релизить контейнер с кастомным типом ресурса, тогда самая новая версия будет подтягиваться
        автоматически.
        """

        container_resource = 2361803169  # https://sandbox.yandex-team.ru/task/1050137419/view

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60
        owner = "ZELIBOBA"
        description = "Run formatting check on the arcadia project"
        project_import_path = sdk2.parameters.String(
            "Arcadia project import path e.g. mail.payments", required=True
        )
        project_path = sdk2.parameters.String(
            "Arcadia project path e.g. /mail/payments", required=True
        )
        commit_hash = sdk2.parameters.String(
            "Arc commit hash e.g. #9fd3c297477cce3985f0e22694966de27a7b451d without #",
            required=True,
        )

        requirements = sdk2.parameters.List(
            "List of required packages with hashes e.g. https://paste.yandex-team.ru/3047746",
            required=True,
        )

        check_isort = sdk2.parameters.Bool("Run isort check")
        isort_options = sdk2.parameters.List("List of isort cmdline options")
        isort_check_path = sdk2.parameters.String(
            "Isort check path, defaults to project_path"
        )

        check_black = sdk2.parameters.Bool("Run black check")
        black_options = sdk2.parameters.List("List of black cmdline options")
        black_check_path = sdk2.parameters.String(
            "Black check path, defaults to project_path"
        )

    @sdk2.header(title="Report", label="report")
    def show_report(self):
        COLOR_GOOD = "#47C971"
        COLOR_BAD = "#FD0D1B"

        def badge(text, color):
            return (
                '<span style="background-color: %s; color: #ffffff; font-weight: bold; padding: 0px 8px">%s</span>'
                % (color, text)
            )

        isort_report = ""
        if self.Parameters.check_isort:
            if self.Context.isort_output:
                isort_report = (
                    badge("Isort FAILED", COLOR_BAD)
                    + "\nErrors:\n"
                    + self.Context.isort_output
                )
            else:
                isort_report = badge("Isort OK", COLOR_GOOD)
        black_report = ""
        if self.Parameters.check_black:
            if self.Context.black_output:
                black_report = (
                    badge("Black FAILED", COLOR_BAD)
                    + "\nErrors:\n"
                    + self.Context.black_output
                )
            else:
                isort_report = badge("Isort OK", COLOR_GOOD)

        return "\n\n".join([isort_report, black_report]).strip().replace("\n", "<br>")

    def make_relative(self, path):
        return path.lstrip("/")

    def get_isort_check_path(self):
        return self.make_relative(
            self.Parameters.isort_check_path or self.Parameters.project_path
        )

    def get_black_check_path(self):
        return self.make_relative(
            self.Parameters.black_check_path or self.Parameters.project_path
        )

    def install_requirements(self, workdir):
        """
        Устанавливаем все зависимости.
        В requirements могут быть указаны hashes, для секурити.
        """
        requirements_path = os.path.join(workdir, "requirements.txt")
        with open(requirements_path, "w") as requirements_file:
            requirements = "\n".join([param for param in self.Parameters.requirements])
            requirements_file.write(requirements)

        with sdk2.helpers.ProcessLog(self, logger="pip-log") as pl:
            try:
                subprocess.check_call(
                    [
                        PYTHON3_EXECUTABLE,
                        "-m",
                        "pip",
                        "install",
                        "--index-url",
                        "https://pypi.yandex-team.ru/simple",
                        "-r",
                        requirements_path,
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(
                        workdir, self.make_relative(self.Parameters.project_path)
                    ),
                )
            finally:
                with open(str(pl.stdout.path), "r") as logs:
                    self.Context.pip_stdout = logs.read()

    def run_isort_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="isort-log") as pl:
            try:
                subprocess.check_call(
                    [
                        PYTHON3_EXECUTABLE,
                        "-m",
                        "isort",
                        "--check",
                        os.path.join(arcadia_src_dir, self.get_isort_check_path()),
                    ]
                    + self.Parameters.isort_options,
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(
                        arcadia_src_dir,
                        self.make_relative(self.Parameters.project_path),
                    ),
                )
            finally:
                with open(str(pl.stdout.path), "r") as logs:
                    self.Context.isort_output = logs.read()

    def run_black_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="black-log") as pl:
            try:
                subprocess.check_call(
                    [
                        PYTHON3_EXECUTABLE,
                        "-m",
                        "black",
                        "--check",
                        os.path.join(arcadia_src_dir, self.get_black_check_path()),
                    ]
                    + self.Parameters.black_options,
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(
                        arcadia_src_dir,
                        self.make_relative(self.Parameters.project_path),
                    ),
                )
            finally:
                with open(str(pl.stdout.path), "r") as logs:
                    self.Context.black_output = logs.read()

    def on_execute(self):
        with arcadia_sdk.mount_arc_path(
            "arcadia-arc://#" + self.Parameters.commit_hash
        ) as arcadia_src_dir:
            self.install_requirements(arcadia_src_dir)

            if self.Parameters.check_isort:
                self.run_isort_cmd(arcadia_src_dir)
            if self.Parameters.check_black:
                self.run_black_cmd(arcadia_src_dir)
