import os

import shutil
import tarfile

import pathlib2

# import sandbox
from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityArtifactsContext
from sandbox.sdk2.vcs.git import Git as SandboxGit
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.environments import SandboxGitLfsEnvironment

from sandbox.sdk2.helpers import ProcessLog, subprocess

STRING_PARAMETERS_SEPARATOR = ' '
STRING_PARAMETERS_DESCRIPTION = 'Separated by "{}"'.format(STRING_PARAMETERS_SEPARATOR)


def github_vcs_root(**kwargs):
    return SandboxGit(
        url='https://github.yandex-team.ru/zen-mobile/zenkit.git',
        push_url='https://github.yandex-team.ru/zen-mobile/zenkit.git',
        **kwargs
    )


def split_param(param):
    return [v for v in param.split(STRING_PARAMETERS_SEPARATOR) if v]


class ZenRunEspressoTests(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Scm') as scm:
            branch = sdk2.parameters.String('Branch')
            commit = sdk2.parameters.String('Commit', required=True)
        build_sdk = sdk2.parameters.Resource('Android sdk', required=True, default=1091933878)
        gradle_targets = sdk2.parameters.String(
            'Gradle targets', description=STRING_PARAMETERS_DESCRIPTION, required=True)
        gradle_arguments = sdk2.parameters.String(
            'Gradle arguments', description=STRING_PARAMETERS_DESCRIPTION, default='')
        # emulator_sdk = sdk2.parameters.Resource('Emulator sdk', required=True, default=971137357)

        with sdk2.parameters.Group('Emulator sdk') as emulator_sdk:
            emulator_skd_url = sdk2.parameters.Url(
                'Emulator url', required=True,
                default='https://s3.mds.yandex.net/zenkit-test/androidnew.tar.gz'
            )
            emulator_skd_hash = sdk2.parameters.Url(
                'Emulator hash', default='a9bb0263f772e3a017338b512f6981335d6e8a76', required=True)

        emulator_launching_config = sdk2.parameters.Resource('Emulator launching config',
                                                             required=True, default=1075246626)
        artifacts_to_publish = sdk2.parameters.String(
            'Artifacts to publish',
            description='Files and directories, which will be added to TeamcityArtifacts resource. {}'.format(
                STRING_PARAMETERS_DESCRIPTION))
        debug = sdk2.parameters.Bool('Debug', default=False)

        _container = sdk2.parameters.Container("Container", default=890998880, required=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = ctc.Tag.BROWSER & ctc.Tag.LINUX_TRUSTY
        dns = ctm.DnsType.DNS64
        environments = [
            SandboxGitLfsEnvironment('2.5.2'),
            SandboxJavaJdkEnvironment('1.8.0'),
            PipEnvironment('get-deps==1.2.3'),
            PipEnvironment('yandex-avd-tools==1.3.2')
        ]

    def artifacts_path(self, *args):
        return str(self.path('teamcity_artifacts', *args))

    def checkout_path(self, *args):
        return str(self.path('checkout-dir', *args))

    def build_sdk_path(self, *args):
        return str(self.path('build-sdk', *args))

    def emulator_sdk_path(self, *args):
        return str(self.path('emulator-sdk', *args))

    # @property
    # @sandbox.common.utils.singleton
    # def emulator_sdk(self):
    #     return str(sdk2.ResourceData(self.Parameters.build_sdk).path)

    def checkout_code(self):
        vcs_root = github_vcs_root(filter_branches=False)
        vcs_root.clone(self.checkout_path(), self.Parameters.branch, commit=self.Parameters.commit)
        vcs_root.execute('submodule', 'update', '--init', '--recursive', cwd=self.checkout_path())

    def get_build_sdk(self):
        archive_path = str(sdk2.ResourceData(self.Parameters.build_sdk).path)
        with tarfile.open(archive_path, 'r:gz') as archive:
            archive.extractall(path=self.build_sdk_path())

        with ProcessLog(self, logger='accept_licenses') as log:
            yes = subprocess.Popen(['yes'], stdout=subprocess.PIPE)
            subprocess.check_call([self.build_sdk_path('sdk/tools/bin/sdkmanager'), '--licenses'],
                                  stdout=log.stdout, stderr=log.stdout, stdin=yes.stdout, )
            yes.kill()

    def get_emulators_sdk(self):
        from get_deps import get_deps
        os.mkdir(self.emulator_sdk_path())  # TODO
        getdeps = get_deps.GetDeps()
        getdeps.deploy(
            url=self.Parameters.emulator_skd_url,
            deploy_rules=[('copy', '*', self.emulator_sdk_path() + '/')],
            sha1=self.Parameters.emulator_skd_hash,
        )

    def gradle(self):
        gradle_args = (split_param(self.Parameters.gradle_arguments)
                       if self.Parameters.gradle_arguments else [])
        with TeamcityArtifactsContext(pathlib2.Path(self.checkout_path())) as artifacts_context:
            env = os.environ.copy()
            env['ANDROID_HOME'] = self.build_sdk_path('sdk')
            env['EMULATOR_SDK_PATH'] = self.emulator_sdk_path()
            return subprocess.call(
                ['./gradlew'] + gradle_args + split_param(self.Parameters.gradle_targets),
                cwd=self.checkout_path(), env=env,
                stdout=artifacts_context.output, stderr=artifacts_context.output)

    def publish_artifacts(self):
        publish = False
        os.mkdir(self.artifacts_path())
        for filename in split_param(self.Parameters.artifacts_to_publish):
            abs_path = os.path.abspath(self.checkout_path(filename))
            if not os.path.exists(abs_path):
                continue
            if os.path.isdir(abs_path):
                shutil.copytree(abs_path, self.artifacts_path(os.path.basename(abs_path)))
            else:
                shutil.copy(abs_path, self.artifacts_path())
        for root, dirs, files in os.walk(self.artifacts_path()):
            if files:
                publish = True
                break
        if publish:
            artifacts_resource = TeamcityArtifacts(
                self, 'Teamcity artifacts', self.artifacts_path())
            sdk2.ResourceData(artifacts_resource).ready()

    def on_execute(self):
        import avd_tools

        self.checkout_code()
        self.get_build_sdk()
        self.get_emulators_sdk()
        avd_tools.utils.kill_stale_processes()
        avd_tools.utils.remove_android_home_dir()
        avd_tools.paths.set_sdk_path(self.emulator_sdk_path())
        try:
            with avd_tools.RunManyAvdWithRetry(
                    str(sdk2.ResourceData(self.Parameters.emulator_launching_config).path), tries=1):
                if self.Parameters.debug:
                    self.suspend()
                exit_code = self.gradle()
                if exit_code:
                    raise TaskFailure('Gradle exited with {}'.format(exit_code))
        finally:
            self.publish_artifacts()
