from os.path import join as pj
from sandbox import sdk2
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.zora import resources
from sandbox.projects.zora.GenResourceFromSvn import GenResourceFromSvn
import logging
import os
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
import sandbox.projects.common.constants as consts
import shutil

logger = logging.getLogger(__name__)

TARGETS = {
    resources.GoZoraBinary.name: {
        'build_path': resources.GoZoraBinary.arcadia_build_path,
        'type': 'build',
        'zora_type': 'gozora'
    },
    resources.GoZoraConfigs.name: {
        'build_path': resources.GoZoraConfigs.arcadia_build_path,
        'type': 'svn',
        'zora_type': 'gozora'
    },
    resources.GoZoraScripts.name: {
        'build_path': resources.GoZoraScripts.arcadia_build_path,
        'type': 'svn',
        'zora_type': 'gozora'
    },
}


class BuildGoZoraBase(ReleaseToNannyTask2, sdk2.Task):

    class Context(sdk2.Task.Context):
        child_tasks = dict()

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        with sdk2.parameters.CheckGroup("Resources to build") as build_targets:
            for r in sorted(TARGETS):
                build_targets.values[r] = r

        with sdk2.parameters.RadioGroup('Build type') as build_type:
            build_type.values.release = build_type.Value('release', default=True)
            build_type.values.debug = 'debug'
            build_type.values.profile = 'profile'

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

        with sdk2.parameters.RadioGroup("Sanitize") as sanitize_group:
            sanitize_group.values.none = sanitize_group.Value(default=True)
            sanitize_group.values.address = None
            sanitize_group.values.leak = None
            sanitize_group.values.memory = None
            sanitize_group.values.thread = None
            sanitize_group.values.undefined = None

        strip_binaries = sdk2.parameters.Bool('Strip result binaries', default=True)

    def on_execute(self):
        with self.memoize_stage.build:
            self.Context.build_targets = self.Parameters.build_targets
            for res in self.Parameters.build_targets:
                if TARGETS[res]['type'] == 'build':
                    task = YaMake2(
                        self,
                        description='Child of {}. Build {}'.format(self.id, res),
                        checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                        use_aapi_fuse=True,
                        use_arc_instead_of_aapi=False,
                        aapi_fallback=True,
                        build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                        build_type=self.Parameters.build_type,
                        arcadia_patch=self.Parameters.arcadia_patch,
                        check_return_code=True,
                        strip_binaries=self.Parameters.strip_binaries,
                        sanitize=None if self.Parameters.sanitize_group == 'none' else self.Parameters.sanitize_group,
                        ya_yt_store=True,
                        ya_yt_token_vault_name='robot-zora-yt-token',
                        ya_yt_token_vault_owner='ZORA',
                        targets=TARGETS[res]['build_path'],
                        arts=pj(TARGETS[res]['build_path'], sdk2.Resource[res].bin_name),
                        result_single_file=True,
                        result_rd='{}, Sandbox task: {}'.format(
                            res,
                            'https://sandbox.yandex-team.ru/task/{}'.format(self.id)
                        ),
                        result_rt=res,
                    )
                elif TARGETS[res]['type'] == 'svn':
                    task = GenResourceFromSvn(
                        self,
                        description='Child of {}. Generate {} resource from svn.'.format(self.id, res),
                        arcadia_url=self.Parameters.checkout_arcadia_from_url,
                        svn_paths=TARGETS[res]['build_path'],
                        resource_type=res
                    )
                elif TARGETS[res]['type'] == 'package':
                    task = sdk2.Task["YA_PACKAGE"](
                        self,
                        description='Child of {}. Build {}'.format(self.id, res),
                        owner=self.owner,
                        checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                        arcadia_patch=self.Parameters.arcadia_patch,
                        package_type='tarball',
                        packages=TARGETS[res]['build_path'],
                        resource_type=res,
                        use_aapi_fuse=True,
                        aapi_fallback=True,
                        raw_package=True,
                        build_type=self.Parameters.build_type,
                        build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                        checkout_mode='auto',
                        sanitize=None if self.Parameters.sanitize_group == 'none' else self.Parameters.sanitize_group,
                        ya_yt_store=True,
                        ya_yt_token_vault_name='robot-zora-yt-token',
                        ya_yt_token_vault_owner='ZORA'
                    )
                task.save().enqueue()
                self.Context.child_tasks[res] = task.id
                self.set_info('{} build task {} started'.format(res, task.id))

            raise sdk2.WaitTask(
                self.Context.child_tasks.values(),
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        with self.memoize_stage.finalize:
            for task_key, task_id in self.Context.child_tasks.items():
                task = sdk2.Task[task_id]
                if task.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))

            for res_name, build_task_id in self.Context.child_tasks.items():
                resource = sdk2.Resource.find(
                    type=res_name,
                    task_id=build_task_id
                ).first()
                if resource:
                    resource_data = sdk2.ResourceData(resource)
                else:
                    raise ce.TaskFailure('Resource {} with task_id {} not found'.format(res_name, build_task_id))
                resource_new = sdk2.Resource[res_name](
                    self,
                    resource.description,
                    resource_data.path.name,
                    ttl=30
                )
                if not os.path.isdir(str(resource_data.path)):
                    shutil.copy(str(resource_data.path), resource_data.path.name)
                else:
                    shutil.copytree(str(resource_data.path), resource_data.path.name)
                sdk2.ResourceData(resource_new).ready()

            self.set_info('done')

    def on_release(self, additional_parameters):
        ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)


class BuildGoZora(BuildGoZoraBase):
    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        with sdk2.parameters.CheckGroup("Resources to build") as build_targets:
            for r in sorted(TARGETS):
                build_targets.values[r] = r

        with sdk2.parameters.RadioGroup('Build type') as build_type:
            build_type.values.release = build_type.Value('release', default=True)
            build_type.values.debug = 'debug'
            build_type.values.profile = 'profile'

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

        with sdk2.parameters.RadioGroup("Sanitize") as sanitize_group:
            sanitize_group.values.none = sanitize_group.Value(default=True)
            sanitize_group.values.address = None
            sanitize_group.values.leak = None
            sanitize_group.values.memory = None
            sanitize_group.values.thread = None
            sanitize_group.values.undefined = None

        strip_binaries = sdk2.parameters.Bool('Strip result binaries', default=True)
