from sandbox import sdk2
from sandbox.common.types import task
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.common.constants as consts

import logging
import os
import shutil


_ZORA_PKGS = {
    'ZORA_PACKAGE': 'robot/zora/packages/zora_package.json',
    'ZORA_CONFIGS_PACKAGE': 'robot/zora/packages/zora_configs_package.json',
    'ZORA_DYNAMIC_CONFIGS_PACKAGE': 'robot/zora/packages/zora_dynamic_configs_package.json',
    'ZORA_ONLINE_PACKAGE': 'robot/zora/packages/zora_package.json',
    'ZORA_ONLINE_CONFIGS_PACKAGE': 'robot/zora/packages/zora_online_configs_package.json',
    'ZORA_ONLINE_DYNAMIC_CONFIGS_PACKAGE': 'robot/zora/packages/zora_online_dynamic_configs_package.json',
    'ZORA_ONLINE_POSTPROCESSING_PACKAGE': 'robot/zora/packages/zora_postprocessing_package.json',
    'PDFETCH_ONLINE_PACKAGE': 'robot/zora/packages/pdfetch_online_package.json',
    'PDFETCH_PACKAGE': 'robot/zora/packages/pdfetch_package.json',
    'ZORA_POSTPROCESSING_PACKAGE': 'robot/zora/packages/zora_postprocessing_package.json',
    'ROTOR_PACKAGE': 'robot/zora/packages/rotor_package.json',
    'ROTOR_EXECUTOR_PACKAGE': 'robot/zora/packages/rotor_executor_package.json',
    'ZORA_CM_PACKAGE': 'robot/zora/packages/zora_cm.json'
}


class BuildZoraPackages(sdk2.Task, nanny.ReleaseToNannyTask2):

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup('Zora packages to build', required=True) as zora_pkgs:
            for p in sorted(_ZORA_PKGS):
                zora_pkgs.values[p] = p

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Svn url for arcadia (you can add '@<commit_number>')",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )

        with sdk2.parameters.RadioGroup('Build type') as build_type:
            build_type.values.release = build_type.Value('release', default=True)
            build_type.values.debug = 'debug'
            build_type.values.profile = 'profile'

        with sdk2.parameters.String('Release task to specified configuration') as release_type:
            release_type.values.none = release_type.Value("none", default=True)
            release_type.values.stable = "stable"
            release_type.values.testing = "testing"
            release_type.values.prestable = "prestable"
            release_type.values.unstable = "unstable"

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

        with sdk2.parameters.RadioGroup("Sanitize") as sanitize_group:
            sanitize_group.values.none = sanitize_group.Value(default=True)
            sanitize_group.values.address = None
            sanitize_group.values.leak = None
            sanitize_group.values.memory = None
            sanitize_group.values.thread = None
            sanitize_group.values.undefined = None

    class Context(sdk2.Context):
        errors = []
        messages = []
        subtasks = {}

    def launch_subtask(self, pkg_path, resource_type):
        subtask = sdk2.Task["YA_PACKAGE"](
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            arcadia_patch=self.Parameters.arcadia_patch,
            package_type='tarball',
            packages=pkg_path,
            resource_type=resource_type,
            use_aapi_fuse=True,
            aapi_fallback=True,
            raw_package=True,
            build_type=self.Parameters.build_type,
            build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
            checkout_mode='auto',
            sanitize=None if self.Parameters.sanitize_group == 'none' else self.Parameters.sanitize_group,
            ya_yt_token_vault_name='robot-zora-yt-token',
            ya_yt_token_vault_owner='ZORA'
        )
        subtask.enqueue()

        return subtask.id

    def check_subtask_status(self, subtask):
        if subtask.status in (task.Status.FAILURE, task.Status.EXCEPTION):
            self.Context.errors.append('Subtask {} failed'.format(subtask.id))
        else:
            self.Context.messages.append('Subtask {} have finished successfully'.format(subtask.id))

    def on_execute(self):
        pkgs_to_build = list(self.Parameters.zora_pkgs)
        logging.info("Build params:\npkgs_to_build: {}".format(pkgs_to_build))

        with self.memoize_stage.run_subtasks:
            tasks_to_wait = []
            for pkg in pkgs_to_build:
                subtask = self.launch_subtask(_ZORA_PKGS[pkg], pkg)
                self.Context.subtasks[pkg] = subtask
                tasks_to_wait.append(subtask)

            raise sdk2.WaitTask(tasks_to_wait, task.Status.Group.FINISH | task.Status.Group.BREAK, wait_all=True)

        subtasks = list(self.find())
        for subtask in subtasks:
            self.check_subtask_status(subtask)

        if self.Context.errors:
            raise SandboxTaskFailureError('One of child tasks failed unexpectedly')

        out_resources = {}
        for pkg in pkgs_to_build:
            res = sdk2.Resource.find(
                type=pkg,
                task_id=self.Context.subtasks[pkg]
            ).first()
            out_resources[pkg] = res.id
            tmp_res = sdk2.ResourceData(res)
            shutil.copytree(str(tmp_res.path), os.path.basename(str(tmp_res.path)))
            out_res = sdk2.Resource[pkg](self, res.description, os.path.basename(str(tmp_res.path)), ttl=30)
            out_data = sdk2.ResourceData(out_res)
            out_data.ready()

        logging.info("Out resources: {}".format(out_resources))

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
