import os
import logging
import tarfile

from os.path import join as pj

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia


class GenResourceFromSvn(sdk2.Task):
    """
    Generate resource with files/folder from svn.
    """

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Svn url for arcadia (you can add '@<commit_number>')",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )

        svn_paths = sdk2.parameters.String(
            "Path in arcadia to files/folders in svn to include in resource (';'-separated).",
            required=True,
            multiline=True
        )

        resource_type = sdk2.parameters.String(
            "Type of output resource",
            required=True
        )

    def on_execute(self):
        res_out_dir = pj(os.getcwd(), self.Parameters.resource_type)
        os.mkdir(res_out_dir)

        vcs_prefix = self.Parameters.arcadia_url.split('@')[0]
        if len(self.Parameters.arcadia_url.split('@')) > 1:
            vcs_revision = self.Parameters.arcadia_url.split('@')[1]
        else:
            vcs_revision = 'HEAD'
            logging.warning("SVN Revision not specified!")

        for entry in self.Parameters.svn_paths.split(';'):
            path = pj(vcs_prefix, entry)
            Arcadia.export(path, res_out_dir, vcs_revision)

        logging.info("Generating resource {}...".format(self.Parameters.resource_type))

        out_tar_path = pj(os.getcwd(), '{}.tar'.format(self.Parameters.resource_type))

        with tarfile.open(out_tar_path, 'w') as out_tar:
            os.chdir(res_out_dir)
            for e in os.listdir(res_out_dir):
                out_tar.add(e)
            os.chdir(os.path.dirname(out_tar_path))

        res_description = "{} resource from {}. Revision: {}".format(
            self.Parameters.resource_type,
            vcs_prefix,
            vcs_revision
        )

        out_res = sdk2.Resource[self.Parameters.resource_type](
            self,
            res_description,
            out_tar_path,
            ttl=30
        )
        out_data = sdk2.ResourceData(out_res)
        out_data.ready()
