import os
import logging
import distutils
import tarfile

from os.path import join as pj

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arc_sdk


class GenResourceFromVcs(sdk2.Task):
    """
    Generate resource with files/folder from vcs.
    """

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "VCS url for arcadia",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )

        vcs_paths = sdk2.parameters.String(
            "Path in arcadia to files/folders in VCS to include in resource (';'-separated).",
            required=True,
            multiline=True
        )

        resource_type = sdk2.parameters.String(
            "Type of output resource",
            required=True
        )

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

    def on_execute(self):
        is_arc_repo = arc_sdk.is_arc_path((self.Parameters.arcadia_url))

        with arc_sdk.mount_arc_path(self.Parameters.arcadia_url) as vcs_path:
            if is_arc_repo:
                vcs_info = arc_sdk.mounted_path_svnversion(vcs_path, True)
                repo = str(vcs_info['repository'])
                revision = str(arc_sdk.mounted_path_svnversion(vcs_path, True)['revision'])
            else:
                vcs_info = arc_sdk.mounted_path_svnversion(vcs_path, False)
                repo = str(vcs_info['repository'])
                revision = str(vcs_info['revision'])

            if self.Parameters.arcadia_patch:
                arc_sdk.apply_patch(self, vcs_path, self.Parameters.arcadia_patch, str(self.path()))

            res_out_dir = pj(os.getcwd(), self.Parameters.resource_type)
            os.mkdir(res_out_dir)

            for entry in self.Parameters.vcs_paths.split(';'):
                path_to_cp = pj(vcs_path, entry)
                distutils.dir_util.copy_tree(path_to_cp, res_out_dir)

        logging.info("Generating resource {}...".format(self.Parameters.resource_type))

        out_tar_path = pj(os.getcwd(), '{}.tar'.format(self.Parameters.resource_type))

        with tarfile.open(out_tar_path, 'w') as out_tar:
            os.chdir(res_out_dir)
            for e in os.listdir(res_out_dir):
                out_tar.add(e)
            os.chdir(os.path.dirname(out_tar_path))

        res_description = "{} resource from {}. Revision: {}".format(
            self.Parameters.resource_type,
            repo,
            revision
        )

        out_res = sdk2.Resource[self.Parameters.resource_type](
            self,
            res_description,
            out_tar_path,
            ttl=30
        )
        out_data = sdk2.ResourceData(out_res)
        out_data.ready()
