from sandbox import sdk2
from sandbox.common.types import task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.common.constants as consts

import os
import logging
import subprocess
import shutil
from os.path import join as pj


_CONFIGS = {
    'ZORA_CLUSTER_CONFIG':              'arcadia:/arc/trunk/arcadia/robot/zora/conf/static/cloud/production',
    'ZORA_TEST_CLUSTER_CONFIG':         'arcadia:/arc/trunk/arcadia/robot/zora/conf/static/cloud/test',
    'ZORA_ONLINE_CLUSTER_CONFIG':       'arcadia:/arc/trunk/arcadia/robot/zora/conf/static/online/production',
    'ZORA_ONLINE_DEV_CLUSTER_CONFIG':   'arcadia:/arc/trunk/arcadia/robot/zora/conf/static/online/dev',
}

_GEN_BIN_PATH = 'robot/zora/tools/cluster_conf_gen'
_ONLINE_DC_LIST = ['sas', 'vla', 'man']
_DC_LIST = ['sas', 'man', 'vla', 'iva']


class GenZoraClusterConf(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup('Configs to generate', required=True) as confs:
            for p in sorted(_CONFIGS):
                confs.values[p] = p

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Svn url for arcadia (you can add '@<commit_number>')",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )

        gen_bin_resource = sdk2.parameters.Resource(
            "Resource id of cluster.conf generator (if empty - will be built)",
            resource_type=sdk2.Resource['CLUSTER_CONF_GENERATOR_BIN']
        )

        yp_token_vault_name = sdk2.parameters.String(
            "YP token vault name",
            required=True
        )

        yp_token_owner = sdk2.parameters.String(
            "YP token vault owner",
            required=True
        )

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

    class Context(sdk2.Context):
        errors = []
        messages = []
        subtasks = {}

    def merge_online_confs(self, dc_list, work_dir):
        conf_data = ''
        merged_conf_path = pj(work_dir, 'cluster.conf')
        for dc in dc_list:
            with open(pj(work_dir, '{}_cluster.conf'.format(dc)), 'r') as conf:
                conf_data += conf.read()
        with open(merged_conf_path, 'w+') as merged_conf:
            merged_conf.write(conf_data)

    def build_generator(self, gen_bin_path):
        subtask = sdk2.Task["YA_MAKE_2"](
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            arcadia_patch=self.Parameters.arcadia_patch,
            targets=gen_bin_path,
            arts=pj(gen_bin_path, 'cluster_conf_gen'),
            result_single_file=True,
            result_rt='CLUSTER_CONF_GENERATOR_BIN',
            result_rd='Cluster.conf generator binary',
            use_aapi_fuse=True,
            aapi_fallback=True,
            build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
            checkout_mode='auto'
        )
        subtask.enqueue()

        return subtask.id

    def check_subtask_status(self, subtask):
        if subtask.status in (task.Status.FAILURE, task.Status.EXCEPTION):
            self.Context.errors.append('Subtask {} failed'.format(subtask.id))
        else:
            self.Context.messages.append('Subtask {} have finished successfully'.format(subtask.id))

    def on_execute(self):
        confs_to_gen = list(self.Parameters.confs)
        logging.info("Build params:\nconfs_to_gen: {}".format(confs_to_gen))

        if not self.Parameters.gen_bin_resource:
            with self.memoize_stage.build_gen_bin:
                subtask = self.build_generator(_GEN_BIN_PATH)
                self.Context.subtasks['CLUSTER_CONF_GENERATOR_BIN'] = subtask

                raise sdk2.WaitTask([subtask], task.Status.Group.FINISH | task.Status.Group.BREAK, wait_all=True)

        subtasks = list(self.find())
        if subtasks:
            for subtask in subtasks:
                self.check_subtask_status(subtask)

        if self.Context.errors:
            raise SandboxTaskFailureError('One of child tasks failed unexpectedly')

        if self.Parameters.gen_bin_resource:
            gen_bin_res = self.Parameters.gen_bin_resource
        else:
            gen_bin_res = sdk2.Resource.find(
                type='CLUSTER_CONF_GENERATOR_BIN',
                task_id=self.Context.subtasks['CLUSTER_CONF_GENERATOR_BIN']
            ).first()
            logging.info("Cluster.conf generator resource: {}".format(gen_bin_res.id))
        gen_bin = str(sdk2.ResourceData(gen_bin_res).path)

        os.environ['YP_TOKEN'] = sdk2.Vault.data(self.Parameters.yp_token_owner, self.Parameters.yp_token_vault_name)

        for conf in self.Parameters.confs:
            out_dir = pj(os.getcwd(), conf)
            os.mkdir(out_dir)
            work_dir = pj(os.getcwd(), os.path.basename(_CONFIGS[conf]))
            os.mkdir(work_dir)
            sdk2.svn.Arcadia.checkout(_CONFIGS[conf], work_dir)

            if 'ONLINE' in conf:
                self.merge_online_confs(
                    _ONLINE_DC_LIST,
                    work_dir
                )
                dc_list = _ONLINE_DC_LIST
                gen_type = 'userproxy'
            else:
                dc_list = _DC_LIST
                gen_type = 'normal'

            prev_cluster_conf = pj(work_dir, 'cluster.conf')

            logging.info("Generating {}...".format(conf))
            cmd = [
                gen_bin,
                '--previous-cluster-conf', prev_cluster_conf,
                '--cluster-conf', pj(out_dir, 'cluster.conf'),
                '--previous-fqdn-to-ip', pj(work_dir, 'fqdn_to_ip.conf'),
                '--fqdn-to-ip', pj(out_dir, 'fqdn_to_ip.conf'),
                '--cfg', pj(work_dir, 'generator.yaml'),
                '--type', gen_type,
                '--cluster',
            ] + dc_list
            if 'ONLINE' in conf:
                cmd.append('--split-cfg-by-dc')
            logging.info("Run cmd: {}".format(' '.join(cmd)))
            subprocess.check_call(cmd)

            for f in os.listdir(out_dir):
                shutil.copy(pj(out_dir, f), work_dir)

            try:
                sdk2.svn.Arcadia.commit(work_dir, 'Updated cluster.conf and fqdn_to_ip.conf for {}. SKIP_CHECK'.format(conf), 'zomb-sandbox-rw')
            except sdk2.vcs.svn.SvnError:
                logging.error("There was a problem with commit!!!")

            shutil.rmtree(work_dir)
