import logging
import time

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2


class UpdateZoraDynamicConfigs(sdk2.Task):
    """
    Updates/releases Zora dynamic configs package.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 60000  # Mb

    class Parameters(sdk2.Task.Parameters):
        conf_type = sdk2.parameters.String(
            "Dynamic configs type. Zora or Rotor. Default: zora",
            default="zora",
            required=True
        )
        build_task_id = sdk2.parameters.Task(
            "Zora/Rotor dynamic configs package build task id",
            task_type="BUILD_ZORA_PACKAGES",
            required=True
        )
        release_resources_to = sdk2.parameters.String(
            "Release types (';'-separated).\nE.g.: \"prestable;stable\"",
            default="prestable;stable",
            required=True
        )
        change_revision = sdk2.parameters.String(
            "Revision of releasing changes."
        )

    def release_tasks(self, conf_type, build_task, release_types, change_rev):
        if build_task.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure("Task to release is in failed status!!!")

        rest_client = common.rest.Client()
        for rel_type in release_types.split(';'):
            logging.info("Releasing {} to {}".format(build_task.id, rel_type))
            payload = {
                "task_id": build_task.id,
                "cc": [],
                "to": [],
                "params": {},
                "message": "Auto-release of {} dynamic configs by UpdateZoraDynamicConfigs".format(conf_type),
                "type": rel_type,
                "subject": "Auto-release of {} dynamic configs by UpdateZoraDynamicConfigs.\nRevision: https://a.yandex-team.ru/arc/commit/{}".format(conf_type, change_rev),
            }
            rest_client.release(payload)
            while rest_client.task[build_task.id].read()['status'] != 'RELEASED':
                time.sleep(60)

    def on_execute(self):
        self.release_tasks(
            self.Parameters.conf_type,
            self.Parameters.build_task_id,
            self.Parameters.release_resources_to,
            self.Parameters.change_revision
        )
