# coding: utf-8

import os
import __builtin__

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.build.parameters as build_params


def target2field_name(target):
    return 'build_%s' % target


class ZoraBinaries(parameters.SandboxInfoParameter):
    name = 'info_field'
    description = 'Zora Binaries'


class StripBinaries(parameters.SandboxBoolParameter):
    name = 'zora_strip_binaries'
    description = 'Strip result binaries'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuse(build_params.UseArcadiaApiFuse):
    default_value = True


class YtStore(build_params.YtStore):
    default_value = True


def init(target_resources, custom_input_parameters):
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        UseArcadiaApiFuse,
        build_params.UseArcInsteadOfArcadiaApiAsDefault,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
        build_params.DefinitionFlags,
    ] + custom_input_parameters

    enabled_builds = {}
    for target_resource in target_resources:
        target = os.path.basename(target_resource.arcadia_build_path)
        field_name = target2field_name(target)
        build = __builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )
        input_parameters.extend([build])
        enabled_builds[field_name] = target_resource

    return input_parameters, enabled_builds


class BaseBuildZoraBins(YaMakeTask, nanny.ReleaseToNannyTask):
    type = 'BASE_BUILD_ZORA_BINS'

    enabled_builds = {}
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        UseArcadiaApiFuse,
        build_params.UseArcInsteadOfArcadiaApiAsDefault,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
    ]
    TARGET_RESOURCES = ()

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM
        self.ctx['clear_build'] = True
        self.ctx[StripBinaries.name] = False

    def get_targets(self):
        targets = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                targets.extend([os.path.dirname(resource_type.arcadia_build_path)])
        return targets

    def get_resources(self):
        resources = {}
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                resource = {
                    'description': resource_type.name,
                    'resource_type': resource_type,
                    'resource_path': os.path.basename(resource_type.arcadia_build_path),
                }
                resources[resource_type.name] = resource

        return resources

    def get_arts(self):
        arts = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                arts.extend([{'path': resource_type.arcadia_build_path}])
        return arts

    def __copy_symbols(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--only-keep-debug', object_file, symbols_file],
            log_prefix='copy_symbols'
        )

    def __strip_binary(cls, object_file):
        process.run_process(
            ['strip', '-g', object_file],
            log_prefix='strip_binary'
        )

    def __link_symbols_with_executable(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--add-gnu-debuglink', symbols_file, object_file],
            log_prefix='debuglink'
        )

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

    # Use name option to rename dir
    def _create_svn_resource(self, path, resource_name, resource_type):
        print path, resource_name, resource_type
        base_url = self.ctx[build_params.ArcadiaUrl.name]

        resource_root = repr(resource_type)
        os.mkdir(resource_root)

        Arcadia.export(
            Arcadia.append(base_url, path),
            os.path.join(resource_root, os.path.basename(path))
        )

        run_process(['tar', '-C', resource_root, '-zcf', resource_name, '.'])

        self.create_resource(
            '',
            resource_name,
            resource_type)
