import os

import yaml
import pyuwsgi

import six

from sandbox.common import config
from sandbox.common.types import misc as ctm

from sandbox.proxy import config as proxy_config


class IndentDumper(yaml.Dumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(IndentDumper, self).increase_indent(flow, False)


def indent_text(text, indent, indent_char=" "):
    return "".join(indent * indent_char + line + "\n" for line in text.splitlines())


def dump_config(config, indent=2):
    # Stored as strings, append them to the dumped config in the end
    shared_sockets = config["uwsgi"].pop("shared-sockets")
    http_sockets = config["uwsgi"].pop("http-sockets")
    caches = config["uwsgi"].pop("caches")
    metrics = config["uwsgi"].pop("metrics")
    mules = config["uwsgi"].pop("mules")

    config = yaml.dump(config, Dumper=IndentDumper, default_flow_style=False, indent=indent)
    config += indent_text(shared_sockets, indent)
    config += indent_text(http_sockets, indent)
    config += indent_text(caches, indent)
    config += indent_text(metrics, indent)
    config += indent_text(mules, indent)

    return config


def get_uwsgi_yaml_config(registry):
    from library.python import resource
    config = yaml.load(resource.find("sandbox/proxy/etc/uwsgi.yaml"), Loader=yaml.SafeLoader)
    uwsgi = config["uwsgi"]

    # Override defaults
    uwsgi["uid"] = registry.proxy.server.api.uid
    uwsgi["gid"] = registry.proxy.server.api.gid

    return dump_config(config)


def get_uwsgi_args():
    if proxy_config.Registry().common.installation == ctm.Installation.LOCAL:
        dbg = proxy_config.Registry().proxy.server.debug
        cmd = [
            "--shared-socket", ":".join([dbg.host, str(dbg.port)]),
            "--http-socket", "=0",
            "--chunked-input-limit", str(100 << 20),
            "--http-chunked-input",
            "--lazy-apps",
            "--chunked-input-timeout", "21600",
            "--processes", "2",
            "--threads", "2",
            "--hook-as-user-atexit", "print:goodbye cruel world",
            "--die-on-term",
            "--module", "sandbox.proxy.app:create_app()",
            "--need-app",
            "--mule=sandbox.proxy.mules.clients:run_safe()",
            "--sharedarea", "200",
            "--log-format", " ".join([
                '[%(ltime)] %(remote_ip) %(user)',
                '"%(method) %(uri) %(proto)" %(status) %(hsize)+%(rsize)',
                '"%(referer)" "%(uagent)" %(msecs)ms [pid: %(pid)|wid: %(wid)|rid: %(rid)]',
            ]),
            "--websocket-pong-tolerance", "30",
            "--enable-metrics",
            "--metrics-no-cores",
            "--metric", "name=http_200,type=counter",
            "--metric", "name=http_201,type=counter",
            "--metric", "name=http_202,type=counter",
            "--metric", "name=http_307,type=counter",
            "--metric", "name=http_400,type=counter",
            "--metric", "name=http_401,type=counter",
            "--metric", "name=http_404,type=counter",
            "--metric", "name=http_405,type=counter",
            "--metric", "name=http_410,type=counter",
            "--metric", "name=http_413,type=counter",
            "--metric", "name=http_500,type=counter",
            "--metric", "name=http_503,type=counter",
            "--metric", "name=http_503,type=counter",
            "--metric", "name=http_OTHER,type=counter",
        ]
        return cmd
    else:
        r, w = os.pipe()
        os.write(w, six.ensure_binary(get_uwsgi_yaml_config(proxy_config.Registry())))
        os.close(w)
        return ["--yaml", "fd://" + str(r)]


if __name__ == "__main__":

    # Check if path to local settings is defined
    config.ensure_local_settings_defined()
    pyuwsgi.run(get_uwsgi_args())
