import time
import logging
import datetime as dt
import threading

import flask

from hbf import ipaddress2

from sandbox.common import fs as common_fs
from sandbox.common import rest as common_rest
from sandbox.common import patterns as common_patterns
from sandbox.common import statistics as common_statistics
from sandbox.common.types import misc as ctm
from sandbox.common.types import user as ctu
from sandbox.common.types import statistics as ctst

from sandbox.proxy import config


class BannedNetworks(object, metaclass=common_patterns.ThreadSafeSingletonMeta):
    UPDATE_INTERVAL = 60  # interval between updating banned networks

    _thread = None

    def __init__(self):
        auth = common_fs.read_settings_value_from_file(config.Registry().proxy.server.auth.token, True)
        self.rest = common_rest.Client(config.Registry().client.rest_url, auth=auth)
        self.banned_networks = {}
        self.banned_resources = {}
        self.whitelist_resources = {}
        self.update_banned_networks()

    def update_banned_networks(self):
        self.banned_networks = self.__perform_banned_networks(self.rest.service.banned_networks.read())
        try:
            self.banned_resources = {
                macros: frozenset(resources) for macros, resources in self.rest.service.banned_resources.read().items()
            }
        except:
            self.banned_resources = {}
        try:
            self.whitelist_resources = {
                macros: frozenset(resources)
                for macros, resources in self.rest.service.whitelist_resources.read().items()
            }
        except:
            self.whitelist_resources = {}

    @classmethod
    def __perform_banned_networks(cls, banned_networks):
        performed_banned_networks = {}
        for net_macros, networks in banned_networks.items():
            performed_networks = []
            for network in networks:
                try:
                    performed_networks.append(ipaddress2.ip_network(network))
                except Exception:
                    logging.exception("Can't create network class")
            performed_banned_networks[net_macros] = performed_networks
        return performed_banned_networks

    def main(self):
        while True:
            time.sleep(self.UPDATE_INTERVAL)
            self.update_banned_networks()

    def check_ip(self, ip):
        banned_networks = self.banned_networks
        ip_struct = ipaddress2.ip_address(ip)
        for net_macros, networks in banned_networks.items():
            for network in networks:
                if ip_struct in network:
                    return net_macros
        return None

    def start(self):
        self._thread = threading.Thread(target=self.main)
        self._thread.daemon = True
        self._thread.start()


def banned_request(user, resource_meta, logger):
    try:
        net_macros = BannedNetworks().check_ip(flask.g.remote_ip)
        if net_macros is not None:
            now = dt.datetime.utcnow()
            if (
                (
                    net_macros in BannedNetworks().banned_resources and
                    resource_meta["type"] in BannedNetworks().banned_resources[net_macros] or
                    net_macros in BannedNetworks().whitelist_resources and
                    resource_meta["type"] not in BannedNetworks().whitelist_resources[net_macros]
                ) and user == ctu.ANONYMOUS_LOGIN
            ):
                return True
            common_statistics.Signaler().push(dict(
                type=ctst.SignalType.PROXY_REQUESTS_DYNAMIC_NETS,
                date=now,
                timestamp=now,
                resource_id=resource_meta["id"],
                resource_type=resource_meta["type"],
                remote_ip=flask.g.remote_ip,
                user_agent=flask.request.headers.get(ctm.HTTPHeader.USER_AGENT, "UNKNOWN"),
                user=user,
                net_macros=net_macros.strip("_"),
                host=flask.request.headers.get("Host", "UNKNOWN"),
                referer=flask.request.headers.get("Referer", "UNKNOWN"),
            ))
            if user == ctu.ANONYMOUS_LOGIN:
                logger.info(
                    "Request from banned address %s for resource %s", flask.g.remote_ip, resource_meta["id"]
                )
    except Exception:
        logger.exception(
            "Failed to check address %s for resource %s.", flask.g.remote_ip, resource_meta["id"]
        )
