import time
import logging

import six
import requests

from sandbox.common import format as common_format


# Socket timeout on any socket operation while streaming data for download request.
DOWNLOAD_SOCKET_TIMEOUT = 30
# CRLF
CRLF = b"\r\n"

# List of known proxy responses to count them
KNOWN_RESPONSES = (
    requests.codes.ACCEPTED,
    requests.codes.BAD_REQUEST,
    requests.codes.CREATED,
    requests.codes.GONE,
    requests.codes.INTERNAL_SERVER_ERROR,
    requests.codes.METHOD_NOT_ALLOWED,
    requests.codes.NOT_FOUND,
    requests.codes.OK,
    requests.codes.REQUEST_ENTITY_TOO_LARGE,
    requests.codes.SERVICE_UNAVAILABLE,
    requests.codes.TEMPORARY_REDIRECT,
    requests.codes.UNAUTHORIZED,
    requests.codes.UNAVAILABLE,
)


def flush_and_stream(logger, response, chunked):
    yield b""
    counter = 0
    duration = time.time()
    logger.debug("Headers flushed. Streaming the %sdata.", "chunked " if chunked else "")
    try:
        for chunk in response:
            counter += len(chunk)
            yield chunk if not chunked else b"".join((
                six.ensure_binary(hex(len(chunk))[2:]), CRLF, chunk, CRLF
            ))
        if chunked:
            yield b"".join((b"0", CRLF, CRLF))
    except GeneratorExit:
        pass
    except BaseException as ex:
        logger.exception("Error streaming data: %s", type(ex))
        raise
    finally:
        duration = time.time() - duration
        logger.debug(
            "Streamed %s in %.2fs @ %s/s.",
            common_format.size2str(counter), duration, common_format.size2str(counter / duration)
        )


class RequestAdapter(logging.LoggerAdapter):

    def __init__(self, logger, rid):
        self.req_id = rid
        super(RequestAdapter, self).__init__(logger, {})

    def process(self, msg, kwargs):
        return "{{{}}} {}".format(self.req_id, msg), kwargs
