import sys
import logging
import logging.config

import flask

from sandbox.common import log as common_log
from sandbox.common import auth as common_auth
from sandbox.common import rest as common_rest
from sandbox.common import proxy as common_proxy

import sandbox.common.types.misc as ctm

from sandbox.proxy import config as proxy_config
from sandbox.proxy import plugins


class Context(object):
    """ The class encapsulates application context and its initialization stuff. """

    LOGGING_LEVEL_ALIASES = {
        logging.DEBUG: "DBG",
        logging.INFO: "INF",
        logging.WARNING: "WRN",
        logging.ERROR: "ERR",
        logging.CRITICAL: "CRI",
        logging.NOTSET: "---",
    }

    def _root_logger(self, name):
        logger = logging.getLogger(name)
        logging.config.dictConfig(self.config.proxy.logging)
        for no, alias in self.LOGGING_LEVEL_ALIASES.items():
            logging.addLevelName(no, alias)
        logging.getLogger().addHandler(common_log.ExceptionSignalsSenderHandler(registry=self.config))
        return logger

    @staticmethod
    def auto_auth():
        auth = None
        if not flask.request:
            return None
        session_id = flask.request.cookies.get(common_proxy.YandexSession.COOKIE_NAME)
        if session_id:
            auth = common_proxy.YandexSession(session_id)
        else:
            auth_header = flask.request.headers.get("Authorization")
            if auth_header and auth_header.startswith("OAuth "):
                auth = common_proxy.OAuth(auth_header[6:])
            else:
                service_ticket = flask.request.headers.get(ctm.HTTPHeader.SERVICE_TICKET)
                user_ticket = flask.request.headers.get(ctm.HTTPHeader.USER_TICKET)
                if service_ticket and user_ticket:
                    auth = common_auth.TVMSession(service_ticket, user_ticket)
        return auth

    def _on_request_to_sandbox_api(self, request: common_rest.Client.Request):
        if request.response is None:
            return
        plugins.request_context.provide_user(request.response.headers.get(ctm.HTTPHeader.CURRENT_USER))
        rs = flask.g.get("request_statistics")
        if rs:
            rs.on_request_to_api(request)
        else:
            self.logger.warning("Impossible to handle request: request statistics aren't initialized")

    def rest(self, logger=None, **kwargs) -> common_rest.Client:
        client = common_rest.Client(
            self.config.client.rest_url,
            auth=kwargs["auth"] if "auth" in kwargs else self.auto_auth(),
            logger=logger or flask.g.logger,
            component=ctm.Component.PROXY,
        )
        client.RETRYABLE_CODES = tuple(set(client.RETRYABLE_CODES) - {client.TOO_MANY_REQUESTS})
        if client.request_callback is None:
            client.request_callback = self._on_request_to_sandbox_api
        return client

    def __init__(self, parent=None, name=None):
        name = self.name = name or self.__class__.__module__.split(".")[0]
        if parent:
            parent.logger.debug("Creating child context '%s'", name)
            self.logger = parent.logger.getChild(name)
            self.config = parent.config
        else:
            self.config = proxy_config.Registry()
            self.logger = self._root_logger(name)
            if not sys.stdout.isatty():
                rl = logging.getLogger()
                for h in rl.handlers:
                    if h.get_name() == "console":
                        rl.removeHandler(h)
                        break
