import io
import os
import re
import abc
import sys
import time
import json
import errno
import queue
import socket
import select
import random
import tarfile
import hashlib
import traceback
import contextlib
import collections
import configparser
import functools as ft
import itertools as it
import threading as th
import distutils.util

from urllib import parse as urlparse

import logging

import six
import flask
import msgpack
import requests

from sandbox.proxy import websocket
from sandbox.proxy import flask_uwsgi_websocket

from sandbox.common import fs as common_fs
from sandbox.common import abc as common_abc
from sandbox.common import mds as common_mds
from sandbox.common import rest as common_rest
from sandbox.common import format as common_format
from sandbox.common import upload as common_upload
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics
from sandbox.common.mds import compression as common_mds_compression
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.database as ctd
import sandbox.common.types.resource as ctr

from sandbox.proxy import common as proxy_common
from sandbox.proxy import context as proxy_context
from sandbox.proxy import plugins
from sandbox.proxy import mds_download as proxy_mds
from sandbox.proxy import response_types as rt
from sandbox.proxy import banned_networks
from sandbox.proxy.mules import clients as clients_mule

try:
    import uwsgi
except ImportError:
    class FakeUWSGI:
        """ Fake uWSGI interface to work with local tests. """

        def __init__(self):
            self.cache = {}

        def cache_get(self, k):
            return self.cache[k]

        def cache_set(self, k, v, _):
            self.cache[k] = v

        def cache_exists(self, k):
            return k in self.cache

        @staticmethod
        def chunked_read():
            return ""

        @staticmethod
        def masterpid():
            return 0

        @staticmethod
        def worker_id():
            return 0

        @staticmethod
        def connection_fd():
            return 0

        @staticmethod
        def metric_inc(_):
            pass

        @staticmethod
        def metric_get(_):
            return 0

        @staticmethod
        def total_requests():
            return 0

    uwsgi = FakeUWSGI()


File = collections.namedtuple("File", ("filename", "contents", "size"))


def websocket_handler(method):
    """
    Decorator for proper logging of requests routed via ``flask.ext.uwsgi_websocket.websocket.WebSocket``.
    Emits ``flask.request_started`` and ``flask.request_finished`` (``WebSocketMiddleware`` doesn't do that)

    :param method: flask.Flask instance's method to be decorated
    """

    @ft.wraps(method)
    def wrapper(flask_app, ws, *a, **kw):
        with flask_app.request_context(ws.environ):
            flask.request_started.send(flask_app)
            response = flask.make_response(method(flask_app, ws, *a, **kw))
            flask.request_finished.send(flask_app, response=response)
        return response

    return wrapper


class DataSource(object):
    """
    Base class to restream files pouring into `/upload/{task_id}` and `/upload/{resource_type}` handlers
    to the corresponding `HTTP_UPLOAD` task. For concrete implementations, see the below classes.
    """

    __metaclass__ = abc.ABCMeta

    IPV4_RE = re.compile(r"^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$")  #: Regular expression to match IPv4 address

    def __init__(self, logger, host, port, files, streamed=0, expected=0):
        self.proto = common_upload.Proto()
        self.logger = logger
        self.files = files
        self.streamed = streamed
        self.expected = expected
        self.duration = 0
        self._sha1 = hashlib.sha1()
        self.sha1 = None
        self.sock = socket.socket(socket.AF_INET if self.IPV4_RE.match(host) else socket.AF_INET6)
        self.sock.settimeout(Application.UPLOAD_SOCKET_TIMEOUT)
        self.sock.connect((host, port))
        super(DataSource, self).__init__()

    def greet(self, cnonce, node_id, request_id):
        greet = self.proto.Greetings(None, cnonce, node_id, request_id)
        self.sock.sendall(self.proto(greet))
        self.logger.debug("Greetings sent.")

    def write(self, chunk):
        self.sock.sendall(self.proto(self.proto.DataChunk(chunk)))
        self.streamed += len(chunk)

    def summarize(self):
        self.sock.sendall(self.proto(self.proto.StreamSummary(self.streamed, self.sha1)))
        self.logger.info(
            "Totally streamed %s (%s bytes) in %.2fs @ %s/s. SHA1 is %r",
            common_format.size2str(self.streamed), self.streamed,
            self.duration, common_format.size2str(self.streamed / self.duration),
            self.sha1
        )

    @abc.abstractmethod
    def stream(self, task_version):
        pass

    @abc.abstractmethod
    def wait(self, rest, tid):
        pass

    def abort(self):
        self.sock.sendall(self.proto(self.proto.Break()))


class ProxySource(DataSource):
    """
    Helper class for streaming files from /upload/<resource_type> to a task.
    After the end of the stream, waits for the task to enter one of finishing states and returns resource info.
    """

    def write(self, chunk):
        super(ProxySource, self).write(chunk)
        self._sha1.update(chunk)

    def stream(self, task_version):
        duration = time.time()
        with tarfile.open(mode="w|", bufsize=0xFFFFF, fileobj=self) as tar:
            for f in self.files:
                self.logger.debug("Sending file %r of size %r", f.filename, f.size)
                obj = io.BytesIO(f.contents)
                info = tarfile.TarInfo(f.filename if len(self.files) == 1 else os.path.sep.join(["data", f.filename]))
                info.size = f.size
                tar.addfile(info, fileobj=obj)

        self.write("")
        self.sha1 = self._sha1.hexdigest()
        self.duration = time.time() - duration
        self.summarize()

    def wait(self, rest, tid):
        correct_statuses = [ctt.Status.PREPARING, ctt.Status.EXECUTING, ctt.Status.FINISHING]
        self.logger.debug("Waiting for task #%s to finish.", tid)
        for _, _ in common_itertools.progressive_yielder(1, 3, Application.UPLOAD_FINISHING_TIMEOUT, False):
            status = rest.task[tid][:]["status"]
            if status == ctt.Status.SUCCESS:
                break
            if status not in correct_statuses:
                return Application.JSONResponse(
                    requests.codes.UNAVAILABLE,
                    "Task #{} switched to incorrect state '{}' (see task's logs for details).".format(tid, status)
                )

        ctx = rest.task[tid].context[:]
        self.logger.debug("Task #%s upload context is %r", tid, ctx["upload"])
        if int(ctx["upload"]["received"]) != self.streamed:
            return Application.JSONResponse(
                requests.codes.UNAVAILABLE,
                "Task received stream of {} bytes, while proxy sent {} bytes".format(
                    ctx["upload"]["received"], self.streamed
                )
            )

        if ctx["upload"]["checksum"] != self._sha1.hexdigest():
            return Application.JSONResponse(
                requests.codes.UNAVAILABLE,
                "Task calculated stream SHA1 {}, while that on proxy's side is {}".format(
                    ctx["upload"]["checksum"], self._sha1.hexdigest()
                )
            )

        return Application.JSONResponse(requests.codes.CREATED, rest.resource[ctx["resource_id"]][:])


def uwsgi_chunker(logger):
    chunk = None
    while True:
        for i in six.moves.xrange(2, -1, -1):
            try:
                chunk = uwsgi.chunked_read()
                break
            except Exception as ex:
                if not i or isinstance(ex, socket.error):
                    raise
                # Maybe it was just EINTR - parent wants to cheap us, but it can be postponed.
                logger.warning("Unable to receive next chunked transfer part: (%s) %s", type(ex), ex)
                time.sleep(.5)

        if not chunk:
            break

        yield chunk


class UwsgiSource(DataSource):
    """
    Helper class to accept data from common.upload script (/upload/<task_id>) via uwsgi
    """

    def stream(self, task_version):
        duration = time.time()
        progress, prev = 0, None
        for chunk in uwsgi_chunker(self.logger):
            if prev is not None:
                self.write(prev)
                curp = self.streamed * 100 / self.expected if self.expected else 100
                if curp != progress:
                    self.logger.debug(
                        "Streamed so far %s (%d bytes, %s%%)",
                        common_format.size2str(self.streamed), self.streamed, curp
                    )
                progress = curp
            prev = chunk
        if task_version > 1:
            sha1_sz = hashlib.sha1().digest_size * 2
            prev, sha1 = prev[:-sha1_sz], prev[-sha1_sz:]
        else:
            sha1 = ""
        self.write(prev)
        if prev:
            # TODO: DROPME: Finish the data stream for protocol version == 1
            self.write("")
        self.sha1 = sha1
        self.duration = time.time() - duration
        self.summarize()

    def wait(self, rest, tid):
        return str(self.streamed)


@six.add_metaclass(rt.PerProcessMeta)
class Checkers(object):
    Stats = collections.namedtuple("CheckerStats", ("queue", "busy", "total"))

    class Job(common_patterns.Abstract):
        __slots__ = ("rqid", "rid", "ua", "cid", "score", "link", "queue", "invalid", "fqdn")
        __defs__ = [None] * 9

        def __gt__(self, other):
            return self.score > other.score

    class WorkerAdapter(logging.LoggerAdapter):
        def __init__(self, logger, wid):
            self.id = wid
            super(Checkers.WorkerAdapter, self).__init__(logger, {})

        def process(self, msg, kwargs):
            return "<worker {}> {}".format(self.id, msg), kwargs

    def __init__(self, ctx):
        self.logger = ctx.logger.getChild("checkers")
        self.head_request_timeout = ctx.config.proxy.fileserver.head_request_timeout
        self.logger.info(
            "Creating %d workers pool in process #%d.", ctx.config.proxy.server.checkers.amount, os.getpid()
        )
        self.pool = [
            th.Thread(target=self._worker, name=str(_))
            for _ in six.moves.xrange(ctx.config.proxy.server.checkers.amount)
        ]
        [setattr(_, "daemon", True) for _ in self.pool]
        self.queue = queue.Queue()
        # Actually, it will be better to use "atomic.AtomicLong(0)" here for such purpose, but it too hard to deal
        # with installing "libffi-dev" on each host, where virtual environment should be prepared.
        self._counter = collections.deque()
        for thread in self.pool:
            th.Thread.start(thread)

    @property
    def stats(self):
        return self.Stats(self.queue.qsize(), len(self._counter), len(self.pool))

    @staticmethod
    def _resolve_hostname(url, logger):
        """
        This is an attempt to mitigate network problems (domain name resolution failures, mostly)
        by manually resolving domain names.

        :return: URL with FQDN replaced with an IPv6 address
        """

        parsed = urlparse.urlparse(url)
        split = parsed.netloc.rsplit(":", 1)
        host, port = split if len(split) == 2 else (split[0], 80)
        try:
            # getaddrinfo returns a list of (family, sock type, protocol, canon. name, (ip address, port, ?, ?)) tuples
            ipaddr = socket.getaddrinfo(host, int(port), socket.AF_INET6, socket.SOCK_STREAM)[0][4][0]
        except socket.error:
            logger.exception("Failed to resolve %s", host)
            return url

        logger.debug("Resolved %s -> %s", host, ipaddr)
        wrapped_ipaddr = ipaddr if "." in ipaddr else "[{}]".format(ipaddr)
        return urlparse.urlunparse(parsed._replace(netloc=":".join((wrapped_ipaddr, port))))

    def _worker(self):
        main_logger = self.WorkerAdapter(self.logger, th.current_thread().name)
        main_logger.info("Started.")
        try:
            while True:
                job = self.queue.get()
                if job is None:
                    break
                self._counter.append(1)
                logger = proxy_common.RequestAdapter(main_logger, job.rqid)
                if job.invalid.is_set():
                    logger.debug("Job invalidated.")
                    continue
                try:
                    s = requests.session()
                    job.link = self._resolve_hostname(job.link, logger)
                    logger.debug("Request HEAD (score %d) %r", job.score, job.link)
                    r = s.head(job.link, timeout=self.head_request_timeout, headers={ctm.HTTPHeader.USER_AGENT: job.ua})
                    r.raise_for_status()
                    job.queue.put((s, job))
                    logger.debug("Job done.")
                except Exception as ex:
                    logger.warning("Resource #%s is not available at %s: %s", job.rid, job.cid, ex)
                    job.queue.put((None, job))
                self._counter.pop()
        finally:
            main_logger.info("Stopped.")


class HashCalculator(common_mds.HashCalculator):
    def __init__(self, queue_, factory_id):
        super(HashCalculator, self).__init__()
        self.__queue = queue_
        self.__factory_id = factory_id
        self.stopped = th.Event()

    @property
    def __id(self):
        return th.current_thread().ident, self.__factory_id

    def start(self):
        pass

    def stop(self):
        self.__queue.put((None, self.__id))
        self.stopped.wait()

    def update_hashes(self, data):
        self._update(data)

    def update(self, data):
        if data != "":
            self.__queue.put((data, self.__id))


@six.add_metaclass(rt.PerProcessMeta)
class HashCalculatorFactory(object):
    def __init__(self):
        self.__queue = queue.Queue()
        self.__hash_calculators = {}
        self.__thread = th.Thread(target=self.__loop)
        self.__thread.daemon = True
        self.__thread.start()

    def __call__(self, factory_id=0):
        thread_id = th.current_thread().ident
        calc_id = (thread_id, factory_id)
        hash_calculator = self.__hash_calculators[calc_id] = HashCalculator(self.__queue, factory_id)
        return hash_calculator

    def __loop(self):
        while True:
            data, calc_id = self.__queue.get()
            hash_calculator = self.__hash_calculators[calc_id]
            if data:
                hash_calculator.update_hashes(data)
            elif data is None:
                hash_calculator.stopped.set()

    def cleanup(self):
        self.__hash_calculators.pop(th.current_thread().ident, None)


class Application(flask.Flask):
    # Socket timeout on any operation with upload target host.
    UPLOAD_SOCKET_TIMEOUT = 90
    # Timeout on waiting for upload task to finish
    UPLOAD_FINISHING_TIMEOUT = 300
    # Default timeout on API requests
    API_DEFAULT_TIMEOUT = 5
    # Task type, which is used to handle HTTP upload.
    UPLOAD_TASK_TYPES = ["HTTP_UPLOAD", "HTTP_UPLOAD_2"]
    # Max allowed Content-Length for direct upload
    DIRECT_UPLOAD_LIMIT = (1 << 30) + 15 * (1 << 10)  # extra 15Kb for delimiters and stuff
    # Maximum allowed priority for upload tasks
    UPLOAD_TASK_MAX_PRIORITY = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.HIGH)
    # Max chunk size for updating client cache
    CLIENTS_CACHE_CHUNK_SIZE = 200
    # maximal no. of attempts to refresh client cache for a given chunk
    REFRESH_ATTEMPTS = 3
    # Known query parameters, which are recognized by file server.
    FILESERVER_KNOWN_ARGS = {"stream", "force_text_mode", "force_binary_mode", "force_json", "tail"}
    # List of query words can be used to search resources (see `sandbox.web.api.v1.resource.ResourceList`)
    KNOWN_SEARCH_FIELDS = {  # TODO: SANDBOX-7154
        "id", "type", "arch", "state", "owner", "client", "task_id", "accessed", "created", "attrs", "any_attr",
        "attr_name", "attr_value", "dependant"
    }
    SUPPORTED_ARCHIVE_EXTENSIONS = (".tar", ".tgz", ".tar.gz")

    CachedResponse = collections.namedtuple("CachedResponse", ["value", "cached"])

    class JSONResponse(flask.Response):
        """ JSON response tuple """

        def __init__(self, code, message, headers=None):
            headers = headers or {}
            headers["Content-Type"] = "application/json"
            if not isinstance(message, str):
                message = json.dumps(message, ensure_ascii=False)
            super(Application.JSONResponse, self).__init__(message, code, headers)

    class JSONError(JSONResponse):
        """ JSON error response tuple """

        def __init__(self, ctx, code_or_error=None, reason=None, headers=None):
            logger = flask.g.logger
            if not code_or_error:
                ei = sys.exc_info()
                buf = io.StringIO()
                traceback.print_exception(ei[0], ei[1], ei[2], None, buf)
                tb = buf.getvalue()
                buf.close()
                if tb[-1:] == "\n":
                    tb = tb[:-1]
                logger.exception("Responding error message with exception")
                super(Application.JSONError, self).__init__(
                    requests.codes.INTERNAL_SERVER_ERROR,
                    {
                        "reason": str(ei[1]),
                        "traceback": tb[:-1] if tb[-1:] == "\n" else tb,
                        "server": ctx.config.this.id,
                    },
                    headers
                )
            elif isinstance(code_or_error, common_rest.Client.HTTPError):
                logger.warning("Proxying error %s", code_or_error)
                super(Application.JSONError, self).__init__(
                    code_or_error.status,
                    code_or_error.response.text,
                    {
                        _: code_or_error.response.headers.get(_)
                        for _ in ["Content-Type"] if _ in code_or_error.response.headers
                    }
                )
            else:
                logger.warning("Responding error code %d: %s", code_or_error, reason)
                super(Application.JSONError, self).__init__(code_or_error, {"reason": reason}, headers)

    def _perform_cached_request(self, key, request):
        key = hashlib.md5(six.ensure_binary(str(key))).hexdigest()
        value = uwsgi.cache_get(key, self.cache_conf.rest.name)

        if value:
            value = msgpack.unpackb(value)
            return self.CachedResponse(value, True)
        else:
            value = request()
            uwsgi.cache_set(
                key, msgpack.packb(value, use_bin_type=True),
                self.cache_conf.rest.expires,
                self.cache_conf.rest.name
            )

        return self.CachedResponse(value, False)

    def _get_user_and_resource_by_id(self, rest, rid):
        rhdrs = rest.HEADERS()
        rest = rest << rest.HEADERS({
            ctm.HTTPHeader.NO_LINKS: "true",
            ctm.HTTPHeader.TOUCH_RESOURCE: "true",
        }) >> rhdrs
        data, cached = self._perform_cached_request(rid, lambda: rest.resource[rid][:])
        user = (
            rest.user.current[:]["login"]  # Response cached - ask for username only
            if cached else
            rhdrs.headers.response.get(ctm.HTTPHeader.CURRENT_USER) or ctu.ANONYMOUS_LOGIN
        )
        return user, data

    def _get_resources_by_args(self, rest, args):
        rest = rest << rest.HEADERS({ctm.HTTPHeader.NO_LINKS: "true"})
        key = u";".join(u"{}:{}".format(k, v) for k, v in sorted(args.items())).encode("utf8")
        return self._perform_cached_request(key, lambda: rest.resource.read(**args)["items"]).value

    def _get_task_by_id(self, rest, tid):
        def request():
            return rest.task[tid].read()
        return self._perform_cached_request(tid, request).value

    @property
    def logger(self):
        return self.ctx.logger

    @property
    def cache_conf(self):
        return self.ctx.config.proxy.server.cache

    def _get_clients_info(self, logger, rest, ids=None):
        if ids is None:
            return {}

        clients = {}
        unknown = set()

        for cid in set(ids):
            value = clients_mule.ClientCache.get_client(cid)
            if value:
                clients[cid] = value
            else:
                unknown.add(cid)

        if unknown:
            unknown = sorted(unknown)
            logger.info("Refreshing client information about %r", unknown)

            for chunk in common_itertools.chunker(unknown, self.CLIENTS_CACHE_CHUNK_SIZE):
                data = []
                for _ in six.moves.xrange(self.REFRESH_ATTEMPTS):
                    try:
                        data = rest.client.read(
                            id=",".join(chunk),
                            limit=len(chunk),
                            fields=["id", "fileserver", "dc", "alive", "tags", "fqdn"],
                        )["items"]
                        break
                    except requests.ConnectionError:
                        pass
                else:
                    logger.error("Failed to refresh client information about chunk %r", chunk)
                clients.update({_["id"]: _ for _ in data})
                for cl in data:
                    clients_mule.ClientCache.add_client(cl)
        return clients

    def _get_client_info(self, client, logger=None, rest=None):
        logger = logger or self.ctx.logger
        cache = self._get_clients_info(logger, rest or self.ctx.rest(logger), [client]).get(client)
        if cache:
            return cache
        logger.warning("Client %r not found", client)

    def _best_source(self, logger, rest, rid, tid, fname, sources, ua):
        mydc = self.ctx.config.this.dc
        cache = self._get_clients_info(logger, rest, sources)

        random.shuffle(sources)
        heap = []
        for _ in sources:
            c = cache.get(_)
            if not c:
                continue
            score = 0
            if c["alive"]:
                score += 1 << 4
            if ctc.Tag.STORAGE in c["tags"]:
                score += 1 << 3
            if c["dc"] and c["dc"] == mydc:
                score += 1 << 2
            heap.append((-score, c))

        heap.sort(key=lambda _: _[0])
        started = time.time()
        chk = Checkers(self.ctx)
        chk_st = chk.stats
        logger.debug(
            "Sources %d, heap: %r, workers: %d/%d/%d",
            len(sources), [(_[0], _[1]["id"]) for _ in heap[:5]], *chk_st
        )
        job = chk.Job(flask.g.req_id, rid, ua, None, None, None, queue.Queue(), th.Event(), None)
        legacy_path = "/".join(common_itertools.chain(ctt.relpath(tid), urlparse.quote(fname)))
        new_path = "/".join(("resource", str(rid), urlparse.quote(fname.split("/")[-1])))
        for s, c in heap[:chk_st.total]:
            j = chk.Job(*(v for _, v in job))
            if not c["fileserver"]:
                logger.warning("Client %r has not fileserver link!", c["id"])
                continue
            link = c["fileserver"] + (new_path if ctc.Tag.NEW_LAYOUT in c["tags"] else legacy_path)
            j.cid, j.score, j.link, j.fqdn = c["id"], s, link, c["fqdn"]
            chk.queue.put(j)

        best = (None,) * 2
        can_wait = chk.head_request_timeout
        for _ in sources:
            try:
                resp = job.queue.get(True, can_wait)
                session, job = resp
                if not session:
                    continue
            except queue.Empty:
                logger.warning("Check result waiting timed out.")
                break

            if not best[0] or best[1] > job:
                best = resp
            can_wait = started + min(can_wait, abs(heap[0][0] - job.score)) - time.time()
            logger.debug("Got session scored %d for %r. Can wait: %.2f", job.score, job.cid, can_wait)
            if can_wait < 0:
                break

        job.invalid.set()
        return best

    @common_patterns.singleton_property
    def __s3_credentials(self):
        cp = configparser.ConfigParser()
        if not cp.read(os.path.expanduser(self.ctx.config.common.mds.s3.credentials)) or not cp.has_section("default"):
            return
        return dict(cp.items("default"))

    def __websocket_proxy(self, url, ws, headers=None):
        logger = self.ctx.logger
        backend_ws = websocket.WebSocket()
        kws = {}
        if headers:
            existent_headers = set(six.iterkeys(headers)) & {"Cookie", "Host", "Origin"}
            kws.update({h.lower(): headers[h] for h in existent_headers})
        logger.debug("Connecting to websocket at %r", url)
        backend_ws.connect(url, **kws)
        backend_ws.sock.setblocking(0)
        try:
            logger.debug("Starting message loop for %r", url)
            while ws.connected:
                msg = ws.recv_nb()
                if msg:
                    backend_ws.send(msg)
                try:
                    backend_msg = backend_ws.recv()
                except websocket.WebSocketConnectionClosedException:
                    break
                except socket.error as ex:
                    if ex.errno == errno.EAGAIN:
                        backend_msg = None
                    else:
                        raise
                if backend_msg:
                    ws.send(backend_msg)
                if not (msg or backend_msg):
                    time.sleep(0.01)
        except IOError:
            pass
        finally:
            logger.debug("Disconnecting from %r", url)
            backend_ws.close()

    def __client_by_task(self, logger, tid):
        rest = self.ctx.rest()
        try:
            task = self._get_task_by_id(rest, tid)
        except common_rest.Client.HTTPError as ex:
            logger.error("Error while fetching info for task #%s: %s", tid, ex)
            return self.JSONError(
                self.ctx,
                requests.codes.SERVICE_UNAVAILABLE,
                "Error while fetching info for task #{}: {}".format(tid, ex)
            )

        client = task["execution"].get("client", {"id": None})["id"]
        if not client:
            return self.JSONError(self.ctx, requests.codes.NOT_FOUND, "Task #{} is not executing".format(tid))
        client_info = self._get_client_info(client, logger, rest)
        if not client_info:
            return self.JSONError(self.ctx, requests.codes.NOT_FOUND, "Client {} not found".format(client))
        return client_info

    def ps(self, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci
        url = "{}ps/{}".format(ci["fileserver"], tid)
        try:
            logger.debug("Request process list via URL %r", url)
            r = requests.get(url, cookies=flask.request.cookies)
            return r.text, r.status_code, dict(r.headers)
        except Exception:
            return self.JSONError(self.ctx)

    def actions(self, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci
        url = "{}actions/{}".format(ci["fileserver"], tid)
        try:
            logger.debug("Request task actions list via URL %r", url)
            r = requests.get(url, cookies=flask.request.cookies, headers=dict(flask.request.headers))
            return r.text, r.status_code, dict(r.headers)
        except Exception:
            return self.JSONError(self.ctx)

    def shell(self, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci

        url = "{}shell/{}".format(ci["fileserver"], tid)
        try:
            logger.debug("Request shell file via URL %r", url)
            r = requests.get(url)
            response = flask.Response(r.text, r.status_code, dict(r.headers))
            response.set_cookie("shell_cookie", ci["id"])
            return response
        except Exception:
            return self.JSONError(self.ctx)

    def shell_static(self, _):
        logger = flask.g.logger
        client = flask.request.cookies.get("shell_cookie")
        if not client:
            return self.JSONError(self.ctx, requests.codes.NOT_FOUND, "shell_cookie not set")

        client_info = self._get_client_info(client, logger)
        if not client_info:
            return self.JSONError(self.ctx, requests.codes.NOT_FOUND, "Client {} not found".format(client)),

        url = client_info["fileserver"] + flask.request.path.lstrip("/")
        try:
            logger.debug("Request shell file via URL %r", url)
            r = requests.get(url, headers=dict(flask.request.headers), params=dict(flask.request.args))
            response = flask.Response(r.content, r.status_code, dict(r.headers))
            return response
        except Exception:
            return self.JSONError(self.ctx)

    @websocket_handler
    def shell_ws(self, ws, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci
        logger.debug("Request shell websocket for client %r", ci["id"])
        with self.request_context(ws.environ):
            headers = flask.request.headers
        url = "{}shell/{}/ws".format(
            urlparse.urlunparse(urlparse.urlparse(ci["fileserver"])._replace(scheme="ws")),
            tid
        )
        self.__websocket_proxy(url, ws, headers=headers)

    @websocket_handler
    def tail(self, ws, tid, relpath):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci

        url = (
            urlparse.urlunparse(urlparse.urlparse(ci["fileserver"])._replace(scheme="ws")) +
            "/".join(it.chain(("tail",), ctt.relpath(tid), (relpath,)))
        )
        self.__websocket_proxy(url, ws)

    @websocket_handler
    def debugger_ws(self, ws, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        meta = {"node_id": self.ctx.config.this.id, "request_id": flask.g.req_id}
        if isinstance(ci, self.JSONResponse):
            meta.update({"error": ci.code, "reason": ci.content})
            ws.send(json.dumps(meta))
            return ""

        with contextlib.closing(socket.socket(socket.AF_INET6)) as sock:
            sock.connect(("sandbox.yandex-team.ru", 80))
            host = sock.getsockname()[0]

        sock = socket.socket(socket.AF_INET6)
        sock.bind(("", 0))
        port = sock.getsockname()[1]

        logger.info("Requested debugger websocket for client %r. Listening on %s:%s", ci["id"], host, port)
        meta.update({"fileserver": ci["fileserver"], "host": host, "port": port})
        ws.send(json.dumps(meta))

        logger.info("Waiting for connection on %s:%s", host, port)
        try:
            sock.listen(0)
            sock.settimeout(self.UPLOAD_SOCKET_TIMEOUT)
            peer, addr = sock.accept()
            logger.debug("Accepted connection from %s:%s", addr[0], addr[1])
            peer.settimeout(None)
        except socket.error:
            logger.exception("Error accepting new debugger connection")
            return ""

        websocket_fd = uwsgi.connection_fd()
        while ws.connected:
            ready, _, _ = select.select([peer, websocket_fd], [], [])
            if websocket_fd in ready:
                data = ws.recv()
                if not data:
                    logger.info("WebSocket closed.")
                    peer.close()
                    break
                logger.debug("Got %d bytes from WebSocket", len(data))
                peer.send(data)
            if peer in ready:
                data = peer.recv(0xFFF)
                if not data:
                    logger.info("Peer socket closed.")
                    break
                logger.debug("Got %d bytes from peer", len(data))
                ws.send_binary(data)
        logger.info("Debugger socket tunnel closed")
        return ""

    def debugger_start(self, tid):
        logger = flask.g.logger
        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci
        url = "{}debugger/{}".format(ci["fileserver"], str(tid))
        logger.info("Starting remote debugger via %r", url)

        try:
            headers = dict(flask.request.headers)
            headers.update({
                ctm.HTTPHeader.TASK_ID: str(tid),
                ctm.HTTPHeader.REAL_IP: flask.g.remote_ip,
                ctm.HTTPHeader.FORWARDED_FOR: flask.g.remote_ip,
                ctm.HTTPHeader.INT_REQUEST_ID: flask.g.req_id,
                ctm.HTTPHeader.BACKEND_NODE: self.ctx.config.this.id,
            })
            r = requests.post(url, data=flask.request.data, headers=headers, timeout=self.UPLOAD_SOCKET_TIMEOUT)
        except Exception as ex:
            logger.exception(ex)
            return self.JSONResponse(
                requests.codes.UNAVAILABLE,
                {"reason": "Fileserver on host {} seems to be down".format(ci["id"])},
            )

        headers = dict(r.headers)
        headers[ctm.HTTPHeader.DATA_SOURCE] = ci["id"]
        logger.debug("Remote debugger start respond %r code.", r.status_code)
        return flask.Response(r.text, r.status_code, headers)

    def taskdir(self, tid=None, relpath=None):
        logger = flask.g.logger
        auth = self.ctx.auto_auth()
        auth_errmsg = "You have no permissions to access task #{} because you are "
        if not auth and self.ctx.config.common.installation != ctm.Installation.LOCAL:
            return self.JSONError(
                self.ctx, requests.codes.UNAUTHORIZED, (auth_errmsg + "not authenticated").format(tid)
            )
        task = tid and self._get_task_by_id(self.ctx.rest(auth=auth), tid)
        if task and task["rights"] != ctu.Rights.WRITE:
            return self.JSONError(
                self.ctx,
                requests.codes.UNAUTHORIZED,
                (auth_errmsg + "not a member of '{}'").format(tid, task["owner"])
            )

        ci = self.__client_by_task(logger, tid)
        if isinstance(ci, self.JSONResponse):
            return ci

        query = urlparse.urlencode(flask.request.args)
        logger.debug(
            "Proxy task #%s (host: %r) execution directory. Relpath is %r, query: %r",
            tid, ci["id"], relpath, query
        )
        try:
            link = list(common_itertools.chain(ci["fileserver"].rstrip("/"), ctt.relpath(tid)))
        except (ValueError, KeyError, StopIteration) as ex:
            rest = self.ctx.rest()
            task = self._get_task_by_id(rest, tid)
            logger.warning("Unable to determine execution directory link: %s", ex)
            return self.JSONResponse(
                requests.codes.GONE,
                {"reason": "Unable to determine {} task #{} execution directory location ({}).".format(
                    task["status"], task["id"], str(ex)
                )},
            )

        path = ["task", str(tid)]
        if relpath:
            relpath = six.ensure_str(relpath)
            relpath = relpath.lstrip("/")
            path.append(relpath)
            link.append(urlparse.quote(relpath))
        link = "/".join(link)
        if query:
            link = "?".join([link, query])
        logger.info("Preparing data stream for link %r hosted by %r", link, ci["id"])

        try:
            headers = dict(flask.request.headers)
            headers.update({
                ctm.HTTPHeader.FORWARDED_PATH: six.ensure_str("/".join(path)),
                ctm.HTTPHeader.TASK_ID: str(tid),
            })
            r = requests.get(link, stream=True, headers=headers)
        except Exception as ex:
            logger.exception(ex)
            return self.JSONResponse(
                requests.codes.UNAVAILABLE,
                {
                    "reason": "Fileserver on host {} seems to be down".format(ci["id"]),
                },
            )

        hdrs = dict(r.headers)
        hdrs[ctm.HTTPHeader.DATA_SOURCE] = ci["id"]
        return flask.Response(
            flask.stream_with_context(proxy_common.flush_and_stream(
                logger, rt.IterableStream(r, logger), r.headers.get("transfer-encoding", "") == "chunked"
            )),
            r.status_code,
            hdrs
        )

    def _upload(
        self,
        logger: logging.Logger,
        rest: common_rest.Client,
        tid: int,
        data_source: DataSource,
        files: list[File] = None,
    ):
        try:
            task = rest.task[tid][:]
            ctx = rest.task[tid].context[:]
            user = rest.user.current[:]["login"]
        except common_rest.Client.HTTPError as ex:
            return self.JSONError(self.ctx, ex)

        if task["author"] != user:
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "Task #{} of type '{}' in '{}' status created by '{}' instead of '{}'".format(
                    tid, task["type"], task["status"], task["author"], user
                )
            )

        if task["type"] not in self.UPLOAD_TASK_TYPES:
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "Task #{} of type '{}' in status '{}' created by '{}' is not of type {}".format(
                    tid, task["type"], task["status"], task["author"], self.UPLOAD_TASK_TYPES
                )
            )

        if flask.request.method == "HEAD" and task["status"] != ctt.Status.EXECUTING:
            if task["status"] == ctt.Status.ENQUEUING:
                logger.debug("Waiting for the task queueing.")
                for _, _ in common_itertools.progressive_yielder(.1, 3, self.UPLOAD_SOCKET_TIMEOUT, False):
                    task = rest.task[tid][:]
                    if task["status"] in (
                        ctt.Status.ENQUEUED, ctt.Status.ASSIGNED, ctt.Status.PREPARING, ctt.Status.EXCEPTION
                    ):
                        break

            # Used for task's priority raise.
            if task["status"] not in (ctt.Status.ENQUEUED, ctt.Status.ASSIGNED, ctt.Status.PREPARING):
                return self.JSONError(
                    self.ctx,
                    requests.codes.METHOD_NOT_ALLOWED,
                    "Task #{} of type '{}' in status '{}' created by '{}' is not enqueued".format(
                        tid, task["type"], task["status"], task["author"]
                    )
                )

            tp = ctt.Priority.make(task["priority"])
            if tp >= self.UPLOAD_TASK_MAX_PRIORITY:
                return self.JSONError(
                    self.ctx,
                    requests.codes.METHOD_NOT_ALLOWED,
                    "Task #{} of type '{}' in status '{}' created by '{}' priority {} is at maximum level".format(
                        tid, task["type"], task["status"], task["author"], tp
                    )
                )

            auth = common_fs.read_settings_value_from_file(self.ctx.config.proxy.server.auth.token, True)
            if not auth:
                logger.warning("No administrative token found (%r)", self.ctx.config.proxy.server.auth.token)
                return self.JSONError(self.ctx, requests.codes.METHOD_NOT_ALLOWED, "No administrative token found")

            srv_rest = self.ctx.rest(auth=auth)
            logger.info("Raising task #%s priority.", tid)
            try:
                srv_rest.task[tid] = {"priority": self.UPLOAD_TASK_MAX_PRIORITY}
            except common_rest.Client.HTTPError as ex:
                return self.JSONError(self.ctx, ex)
            return ""

        streamed = int(ctx["upload"].get("received", 0))
        if flask.request.method == "HEAD" and task["status"] == ctt.Status.EXECUTING:
            logger.debug("Waiting for already streamed data amount information.")
            for slept, _ in common_itertools.progressive_yielder(.1, 3, self.UPLOAD_SOCKET_TIMEOUT, False):
                if slept:
                    task = rest.task[tid][:]
                    ctx = rest.task[tid].context[:]
                    streamed = int(ctx["upload"].get("received", 0))
                if streamed or task["status"] != ctt.Status.EXECUTING:
                    break
            else:
                return self.JSONError(
                    self.ctx,
                    requests.codes.INTERNAL_SERVER_ERROR,
                    "Task #{} of type '{}' in status '{}' not provided correct context value.".format(
                        tid, task["type"], task["status"]
                    )
                )

            logger.info("Attempt to resume upload process at %d bytes.", streamed)
            return self.JSONResponse(
                requests.codes.OK,
                {
                    "task": tid,
                    "stream": {
                        "received": streamed,
                        "expected": int(ctx["upload"]["amount"]),
                    }
                },
                {
                    ctm.HTTPHeader.ACCEPT_RANGES: "bytes",
                    ctm.HTTPHeader.RESUME_AT: str(streamed),
                    ctm.HTTPHeader.CONTENT_RANGE: "bytes {}-/*".format(streamed)
                }
            )

        if not streamed and task["status"] != ctt.Status.PREPARING:
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "Task #{} of type '{}' is in '{}' status instead of '{}'".format(
                    tid, task["type"], task["status"], ctt.Status.PREPARING
                )
            )

        logger.debug("Waiting for connection target information.")
        for _ in common_itertools.progressive_yielder(.1, 3, self.UPLOAD_SOCKET_TIMEOUT, False):
            try:
                ctx = rest.task[tid].context[:]
            except common_rest.Client.HTTPError as ex:
                return self.JSONError(self.ctx, ex)

            if all(_ in ctx["upload"] for _ in ("target", "amount")):
                break
            else:
                logger.info("It seems that task's context isn't filled correctly yet.")

        try:
            target, expected = ctx["upload"]["target"], int(ctx["upload"]["amount"])
            host, _, port = target.rpartition(":")
            port = int(port)
            assert ctm.Upload.VERSION >= ctx["upload"]["version"], "incorrect upload protocol version"
            assert host, "wrong host {!r}".format(host)
        except (AssertionError, ValueError, TypeError, KeyError) as ex:
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "Invalid upload context ({}): {}".format(ctx["upload"], str(ex))
            )

        logger.info(
            "Amount of data to be uploaded: %d. Client's upload script version: %r. Connecting to %s:%s",
            expected, ctx["upload"]["version"], host, port
        )

        try:
            streamer = data_source(logger, host, port, files, streamed, expected)
            streamer.greet(ctx["upload"]["cnonce"], self.ctx.config.this.id, flask.g.req_id)
        except Exception as ex:
            return self.JSONError(
                self.ctx,
                requests.codes.SERVICE_UNAVAILABLE,
                "Unable to connect to upload target '{}': {}".format(target, str(ex))
            )

        if streamer.streamed:
            logger.info(
                "Resuming upload. Streamed already %d out of %d bytes, about %d bytes left.",
                streamer.streamed, streamer.expected, streamer.expected - streamer.streamed
            )
        else:
            logger.info("Streaming about %d bytes", streamer.expected)

        try:
            streamer.stream(ctx["upload"]["version"])
            return streamer.wait(rest, tid)

        except Exception as ex:
            logger.exception("Error streaming data to upload target.")
            if not isinstance(ex, socket.error):
                streamer.abort()
            return self.JSONError(
                self.ctx,
                requests.codes.SERVICE_UNAVAILABLE,
                "Error streaming data to upload target (streamed {} out of {} bytes): {}".format(
                    streamer.streamed, streamer.expected, str(ex)
                )
            )
        finally:
            streamer.sock.close()

    def direct_upload(self, rtype):
        if flask.request.method == "OPTIONS":
            return self.make_default_options_response()

        logger = flask.g.logger
        logger.info("Files are being uploaded via direct HTTP request to proxy")

        request_length = int(flask.request.headers.get(ctm.HTTPHeader.CONTENT_LENGTH))
        if not request_length:
            return self.JSONError(self.ctx, requests.codes.BAD_REQUEST, "No Content-Length header specified")
        elif request_length > self.DIRECT_UPLOAD_LIMIT:
            return self.JSONError(
                self.ctx,
                requests.codes.REQUEST_ENTITY_TOO_LARGE,
                "Total size of files to upload exceeds 1GB, consider REMOTE_COPY_RESOURCE task instead"
            )

        if not flask.request.form.get("description"):
            return self.JSONError(self.ctx, requests.codes.BAD_REQUEST, "No description specified")
        if not flask.request.files:
            return self.JSONError(self.ctx, requests.codes.BAD_REQUEST, "No files attached")

        auth = self.ctx.auto_auth()
        if not auth:
            return self.JSONError(
                self.ctx,
                requests.codes.UNAUTHORIZED,
                "Please provide either authorization header or cookie"
            )

        rest = self.ctx.rest(auth=auth)
        rest.DEFAULT_TIMEOUT = self.API_DEFAULT_TIMEOUT
        rest.reset()
        rest = rest << common_rest.Client.HEADERS({
            ctm.HTTPHeader.READ_PREFERENCE: ctd.ReadPreference.val2str(ctd.ReadPreference.PRIMARY)
        })
        files = []
        for field in (flask.request.files.getlist(field) for field in six.iterkeys(flask.request.files)):
            for f in field:
                contents = f.read()
                files.append(File(f.filename, contents, len(contents)))
                logger.debug("File %r of size %r obtained from HTTP request", files[-1].filename, files[-1].size)
        amount = sum(f.size for f in files)
        if amount > request_length:
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "Total size of files to upload exceeds Content-Length header value"
            )

        logger.info("Creating the upload task from scratch and filling context")
        try:
            author = rest.user.current[:]["login"]
        except Exception as ex:
            return self.JSONError(self.ctx, ex)

        # expected payload: "a=1234,b=5678,c=DEFGH IJKLM"
        attrs = flask.request.form.get("attrs") or None
        if attrs:
            attrs = {
                k.strip(): v.strip()
                for k, v in (_.split("=") for _ in attrs.split(","))
            }

        task_id = rest.task.create(
            type=self.UPLOAD_TASK_TYPES[-1],
            fail_on_any_error=True,
            custom_fields=[
                dict(name="resource_arch", value=flask.request.form.get("arch", ctm.OSFamily.ANY)),
                dict(name="resource_type", value=rtype),
                dict(name="resource_attrs", value=attrs),
            ],
            context={"upload": {
                "version": ctm.Upload.VERSION,
                "stream": ctm.Upload.Stream.PLAIN,
                "amount": amount,
            }},
            notifications=[],
            author=author,
            owner=flask.request.form.get("owner", author),
            description=flask.request.form["description"],
            priority=int(ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)),
            requirements=dict(
                # 10 Mb is reserved for disk_usage.yaml (see agentr.daemon.monitoring_setup() for that);
                # 1 Mb is kept for logs and all that, just in case.
                disk_space=amount + (11 << 20),
            ),
        )["id"]

        logger.debug("Starting the task")
        rest.batch.tasks.start.update(id=task_id, comment="Starting upload")
        logger.debug("Waiting for the task queueing.")
        for _, _ in common_itertools.progressive_yielder(.1, 3, self.UPLOAD_SOCKET_TIMEOUT, False):
            task = rest.task[task_id][:]
            if task["status"] == ctt.Status.PREPARING:
                break
            if task["status"] == ctt.Status.EXCEPTION:
                return self.JSONError(
                    self.ctx,
                    requests.codes.UNAVAILABLE,
                    "Task #{} of type '{}' is in '{}' status instead of '{}'".format(
                        task_id, task["type"], task["status"], ctt.Status.PREPARING
                    )
                )

        def premature_reply():
            yield ""
            content, return_code, headers = self._upload(logger, rest, task_id, ProxySource, files)
            yield content

        return flask.Response(
            flask.stream_with_context(premature_reply()),
            requests.codes.ACCEPTED,
            {ctm.HTTPHeader.TASK_ID: task_id}
        )

    def upload(self, tid):
        logger = flask.g.logger
        rest = self.ctx.rest()
        rest.DEFAULT_TIMEOUT = self.API_DEFAULT_TIMEOUT
        rest.reset()
        rest = rest << common_rest.Client.HEADERS({
            ctm.HTTPHeader.READ_PREFERENCE: ctd.ReadPreference.val2str(ctd.ReadPreference.PRIMARY)
        })
        logger.info("%s data upload for task #%r", ("Prepare" if flask.request.method == "HEAD" else "Handle"), tid)
        return self._upload(logger, rest, tid, UwsgiSource)

    def _mds_upload_file(self, rid, namespace, logger):
        tar_reader = common_mds.TARReader(
            common_mds.ChunkedFileObject(
                uwsgi_chunker(logger), float("inf"), hash_calculator_factory=HashCalculatorFactory()
            ),
            logger,
        )
        mds_settings = self.ctx.config.common.mds
        header, fileobj = next(iter(tar_reader))
        logger.debug(
            "Uploading file from resource #%s of size %s to MDS namespace %s",
            rid, common_format.size2str(header.size), namespace or ctr.DEFAULT_S3_BUCKET
        )
        executable = bool(header.mode & 0o111)
        try:
            mds_key, metadata = common_mds.S3().upload_file(
                fileobj,
                header.name,
                namespace=namespace,
                resource_id=rid,
                size=header.size,
                executable=executable,
                logger=logger,
                mds_settings=mds_settings,
            )
            skynet_id = common_mds.S3().skyboned_add(
                metadata, rid, namespace=namespace, logger=logger, mds_settings=mds_settings
            )
        except common_mds.MDS.Exception as ex:
            logger.error("Error occurred while uploading file to MDS: %s", ex)
            raise
        update = dict(
            file_name=header.name,
            skynet_id=skynet_id,
            size=header.size,
            state=ctr.State.READY,
            multifile=False,
            executable=executable,
            md5=fileobj.hashes.md5,
            mds=dict(
                key=mds_key,
                namespace=namespace,
            )
        )
        return update

    def _mds_upload_directory(self, rid, namespace, logger):
        logger.debug(
            "Uploading directory from resource #%s to MDS namespace %s",
            rid, namespace or ctr.DEFAULT_S3_BUCKET
        )
        mds_settings = self.ctx.config.common.mds
        try:
            collector = common_mds.TarMetaCollector(
                logger=logger, hash_calculator_factory=ft.partial(HashCalculatorFactory(), factory_id=1)
            )
            file_obj = common_mds.ChunkedFileObject(
                uwsgi_chunker(logger), float("inf"), [collector], hash_calculator_factory=HashCalculatorFactory()
            )
            mds_key, metadata = common_mds.S3().upload_directory(
                file_obj,
                namespace=namespace,
                resource_id=rid,
                logger=logger,
                mds_settings=mds_settings,
                tar_dir=True
            )
            skynet_id = common_mds.S3().skyboned_add(
                metadata, rid, namespace=namespace, logger=logger, mds_settings=mds_settings, share_fullpath=True
            )
        except common_mds.MDS.Exception as ex:
            logger.error("Error occurred while uploading directory to MDS: %s", ex)
            raise
        total_size = sum(item.size or 0 for item in metadata[2:])
        update = dict(
            file_name=file_obj.tar_name,
            skynet_id=skynet_id,
            size=total_size,
            state=ctr.State.READY,
            multifile=True,
            mds=dict(
                key=mds_key,
                namespace=namespace,
            )
        )
        return update

    def _mds_upload_tar(self, rid, namespace, compression_type, logger):
        if compression_type == common_mds_compression.base.CompressionType.TGZ:
            collector = common_mds.TgzMetaCollector(logger=logger)
        else:
            collector = common_mds.TarMetaCollector(logger=logger)
        tar_reader = common_mds.TARReader(
            common_mds.ChunkedFileObject(
                uwsgi_chunker(logger), float("inf"),
                hash_calculator_factory=HashCalculatorFactory(), logger=logger
            ),
            logger,
            handlers=[collector],
        )
        mds_settings = self.ctx.config.common.mds
        header, fileobj = next(iter(tar_reader))

        logger.debug(
            "Uploading tar from resource #%s of size %s to MDS namespace %s",
            rid, common_format.size2str(header.size), namespace or ctr.DEFAULT_S3_BUCKET
        )
        executable = bool(header.mode & 0o111)
        uncommited_key = None
        try:
            mds_key, metadata = common_mds.S3().upload_tar(
                fileobj,
                header.name,
                namespace=namespace,
                resource_id=rid,
                size=header.size,
                compression_type=compression_type,
                logger=logger,
                mds_settings=mds_settings,
            )

            uncommited_key = mds_key
            skynet_id = common_mds.S3().skyboned_add(
                metadata, rid, namespace=namespace, logger=logger, mds_settings=mds_settings
            )
            uncommited_key = None
        except common_mds.MDS.Exception as ex:
            logger.error("Error occurred while uploading directory to MDS: %s", ex)
            raise
        finally:
            if uncommited_key:
                logger.warning("Removing uploaded file after failed announcing to Skynet Skybone")
                common_mds.S3().delete(
                    uncommited_key, False, namespace=namespace, logger=logger, mds_settings=mds_settings
                )
        update = dict(
            file_name=header.name,
            skynet_id=skynet_id,
            size=header.size,
            state=ctr.State.READY,
            multifile=False,
            executable=executable,
            md5=fileobj.hashes.md5,
            mds=dict(
                key=mds_key,
                namespace=namespace
            )
        )
        return update

    def redirect(self, rid=None, rtype=None, relpath=None, user=None, without_mds=False):
        """
        Main method for processing request

        :param rid: Resource id
        :param rtype: Resource type
        :param relpath: relative path in the resource
        :param user: user login who made request
        :param without_mds: don't take into account source in MDS.
                            Currently used for browsing archives that have source in MDS
        :return: response
        """
        logger = flask.g.logger
        service_args = {"order_by": -1, "fields": "id", "limit": 1}

        args, query = {}, {}
        for k in flask.request.args:
            v = flask.request.args.getlist(k)
            (query if k in Application.FILESERVER_KNOWN_ARGS else args)[k] = ",".join(v) if len(v) > 1 else v[0]
        do_redirect = distutils.util.strtobool(args.pop("redirect", "0"))
        # TODO: SANDBOX-7154
        for k in six.viewkeys(args) - Application.KNOWN_SEARCH_FIELDS:
            args.pop(k, None)
        by_id = False
        if rid:
            by_id = not args
            args["id"] = rid
        if rtype:
            args["type"] = rtype
            args.setdefault("state", ctr.State.READY)
        if relpath:
            relpath = relpath.lstrip("/")
        args.update(service_args)

        rest = self.ctx.rest()
        rp = flask.request.headers.get(ctm.HTTPHeader.READ_PREFERENCE)
        if rp:
            rest = rest << rest.HEADERS({ctm.HTTPHeader.READ_PREFERENCE: rp})
        if not by_id:
            logger.debug("Query resources list with %r. Relpath is %r, query: %r", args, relpath, query)
            try:
                docs = self._get_resources_by_args(rest, args)
            except common_rest.Client.HTTPError as ex:
                return self.JSONError(self.ctx, ex)
            if not docs:
                return self.JSONError(self.ctx, requests.codes.NOT_FOUND, "No resources found by the query specified")
            rid = docs[0]["id"]
        else:
            logger.debug("Request by resource ID %r. Relpath is %r, query: %r", rid, relpath, query)

        if do_redirect:
            return self.JSONResponse(
                requests.codes.TEMPORARY_REDIRECT,
                {"reason": "Redirecting to resource #{}".format(rid)},
                {
                    "Location": flask.request.host_url + str(rid) + ("/" + relpath if relpath else ""),
                    ctm.HTTPHeader.RESOURCE_ID: rid
                }
            )

        try:
            req_user, resource = self._get_user_and_resource_by_id(rest, rid)
        except common_rest.Client.HTTPError as ex:
            return self.JSONError(self.ctx, ex)
        if user is None:
            user = req_user

        logger.info(
            "Creating redirect location for %r: %s resource #%s (%s). Relpath is %r, query: %r",
            user, resource["state"], resource["id"], resource["type"], relpath, query
        )

        flask.g.request_statistics.provide_resource_meta(resource)

        if banned_networks.banned_request(user, resource, logger):
            return self.JSONError(
                self.ctx, requests.codes.UNAUTHORIZED, "Unauthorized request from dynamic nets."
            )

        sources = resource["sources"]
        is_browser = "text/html" in [_.strip() for _ in flask.request.headers.get("Accept", "").split(",")]
        is_browser = is_browser or flask.request.headers.get(ctm.HTTPHeader.IGNORE_STATE)

        mds_redirect = flask.request.headers.get(ctm.HTTPHeader.MDS_REDIRECT)
        if mds_redirect:
            mds_redirect = distutils.util.strtobool(mds_redirect)
        mds = resource.get("mds")
        if mds_redirect and mds and not resource.get("multifile"):
            return self.JSONResponse(
                requests.codes.TEMPORARY_REDIRECT,
                {"reason": "Redirecting to MDS for resource #{}".format(rid)},
                {
                    "Location": common_mds.s3_link(
                        mds["key"], mds.get("namespace"), mds_settings=self.ctx.config.common.mds
                    )
                }
            )

        if resource["state"] != ctr.State.READY and not (is_browser and sources):
            task = self._get_task_by_id(rest, resource["task"]["id"])
            _TSG = ctt.Status.Group
            return (
                self.JSONError(
                    self.ctx,
                    requests.codes.GONE,
                    "Resource #{} is {} and its task #{} is {}".format(
                        resource["id"], resource["state"], task["id"], task["status"]
                    )
                )
                if task["status"] not in it.chain(_TSG.QUEUE, _TSG.EXECUTE, _TSG.WAIT) else
                self.JSONResponse(
                    requests.codes.TEMPORARY_REDIRECT,
                    {"reason": "Resource #{} is not ready yet".format(resource["id"])},
                    {
                        ctm.HTTPHeader.LOCATION: "?".join(
                            [six.ensure_str(flask.request.url), six.ensure_str(flask.request.query_string)]
                        ),
                        ctm.HTTPHeader.RESOURCE_ID: resource["id"],
                        ctm.HTTPHeader.RETRY_AFTER: int(max(task["execution"]["estimated"], 15)),
                    }
                )
            )

        rid = str(resource["id"])
        ua = flask.request.headers.get(ctm.HTTPHeader.USER_AGENT, "UNKNOWN")

        # download single empty file
        if resource["md5"] == ctr.EMPTY_FILE_MD5 and resource["size"] == 0 and not resource["multifile"]:
            headers = {
                ctm.HTTPHeader.DATA_SOURCE: "Proxy",
                ctm.HTTPHeader.RESOURCE_ID: resource["id"],
                ctm.HTTPHeader.CONTENT_LENGTH: 0,
            }
            return flask.Response("", requests.codes.OK, headers)

        mds_downloader = proxy_mds.MdsDownloader(self.ctx, self.redirect, logger)
        mds_metadata_resp = (
            None
            if without_mds else
            mds_downloader.get_metadata_from_mds(mds, rid, resource.get("multifile"))
        )
        if mds_metadata_resp is not None and not (200 <= mds_metadata_resp.status_code <= 299):
            error_code = reason = None
            if mds_metadata_resp.status_code == requests.codes.TOO_MANY_REQUESTS:
                reason = "MDS API quota exceeded"
                error_code = requests.codes.TOO_MANY_REQUESTS
            elif mds_metadata_resp.status_code == requests.codes.NOT_FOUND and resource.get("multifile"):
                reason = "Resource metadata not found for multifile resource"
                error_code = requests.codes.NOT_FOUND
            # single files do not have metadata in MDS, so it's an expected response
            elif mds_metadata_resp.status_code != requests.codes.NOT_FOUND:
                reason = mds_metadata_resp.text
                error_code = mds_metadata_resp.status_code
            if error_code:
                return self.JSONError(self.ctx, code_or_error=error_code, reason=reason)

        resource_is_archive = any(resource["file_name"].endswith(ext) for ext in self.SUPPORTED_ARCHIVE_EXTENSIONS)
        if mds_metadata_resp is None and not resource.get("multifile") and relpath and resource_is_archive:
            without_mds = True

        if mds and not without_mds and flask.request.method in (ctm.RequestMethod.GET, ctm.RequestMethod.HEAD):
            flask.g.request_statistics.provide_data_source(mds.get("namespace"))
            return mds_downloader.download_from_mds(query, resource, relpath, mds_metadata_resp)

        session, job = self._best_source(logger, rest, rid, resource["task"]["id"], resource["file_name"], sources, ua)

        if not session:
            if without_mds:
                error_template = (
                    "Resource #{} is not-indexed tar archive and have no sources on Sandbox cluster, it's not browsable"
                    if resource_is_archive else "Resource #{} is a file, it's not browsable."
                )
                return self.JSONError(
                    self.ctx,
                    requests.codes.UNPROCESSABLE_ENTITY,
                    error_template.format(rid),
                    {ctm.HTTPHeader.RESOURCE_ID: rid},
                )
            return self.JSONError(
                self.ctx,
                requests.codes.SERVICE_UNAVAILABLE,
                "Resource #{} is not available at all sources".format(rid),
                {ctm.HTTPHeader.RESOURCE_ID: rid},
            )
        flask.g.request_statistics.provide_data_source(job.fqdn)
        link = job.link
        if relpath is not None:
            if isinstance(relpath, six.text_type):
                relpath = relpath.encode("utf-8")
            link = "/".join([link, urlparse.quote(relpath.strip())])
        if query:
            link = '?'.join([link, urlparse.urlencode(query)])
        method = flask.request.method
        logger.info("Preparing %s for link %r", ("data stream" if method == ctm.RequestMethod.GET else "headers"), link)
        hdrs = dict(flask.request.headers)
        hdrs.update({
            ctm.HTTPHeader.USER_AGENT: ua,
            ctm.HTTPHeader.RESOURCE_ID: rid,
            ctm.HTTPHeader.TASK_ID: str(resource["task"]["id"]),
            ctm.HTTPHeader.FORWARDED_PATH: b"/".join(
                (six.ensure_binary(rid), six.ensure_binary(relpath))
            ) if relpath else rid
        })
        r = (session.get if method == ctm.RequestMethod.GET else session.head)(
            link,
            stream=True,
            headers=hdrs,
            timeout=proxy_common.DOWNLOAD_SOCKET_TIMEOUT
        )
        hdrs = dict(r.headers)
        hdrs.update({
            "X-Resource-Id": rid,  # TODO: SANDBOX-4928: Deprecated.
            ctm.HTTPHeader.RESOURCE_ID: rid,
            ctm.HTTPHeader.DATA_SOURCE: job.cid,
        })
        return flask.Response(
            flask.stream_with_context(proxy_common.flush_and_stream(
                logger, rt.IterableStream(r, logger), r.headers.get("transfer-encoding", "") == "chunked"
            )),
            r.status_code,
            hdrs
        )

    def mds_upload(self, rid):
        logger = flask.g.logger
        rest = self.ctx.rest()
        rest.DEFAULT_TIMEOUT = self.API_DEFAULT_TIMEOUT
        rest.reset()
        rest = rest << common_rest.Client.HEADERS({
            ctm.HTTPHeader.READ_PREFERENCE: ctd.ReadPreference.val2str(ctd.ReadPreference.PRIMARY)
        })

        admin_token = self.ctx.config.proxy.server.auth.token
        auth = common_fs.read_settings_value_from_file(admin_token, True)
        if not auth:
            logger.warning("No administrative token found (%r)", admin_token)
            return self.JSONError(self.ctx, requests.codes.METHOD_NOT_ALLOWED, "No administrative token found")

        try:
            resource = rest.resource[rid][:]
        except common_rest.Client.HTTPError as ex:
            if ex.status != requests.codes.TOO_MANY_REQUESTS:
                raise
            return self.JSONError(
                self.ctx, requests.codes.TOO_MANY_REQUESTS, "Sandbox API quota exceeded: {}".format(str(ex))
            )
        if resource["rights"] != ctu.Rights.WRITE:
            return self.JSONError(
                self.ctx, requests.codes.UNAUTHORIZED, "You have no rights to modify resource #{}".format(rid)
            )
        if resource["state"] != ctr.State.NOT_READY:
            return self.JSONError(
                self.ctx, requests.codes.BAD_REQUEST, "Resource #{} in state {}".format(rid, resource["state"])
            )
        group = common_abc.cached_sandbox_group(resource["owner"], rest_client=rest)
        bucket = group and (group["mds_quota"] or {}).get("name")
        namespace, bucket_stats = common_mds.S3.check_bucket(
            bucket=bucket, mds_settings=self.ctx.config.common.mds
        )

        flask.g.request_statistics.provide_data_source(namespace)

        free_space = bucket_stats["max_size"] - bucket_stats["used_space"]
        total_size = flask.request.headers.get(ctm.HTTPHeader.TOTAL_SIZE)

        if (
            not self.ctx.config.proxy.server.allow_ttl_inf and
            resource.get("attributes", {}).get("ttl") == "inf" and
            (not namespace or namespace == ctr.DEFAULT_S3_BUCKET) and
            resource["owner"] != self.ctx.config.common.service_group
        ):
            return self.JSONError(
                self.ctx, requests.codes.BAD_REQUEST,
                "Forbidden to upload resource {} with ttl 'inf' to common bucket {}."
                "Use mds bucket of your service for resources with 'inf' ttl or set ttl not equal 'inf'".format(
                    rid, namespace or ctr.DEFAULT_S3_BUCKET
                )
            )

        if total_size:
            try:
                total_size = int(total_size)
            except ValueError:
                total_size = None
            if total_size is not None and free_space < total_size:
                return self.JSONError(
                    self.ctx,
                    requests.codes.BAD_REQUEST,
                    "Bucket {} has insufficient free space {} to upload {}".format(
                        namespace or ctr.DEFAULT_S3_BUCKET,
                        common_format.size2str(free_space), common_format.size2str(total_size)
                    )
                )

        multifile = flask.request.headers.get(ctm.HTTPHeader.MULTIFILE)
        tar = flask.request.headers.get(ctm.HTTPHeader.TAR)

        if multifile:
            multifile = distutils.util.strtobool(multifile)
        if tar:
            try:
                tar = int(tar)
            except (TypeError, ValueError):
                tar = int(distutils.util.strtobool(tar))
        try:
            if multifile:
                update = self._mds_upload_directory(rid, namespace, logger)
            elif tar:
                update = self._mds_upload_tar(rid, namespace, tar, logger)
            else:
                update = self._mds_upload_file(rid, namespace, logger)
        except common_mds.MDS.TooManyRequests as ex:
            return self.JSONError(
                self.ctx, requests.codes.TOO_MANY_REQUESTS, "MDS api quota exceeded: {}".format(ex)
            )
        except common_mds.MDS.InsufficientSpace as ex:
            return self.JSONError(
                self.ctx, requests.codes.INSUFFICIENT_STORAGE, "Sandbox lacks of storage in MDS: {}".format(ex)
            )
        except common_mds.MDS.InsufficientBucketSpace as ex:
            help_link = "https://docs.yandex-team.ru/sandbox/resources#lack-of-storage"
            return self.JSONError(
                self.ctx,
                requests.codes.BAD_REQUEST,
                "There is insufficient space in MDS bucket, see {}: {}".format(help_link, ex)
            )
        finally:
            HashCalculatorFactory().cleanup()

        srv_rest = self.ctx.rest(auth=auth if self.ctx.config.server.auth.enabled else None)
        srv_rest.resource[rid] = {k: v for k, v in six.iteritems(update) if v is not None}
        return self.JSONResponse(
            requests.codes.OK,
            {
                "mds_url": common_mds.s3_link(
                    update["mds"]["key"], namespace=namespace, mds_settings=self.ctx.config.common.mds
                ),
                "skynet_id": update.get("skynet_id")
            }
        )

    @staticmethod
    def check():
        return "OK"

    @staticmethod
    def _collect_sensors():
        def metric(name, sensor, deriv=False):
            ret = {
                "labels": {"sensor": sensor},
                "value": uwsgi.metric_get(name),
            }
            if deriv:
                ret["mode"] = "deriv"
            return ret

        # Metrics
        yield metric("core.busy_workers", "workers_busy")
        yield metric("core.idle_workers", "workers_idle")
        yield metric("core.total_tx", "total_tx_bytes", deriv=True)

        # Response codes
        for code in it.chain(proxy_common.KNOWN_RESPONSES, ["OTHER"]):
            yield {
                "labels": {"sensor": "response_code", "code": code},
                "value": uwsgi.metric_get("http_{}".format(code)),
                "mode": "deriv",
            }

        yield {
            "labels": {"sensor": "total_requests"},
            "value": uwsgi.total_requests(),
            "mode": "deriv",
        }

    def solomon_sensors(self):
        return self.JSONResponse(
            requests.codes.OK,
            {"sensors": list(self._collect_sensors())}
        )

    def resource_links(self, link_id, relpath=None):
        link = self.ctx.rest().resource.link.update(id=link_id)
        return self.redirect(rid=link["resource_id"], relpath=relpath, user=link["author"])

    def __init__(self, debug=False):
        self.ctx = proxy_context.Context()
        banned_networks.BannedNetworks().start()
        if self.ctx.config.common.statistics.enabled:
            common_statistics.Signaler(
                common_statistics.UASignalHandlerForProxy(),
                component=ctm.Component.PROXY,
                config=self.ctx.config
            )
        self.logger.info("Initializing in %s mode. PID is %d", "debug" if debug else "production", os.getpid())
        common_mds.S3.set_s3_logging_level(logging.INFO)

        super(Application, self).__init__(self.ctx.name)

        self.debug = debug

        plugins.request_context.init_plugin(self)
        plugins.request_metrics.init_plugin(self)

        self.route("/http_check")(self.check)

        self.route("/ps/<int:tid>")(self.ps)
        self.route("/actions/<int:tid>")(self.actions)

        self.route("/task/<int:tid>")(self.taskdir)
        self.route("/task/<int:tid>/")(self.taskdir)
        self.route("/task/<int:tid>/<path:relpath>")(self.taskdir)

        self.route("/<int:rid>", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(self.redirect)
        self.route("/<int:rid>/<path:relpath>", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(self.redirect)
        self.route("/<int:rid>/", defaults={"relpath": "/"}, methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(
            self.redirect
        )
        self.route("/last/<string:rtype>", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(self.redirect)
        self.route("/last/<string:rtype>/<path:relpath>", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(
            self.redirect
        )
        self.route(
            "/last/<string:rtype>/",
            defaults={"relpath": "/"}, methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])(
            self.redirect
        )

        self.route("/upload/<int:tid>", methods=[ctm.RequestMethod.PUT, ctm.RequestMethod.HEAD])(self.upload)
        self.route("/upload/mds/<int:rid>", methods=[ctm.RequestMethod.PUT])(self.mds_upload)
        self.route(
            "/upload/<string:rtype>",
            methods=[ctm.RequestMethod.PUT, ctm.RequestMethod.POST, ctm.RequestMethod.OPTIONS])(
            self.direct_upload
        )

        self.ws = flask_uwsgi_websocket.WebSocket(self)
        self.ws.route("/shell/<int:tid>/ws")(self.shell_ws)
        self.ws.route("/tail/<int:tid>/<path:relpath>")(self.tail)
        self.ws.route("/debugger/<int:tid>")(self.debugger_ws)
        self.route("/debugger/<int:tid>", methods=[ctm.RequestMethod.POST])(self.debugger_start)
        self.route("/shell/<int:tid>")(self.shell)

        self.route("/shell/static/<string:_>")(self.shell_static)
        self.route("/shell/static/fonts/<string:_>")(self.shell_static)
        self.route("/shell/static/images/<string:_>")(self.shell_static)
        self.route("/shell/theme/<path:_>")(self.shell_static)
        self.route("/shell/themes/<path:_>")(self.shell_static)

        self.route("/solomon/sensors")(self.solomon_sensors)

        self.route("/resource/link/<string:link_id>")(self.resource_links)
        self.route("/resource/link/<string:link_id>/<path:relpath>")(self.resource_links)
