import os
import json
import string
import functools as ft

import flask
import requests

from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common import mds as common_mds
from sandbox.common import enum as common_enum
from sandbox.common import format as common_format
from sandbox.common.mds import compression as common_mds_compression

from sandbox.proxy import common as proxy_common
from sandbox.proxy import response_types as rt


class DirectoryItemType(common_enum.Enum):
    REGULAR = None
    DIRECTORY = None
    LINK = None
    UNKNOWN = None


JSON_DIR_ITEM_TYPES_MAP = {
    ctr.FileType.FILE: DirectoryItemType.REGULAR,
    ctr.FileType.DIR: DirectoryItemType.DIRECTORY,
    ctr.FileType.SYMLINK: DirectoryItemType.LINK,
}


class MdsDownloader:
    """Helps to retrieve data from MDS (download, proxy, browse content, ...)"""
    S3_REQUEST_HEADERS_WHITELIST = [
        ctm.HTTPHeader.ACCEPT,
        ctm.HTTPHeader.ACCEPT_RANGES,
        ctm.HTTPHeader.RANGE,
        ctm.HTTPHeader.TRANSFER_ENCODING,
        ctm.HTTPHeader.ACCEPT_ENCODING,
    ]

    MDS_RESOURCE_CONTENTS_TEMPLATE = """
    <!DOCTYPE html>
    <html>
      <head>
        <title>Sandbox resource #$resource_id of task #$task_id at MDS.</title>
        <style>
          body {
             font-family: monospace;
             font-size: 0.95em;
          }
          a {
            text-decoration: none;
          }
          a:hover {
            text-decoration: underline;
          }
          td {
            padding: 0px 5px 0px 5px;
            border-right: 1px dotted #aaa;
            white-space: nowrap;
          }
          .title {
             # white-space: nowrap;
          }
          .head-link {
             color: #aaa;
          }
          .file-row {
            float: left;
            width: 655px;
          }
        </style>
      </head>
      <body>
        <h3 class="title">
          Sandbox <a href="https://$base_url/resource/$resource_id/view" target="_blank">resource #$resource_id</a>
          of <a href="https://$base_url/task/$task_id/view" target="_blank">task #$task_id</a> at MDS:
        </h3>
        <table cellspacing="1" cellpadding="0">
        $items
        </table>
      </body>
    </html>
    """

    MDS_RESOURCE_DIRECTORY_ITEM_TEMPLATE = {
        ctr.FileType.DIR: """
        <tr$line_bg>
          <td><a title="$name" href="$url"><b>$short_name</b></a></td>
          <td>&mdash;</td>
          <td>&mdash;</td>
          <td><a title="Download as tarball" href="$tarball_url">download</a></td>
        </tr>
        """,
        ctr.FileType.FILE: """
        <tr$line_bg>
          <td><a title="$name" href="$url">$short_name</a></td>
          <td>$size</td>
          <td><a title="Show file as plain text" href="$url?force_text_mode=1">as text</a></td>
          <td><a title="Download as binary" href="$url?force_binary_mode=1">download</a></td>
        </tr>
        """,
        ctr.FileType.SYMLINK: """
        <tr$line_bg>
          <td><a title="$name" href="$url">@$short_name</a></td>
          <td>$size</td>
          <td><a title="Show file as plain text" href="$url?force_text_mode=1">as text</a></td>
          <td><a title="Download as binary" href="$url?force_binary_mode=1">download</a></td>
        </tr>
        """,
    }
    MDS_RESOURCE_DIRECTORY_ITEM_TEMPLATE[ctr.FileType.TOUCH] = MDS_RESOURCE_DIRECTORY_ITEM_TEMPLATE[ctr.FileType.FILE]

    def __init__(self, ctx, redirect_func, logger):
        self.redirect_func = redirect_func
        self.ctx = ctx
        self.logger = logger

    def _request_to_s3(self, *args, **kwargs) -> requests.Response:
        response = requests.get(*args, **kwargs)
        rs = flask.g.get("request_statistics")
        if rs:
            rs.on_request_to_s3(response)
        else:
            self.logger.warning("Impossible to handle request to S3: request statistics aren't initialized")
        return response

    def download_from_mds(self, query, resource, relpath, mds_metadata_resp):
        rid = resource["id"]
        mds = resource.get("mds")
        req_hdrs = {k: v for k, v in flask.request.headers.items() if k in self.S3_REQUEST_HEADERS_WHITELIST}
        stream = query.get("stream")
        force_text_mode = query.get("force_text_mode")
        force_json = query.get("force_json")
        mds_metadata = mds_metadata_resp and mds_metadata_resp.json()

        # Download from MDS
        link = common_mds.s3_link(mds["key"], mds.get("namespace"), mds_settings=self.ctx.config.common.mds)
        if resource.get("multifile") and stream in ("tar", "tgz"):
            self.logger.debug("Downloading resource #%s as %s stream", rid, stream.upper())
            file_name = "{}.tar".format(os.path.basename(relpath.rstrip("/")) if relpath else str(rid))
            if stream == "tgz":
                file_name += ".gz"
                content_type = "application/x-gzip"
            else:
                content_type = "application/x-gtar"
            resp_hdrs = self.__mds_response_headers(
                mds_metadata_resp, os.path.basename(file_name), content_type, rid, stream, query
            )
            resp_hdrs.pop(ctm.HTTPHeader.CONTENT_LENGTH, None)
            resp_hdrs.pop(ctm.HTTPHeader.CONTENT_ENCODING, None)
            response_cls = rt.ReliableTarStream if stream == "tar" else rt.ReliableTgzStream
            return flask.Response(
                flask.stream_with_context(proxy_common.flush_and_stream(
                    self.logger,
                    response_cls(
                        mds_metadata,
                        mds.get("namespace"),
                        self.ctx.config.common.mds,
                        ft.partial(self.__s3_headers, req_hdrs),
                        proxy_common.DOWNLOAD_SOCKET_TIMEOUT,
                        os.path.join(os.path.basename(resource["file_name"]), relpath) if relpath else relpath,
                        self.logger,
                        mds_metadata_resp,
                    ),
                    mds_metadata_resp.headers.get("transfer-encoding", "") == "chunked"
                )) if flask.request.method == ctm.RequestMethod.GET else "",
                mds_metadata_resp.status_code,
                resp_hdrs
            )
        elif resource.get("multifile"):
            resp_hdrs = self.__mds_response_headers(
                mds_metadata_resp,
                os.path.basename(resource["file_name"]),
                (
                    force_json and "application/json; charset=UTF-8" or
                    force_text_mode and "text/plain; charset=UTF-8"
                ),
                rid, stream, query
            )
            if len(mds_metadata) == 1 and mds_metadata[0]["type"] == ctr.FileType.TOUCH:
                resp_hdrs[ctm.HTTPHeader.CONTENT_LENGTH] = "0"
                resp_hdrs[ctm.HTTPHeader.CONTENT_TYPE] = "text/plain"
                return flask.Response("", mds_metadata_resp.status_code, resp_hdrs)
            else:
                if force_json:
                    return flask.Response(mds_metadata_resp.text, mds_metadata_resp.status_code, resp_hdrs)
                return self.__browse_mds_data(
                    link, resource, relpath, mds, mds_metadata_resp, mds_metadata[0]["type"], query, resp_hdrs, req_hdrs
                )
        elif relpath is not None:
            if not isinstance(mds_metadata, list) or mds_metadata[0]["key"] != mds["key"]:
                return self.redirect_func(rid=rid, relpath=relpath, without_mds=True)

            resp_hdrs = self.__mds_response_headers(
                mds_metadata_resp,
                os.path.basename(relpath) if relpath else os.path.basename(resource["file_name"]),
                (
                    force_json and "application/json; charset=UTF-8" or
                    force_text_mode and "text/plain; charset=UTF-8"
                ),
                rid, stream, query
            )

            if force_json:
                return flask.Response(mds_metadata_resp.text, mds_metadata_resp.status_code, resp_hdrs)

            error = common_mds.S3.get_metadata_error(mds_metadata)
            if error:
                return flask.Response(
                    error,
                    requests.codes.UNPROCESSABLE_ENTITY,
                    {
                        ctm.HTTPHeader.CONTENT_TYPE: "text/html; charset=UTF-8",
                        ctm.HTTPHeader.CONTENT_LENGTH: len(error),
                        ctm.HTTPHeader.RESOURCE_ID: rid,
                        ctm.HTTPHeader.DATA_SOURCE: "MDS",
                    }
                )

            return self.__browse_mds_data(
                link, resource, relpath, mds, mds_metadata_resp, ctr.FileType.FILE, query, resp_hdrs, req_hdrs
            )
        else:
            # Download single file
            return self.__proxying_file_from_mds(link, mds, rid, query, req_hdrs)

    def __browse_mds_data(
        self, link, resource, relpath, mds, resp, browse_type: ctr.FileType, query, resp_hdrs, req_hdrs
    ) -> flask.Response:
        rid = resource["id"]
        if browse_type in (ctr.FileType.DIR, ctr.FileType.TARDIR):
            file_name = os.path.basename(resource["file_name"])
            full_relpath = os.path.join(file_name, relpath) if relpath else file_name
        else:
            file_name = ""
            full_relpath = relpath
        full_relpath = full_relpath.rstrip("/")
        depth = full_relpath.count("/") + 1
        if not relpath and browse_type == ctr.FileType.FILE:
            depth = 0
        dir_items = []
        data = resp.json()[int(browse_type in (ctr.FileType.FILE, ctr.FileType.TARDIR)):]
        path_to_item = {item["path"]: item for item in data}

        # add missing directories
        for item in data:
            if item["type"] != ctr.FileType.DIR:
                continue
            parent_dir = os.path.dirname(item["path"])
            while parent_dir:
                if parent_dir not in path_to_item:
                    parent_item = dict(item, path=parent_dir)
                    path_to_item[parent_dir] = parent_item
                    data.append(parent_item)
                parent_dir = os.path.dirname(parent_dir)

        item = path_to_item.get(full_relpath)
        if not item:
            # Redirect to fileserver to browse
            return self.redirect_func(rid=rid, relpath=relpath, without_mds=True)

        if item["type"] == ctr.FileType.SYMLINK:
            while item["type"] == ctr.FileType.SYMLINK:
                path = os.path.normpath(os.path.join(os.path.dirname(item["path"]), item["symlink"]))
                item = path_to_item[path]
            if item["type"] == ctr.FileType.DIR:
                if browse_type in (ctr.FileType.DIR, ctr.FileType.TARDIR):
                    relpath = item["path"][len(file_name) + 1:]
                    full_relpath = os.path.join(file_name, relpath) if relpath else file_name
                else:
                    full_relpath = item["path"]
        if item["type"] == ctr.FileType.TOUCH:
            resp_hdrs[ctm.HTTPHeader.CONTENT_LENGTH] = "0"
            resp_hdrs[ctm.HTTPHeader.CONTENT_TYPE] = "text/plain"
            return flask.Response("", resp.status_code, resp_hdrs)
        if item["type"] == ctr.FileType.FILE:
            # Download single file from multi file resource
            if browse_type == ctr.FileType.DIR:
                link = common_mds.s3_link(
                    item["key"], mds.get("namespace"), mds_settings=self.ctx.config.common.mds
                )
                return self.__proxying_file_from_mds(link, mds, rid, query, req_hdrs)
            else:
                return self.__proxying_file_from_mds(
                    link, mds, rid, query, req_hdrs, relpath=relpath, offset=item["offset"],
                    size=item["size"], ctype=item.get("mime")
                )

        if not full_relpath.endswith("/"):
            full_relpath += "/"
        for item in data:
            if full_relpath != "/" and not item["path"].startswith(full_relpath):
                continue
            item_depth = item["path"].count(os.sep)
            if item_depth == depth:
                dir_items.append(item)
        dir_items.sort(key=lambda _: (_["type"] != ctr.FileType.DIR, _["path"]))

        if common_mds.MimeTypes.JSON in req_hdrs.get(ctm.HTTPHeader.ACCEPT, ""):
            return self.__json_dir_list(dir_items, resource, file_name, browse_type)

        return self.__html_dir_list(dir_items, resource, relpath, file_name, browse_type)

    def __html_dir_list(self, dir_items, resource, relpath, file_name, browse_type: ctr.FileType) -> flask.Response:
        rid = resource["id"]
        rendered_items = []
        if relpath and relpath != file_name:
            url = os.path.join(os.sep, str(rid), os.path.dirname(relpath))
            rendered_items.append(
                string.Template(self.MDS_RESOURCE_DIRECTORY_ITEM_TEMPLATE[ctr.FileType.DIR]).substitute(
                    name=os.pardir,
                    short_name=os.pardir,
                    size=common_format.size2str(0),
                    url=url,
                    tarball_url=url + "?stream=tgz",
                    line_bg=' style="background-color: #EEC"' if len(rendered_items) % 2 else "",
                )
            )
        for item in dir_items:
            name = os.path.basename(item["path"])
            url = os.path.join(
                os.sep, str(rid),
                item["path"][len(file_name) + 1 if browse_type in (ctr.FileType.DIR, ctr.FileType.TARDIR) else 0:]
            )
            rendered_items.append(
                string.Template(self.MDS_RESOURCE_DIRECTORY_ITEM_TEMPLATE[item["type"]]).substitute(
                    name=name,
                    short_name=name[:200] + "..." if len(name) > 203 else name,
                    size=common_format.size2str(item.get("size") or 0),
                    url=url,
                    tarball_url=url + "?stream=tgz",
                    line_bg=' style="background-color: #EEC"' if len(rendered_items) % 2 else "",
                )
            )
        rendered_response = string.Template(self.MDS_RESOURCE_CONTENTS_TEMPLATE).substitute(
            resource_id=rid,
            task_id=resource["task"]["id"],
            base_url=self.ctx.config.server.web.address.host,
            items="\n".join(rendered_items),
        )
        return flask.Response(
            rendered_response if flask.request.method == ctm.RequestMethod.GET else "",
            requests.codes.OK,
            {
                ctm.HTTPHeader.CONTENT_TYPE: "text/html; charset=utf-8",
                ctm.HTTPHeader.CONTENT_LENGTH: len(rendered_response),
                ctm.HTTPHeader.RESOURCE_ID: rid,
                ctm.HTTPHeader.DATA_SOURCE: "MDS",
            }
        )

    @staticmethod
    def __json_dir_list(dir_items, resource, file_name, browse_type: ctr.FileType) -> flask.Response:
        rid = resource["id"]
        json_data = {}
        proxy_base_url = f"https://{flask.request.headers[ctm.HTTPHeader.HOST]}"
        for item in dir_items:
            name = os.path.basename(item["path"])
            if not name:
                continue
            path = os.path.join(
                os.sep, str(rid),
                item["path"][len(file_name) + 1 if browse_type in (ctr.FileType.DIR, ctr.FileType.TARDIR) else 0:]
            )
            json_data[name] = {
                "url": proxy_base_url + path,
                "path": path,
                "size": item.get("size") or 0,
                "type": JSON_DIR_ITEM_TYPES_MAP.get(item["type"], DirectoryItemType.UNKNOWN),
                "time": {
                    "created": resource["time"]["created"],
                    "modified": resource["time"]["created"],
                },
            }
        json_data_str = json.dumps(json_data) if flask.request.method == ctm.RequestMethod.GET else ""
        return flask.Response(
            json_data_str,
            requests.codes.OK,
            {
                ctm.HTTPHeader.CONTENT_TYPE: "application/json; charset=utf-8",
                ctm.HTTPHeader.CONTENT_LENGTH: len(json_data_str),
                ctm.HTTPHeader.RESOURCE_ID: rid,
                ctm.HTTPHeader.DATA_SOURCE: "MDS",
            }
        )

    def __proxying_file_from_mds(
        self, mds_link, mds, rid, query, req_hdrs, relpath=None, offset=None, size=None, ctype=None
    ):
        stream = query.get("stream")
        force_text_mode = query.get("force_text_mode")
        index_link = common_mds.s3_link(
            mds["key"] + ".index", mds.get("namespace"), mds_settings=self.ctx.config.common.mds
        )
        self.logger.debug("Downloading index for resource #%s at %s", rid, index_link)
        resp = self._request_to_s3(
            index_link,
            headers=self.__s3_headers({}, index_link),
            timeout=proxy_common.DOWNLOAD_SOCKET_TIMEOUT,
        )
        compressed_index = (
            common_mds_compression.base.Index.load(resp.content)
            if resp.status_code == requests.codes.OK else
            None
        )

        self.logger.debug("Downloading file of resource #%s from %s", rid, mds_link)

        point = None
        if offset is not None:
            if compressed_index is None:
                req_hdrs["Range"] = "bytes={}-{}".format(offset, offset + size - 1)
            else:
                point = compressed_index.get_point(offset)
                req_hdrs["Range"] = "bytes={}-".format(point.comp_offset)

        requests_method = requests.get if flask.request.method == ctm.RequestMethod.GET else requests.head
        req_hdrs.setdefault(ctm.HTTPHeader.ACCEPT_ENCODING, "identity")
        mds_resp = requests_method(
            mds_link,
            stream=True,
            headers=self.__s3_headers(req_hdrs, mds_link),
            timeout=proxy_common.DOWNLOAD_SOCKET_TIMEOUT
        )
        orig_raw_stream_func = mds_resp.raw.stream

        def raw_stream_func(*args, **kws):
            kws["decode_content"] = False
            return orig_raw_stream_func(*args, **kws)

        mds_resp.raw.stream = raw_stream_func
        if relpath:
            fname = os.path.basename(relpath)
        else:
            fname = os.path.basename(mds_link)

        if mds_resp.status_code == 206:
            mds_resp.status_code = 200
            del mds_resp.headers[ctm.HTTPHeader.CONTENT_RANGE]

        response_hdrs = self.__mds_response_headers(
            mds_resp, fname, ("text/plain; charset=UTF-8" if force_text_mode else ctype), rid, stream, query
        )
        response = rt.ReliableStreamFromMDS(
            mds_resp,
            proxy_common.DOWNLOAD_SOCKET_TIMEOUT,
            self.logger,
        )
        if compressed_index is not None and offset is not None:
            response = compressed_index.extract(
                common_mds.ChunkedFileObject(iter(response), float("inf"), hash_calculator_factory=None),
                point, offset, size
            )
            response_hdrs[ctm.HTTPHeader.CONTENT_LENGTH] = str(size)
        return rt.FlaskResponseFromMDS(
            flask.stream_with_context(proxy_common.flush_and_stream(
                self.logger, response, mds_resp.headers.get("transfer-encoding", "") == "chunked"
            )) if flask.request.method == ctm.RequestMethod.GET else "",
            mds_resp.status_code,
            response_hdrs,
        )

    def get_metadata_from_mds(self, mds, rid, multifile):
        if not mds:
            return None
        metadata_link = common_mds.s3_link(
            str(rid), mds.get("namespace"), mds_settings=self.ctx.config.common.mds
        )
        if multifile:
            self.logger.debug("Downloading metadata for resource #%s from %s", rid, metadata_link)
            mds_metadata_resp = self._request_to_s3(
                metadata_link,
                headers=self.__s3_headers({}, metadata_link),
                timeout=proxy_common.DOWNLOAD_SOCKET_TIMEOUT,
            )
            if mds_metadata_resp.status_code == requests.codes.OK:
                return mds_metadata_resp
            self.logger.debug("Multifile resource #%s uses old storage schema", rid)
            metadata_link = common_mds.s3_link(
                mds["key"], mds.get("namespace"), mds_settings=self.ctx.config.common.mds
            )
        self.logger.debug("Downloading metadata for resource #%s from %s", rid, metadata_link)
        mds_metadata_resp = self._request_to_s3(
            metadata_link,
            headers=self.__s3_headers({}, metadata_link),
            timeout=proxy_common.DOWNLOAD_SOCKET_TIMEOUT,
        )
        return mds_metadata_resp

    def __s3_headers(self, headers, url):
        # FIXME: temporary turn off authorization [SANDBOX-8184]
        # s3_credentials = self.__s3_credentials
        # if not s3_credentials:
        #     return headers
        # amz_date = dt.datetime.utcnow().strftime("%a, %d %b %Y %H:%M:%S GMT")
        # data = "GET\n\n\n\nx-amz-date:{}\n{}".format(amz_date, urlparse.urlparse(url).path)
        # signature = base64.b64encode(hmac.new(s3_credentials["aws_secret_access_key"], data, hashlib.sha1).digest())
        # headers = dict(headers)
        # headers.update({
        #     "x-amz-date": amz_date,
        #     ctm.HTTPHeader.AUTHORIZATION: "AWS {}:{}".format(s3_credentials["aws_access_key_id"], signature)
        # })
        return headers

    @staticmethod
    def __mds_response_headers(response, fname, ctype, rid, stream, query):
        rhdrs = dict(response.headers)
        rhdrs.update({
            ctm.HTTPHeader.RESOURCE_ID: rid,
            ctm.HTTPHeader.DATA_SOURCE: "MDS",
            ctm.HTTPHeader.CONTENT_DISPOSITION: '{}; filename="{}"'.format(
                (
                    "attachment"
                    if stream or query.get("force_binary_mode") else
                    "inline"
                ),
                fname
            ),
        })
        if ctype is not None:
            rhdrs[ctm.HTTPHeader.CONTENT_TYPE] = ctype or common_mds.MimeTypes.DEFAULT_TYPE
        else:
            ctype = rhdrs.get(ctm.HTTPHeader.CONTENT_TYPE).strip()
        if ctype in ("text/plain", "text/html"):
            rhdrs[ctm.HTTPHeader.CONTENT_TYPE] += "; charset=UTF-8"
        return rhdrs
