import time
import msgpack

try:
    import uwsgi
except ImportError:
    uwsgi = None

from sandbox import common

from sandbox.proxy import config
from sandbox.proxy import context

__all__ = ("run_safe",)


class ClientCache(object):
    CACHE_SLEEP_INTERVAL = 20  # Update interval in seconds

    def __init__(self, ctx):
        self.limit = 10000
        self.ctx = ctx

    @common.patterns.singleton_classproperty
    def client(cls):
        auth = common.fs.read_settings_value_from_file(config.Registry().proxy.server.auth.token, True)
        return common.rest.Client(config.Registry().client.rest_url, auth=auth)

    @classmethod
    def add_client(cls, client):
        cache_cfg = config.Registry().proxy.server.cache.client
        uwsgi.cache_update(
            client["id"],
            msgpack.packb(client, use_bin_type=True),
            cache_cfg.expires,
            cache_cfg.name,
        )

    @classmethod
    def get_client(cls, client_id):
        value = uwsgi.cache_get(client_id, config.Registry().proxy.server.cache.client.name)
        if value:
            value = msgpack.unpackb(value)
        return value

    def update_cache_loop(self):
        try:
            while True:
                self.ctx.logger.info("Update clients cache")
                data = self.client.client.read(
                    limit=self.limit,
                    fields=["id", "fileserver", "dc", "alive", "tags", "fqdn"],
                )
                self.limit = data["total"]
                for client in data["items"]:
                    self.add_client(client)
                time.sleep(self.CACHE_SLEEP_INTERVAL)
        except:
            self.ctx.logger.exception("Got exception in ClientCache mule!")
            raise


def run_safe():
    cache = ClientCache(context.Context())
    cache.update_cache_loop()
