"""
Plugin creates request context
"""

import os
import uuid
import socket

import flask

try:
    import uwsgi
except ImportError:
    uwsgi = None

from sandbox.common.types import misc as ctm
from sandbox.common.types import user as ctu

from sandbox.proxy import common as proxy_common


def _get_request_remote_ip():
    # First try to get remote IP from X-Forwarded-For and alike headers
    for addr in reversed(flask.request.access_route):
        if addr == "0.0.0.0":
            continue
        return addr

    # Determine peer's address via request's file descriptor number
    try:
        fd = flask.request.environ["wsgi.input"].fileno()
        sock = socket.fromfd(fd, socket.AF_INET6, socket.SOCK_STREAM)
        return sock.getpeername()[0]
    except Exception:
        pass  # Normal situation for batch subrequests where `wsgi.input` is simulated

    # Fallback for flask default
    return flask.request.remote_addr


def _remove_ipv6_prefix(addr):
    if addr.startswith("::ffff:"):
        return addr[7:]
    return addr


def _on_request_started(sender, **extra):
    req_id = flask.request.headers.get(ctm.HTTPHeader.REQUEST_ID) or ("_" + uuid.uuid4().hex[1:])
    remote_ip = _remove_ipv6_prefix(_get_request_remote_ip())
    logger = proxy_common.RequestAdapter(flask.current_app.ctx.logger, req_id)

    flask.g.req_id = req_id
    flask.g.logger = logger
    flask.g.remote_ip = remote_ip
    flask.g.user = ctu.ANONYMOUS_LOGIN

    logger.debug("Handling request from %s by worker %d (pid %d)", remote_ip, uwsgi.worker_id(), os.getpid())

    uwsgi.set_logvar("login", flask.g.user)
    uwsgi.set_logvar("req_id", str(req_id)[:8])
    uwsgi.set_logvar("remote_ip", remote_ip)


def _on_request_finished(sender, response, **extra):
    """
    :type response: flask.Response
    """
    response.headers[ctm.HTTPHeader.INT_REQUEST_ID] = flask.g.req_id
    response.headers[ctm.HTTPHeader.BACKEND_NODE] = flask.current_app.ctx.config.this.id

    # allow for cross-origin resource sharing
    origin = flask.request.environ.get("HTTP_ORIGIN", "*")
    response.headers["Access-Control-Allow-Origin"] = origin
    if origin != "*":
        response.headers["Access-Control-Allow-Credentials"] = "true"
    headers = flask.request.headers.get("Access-Control-Request-Headers")
    if headers:
        response.headers["Access-Control-Allow-Headers"] = headers

    if response.status_code in proxy_common.KNOWN_RESPONSES:
        uwsgi.metric_inc("http_{}".format(response.status_code))
    else:
        uwsgi.metric_inc("http_OTHER")


def _on_request_exception(sender, exception, **extra):
    uwsgi.metric_inc("http_500")


def init_plugin(app: flask.Flask):
    flask.request_started.connect(_on_request_started, app)
    flask.request_finished.connect(_on_request_finished, app)
    flask.got_request_exception.connect(_on_request_exception, app)


def provide_user(user_name: str):
    if user_name is None or user_name == ctu.ANONYMOUS_LOGIN:
        return

    current_user = flask.g.get("user")
    if user_name != current_user:
        if current_user == ctu.ANONYMOUS_LOGIN:
            flask.g.user = user_name
            uwsgi.set_logvar("login", user_name)
        else:
            flask.g.logger.warning("Provided new non-anonymous user %s, current user is %s", user_name, current_user)
