from __future__ import absolute_import

import os
import re
import time
import httplib

from sandbox import common

from rem_to_step import config
from rem_to_step import distributed


class TagEvent(common.utils.Enum):
    UNSET = 0
    SET = 1
    RESET = 2


class Translator(object):
    def __init__(self):
        log = self.config.log
        self.__logger = common.log.setup_log(os.path.join(log.root, log.name), log.level)
        self.__node = distributed.Node(logger=self.__logger, **self.config.distributed)
        step_token = open(self.config.step.token).read().strip()
        self.__step = common.rest.Client(
            base_url=self.config.step.url,
            auth=common.proxy.OAuth(step_token),
            logger=self.__logger,
            total_wait=self.config.step.timeout
        )
        self.init_step_events()

    def __event_name(self, name):
        return "{}_{}".format(self.config.step.event_prefix, name)

    def init_step_events(self):
        self.__logger.info("Creating events' descriptions")
        for name, pattern in self.tracked_tags.iteritems():
            event_name = self.__event_name(name)
            try:
                self.__step.event_desc[event_name][:]
            except common.rest.Client.HTTPError as ex:
                if ex.status == httplib.NOT_FOUND:
                    self.__step.event_desc(
                        name=event_name,
                        required_fields=list(common.utils.chain(("tag_name", "tag_op"), pattern.groupindex)),
                        description="REM tag {}".format(event_name),
                        public=False
                    )

    @common.utils.singleton_property
    def config(self):
        return config.Registry().rem_to_step

    @common.utils.singleton_property
    def yt(self):
        # noinspection PyUnresolvedReferences
        import yt.wrapper as yt
        yt.config["proxy"]["url"] = self.config.yt.cluster
        return yt

    @common.utils.singleton_property
    def tracked_tags(self):
        return {item["name"]: re.compile(item["pattern"]) for item in self.config.tracked_tags}

    def start(self):
        self.__node.start()

    def loop(self):
        self.__logger.info("Starting main loop")
        while True:
            if self.__node.role != self.__node.LEADER:
                time.sleep(self.config.distributed.election_timeout)
                continue
            start_version = self.__node.version
            try:
                journal = list(self.yt.select_rows(
                    "* FROM [{}] WHERE global_version > {} ORDER BY global_version LIMIT 10000".format(
                        self.config.yt.table_journal, start_version
                    )
                ))
            except Exception as ex:
                self.__logger.error("Error occurred while querying new records: %s", ex)
            else:
                self.__logger.info("Got %s record(s) greater then %s", len(journal), start_version)
                for record in journal:
                    global_version = record["global_version"]
                    tag_name = record["tag_name"]
                    tag_event = record["event"]
                    try:
                        for name, pattern in self.tracked_tags.iteritems():
                            m = pattern.match(tag_name)
                            if not m:
                                continue
                            self.__node.version = global_version
                            try:
                                event_name = self.__event_name(name)
                                event_params = m.groupdict()
                                # noinspection PyUnresolvedReferences
                                event_params.update(tag_name=tag_name, tag_event=TagEvent.val2str(tag_event))
                                self.__logger.info(
                                    "[%s] Sending event %s with parameters %s",
                                    global_version, event_name, event_params
                                )
                                for retry in xrange(self.config.step.retries.max):
                                    if retry:
                                        self.__logger.info("Retrying(%s)...", retry)
                                    try:
                                        self.__step.events(events=[dict(name=event_name, params=event_params)])
                                        break
                                    except (self.__step.TimeoutExceeded, self.__step.HTTPError) as ex:
                                        self.__logger.warning("Error while sending event to STEP: %s", ex)
                                        time.sleep(self.config.step.retries.interval)
                                else:
                                    raise
                                self.__logger.info("[%s] Event %s sent successfully", global_version, event_name)
                            except Exception as ex:
                                self.__logger.error("Error while sending event to STEP: %s", ex)
                            break
                        else:
                            self.__node.version = global_version
                    except self.__node.NotLeader:
                        break

            time.sleep(3)


def main():
    translator = Translator()
    translator.start()
    translator.loop()
