# -*- coding: utf-8 -*-

from __future__ import absolute_import, unicode_literals

import os
import time
import logging
import socket

from sandbox.sandboxsdk.process import run_process

logger = logging.getLogger(__name__)


def get_port_info(port):
    """
        Вызывать lsof/sockstat для указанного порта.
        Работает для linux/freebsd.
        Если OS не linux или freebsd, то ничего не происходит.

        :param port: номер порта
        :return: путь до файла с логом; None, если ничего не было вызвано
    """
    arch = os.uname()[0].lower()
    if arch in ('linux', 'darwin'):
        cmd = ['lsof', '-i', ':{0}'.format(port)]
    elif arch == 'freebsd':
        cmd = ['sockstat', '-4', '|', 'grep', str(port)]
    else:
        cmd = None
    if cmd:
        process = run_process(
            cmd, shell=True, log_prefix='info_port_{0}'.format(port),
            check=False, wait=False
        )
        return process.stdout_path
    else:
        return None


def is_port_free(port, host='localhost', print_info=False):
    """
        Внимание!!! порт должен быть из диапазона локальных портов (как правило port < 32768), иначе при проверке
        порта может возникнуть проблема self TCP connection https://nda.ya.ru/3ThqLT

        Проверить, свободен ли порт port на хосте host

        :param port: порт в виде числа или строки
        :param host: назвнаие хоста
        :param print_info: запускать netstat/sockstat для указанного порта
            делает вывод для текущего хоста
        :return: True, если порт свободен; в противном случае возвращает False
    """
    logger.info('Check port %s for host %s', port, host)
    if print_info:
        get_port_info(port)
    try:
        sock = socket.create_connection((host, int(port)), timeout=5)
        sock.close()
    except socket.timeout:
        logger.error(
            'socket.timeout for port {0}, host {1}'.format(port, host)
        )
        return False
    except socket.error:
        return True
    return False


def wait_port_is_free(port, host='localhost', timeout=10, check_interval=1):
    """
        Подождать, пока порт будет свободен

        :param port: проверяемый порт
        :param host: проверяемый хост
        :param timeout: сколько ждать (в секундах)
        :param check_interval: через какой промежуток времени проверять (в секундах)
        :return: True, если порт освободился, False в противном случае
    """
    start_time = time.time()
    while (time.time() - start_time) < timeout:
        if is_port_free(port, host):
            return True
        time.sleep(check_interval)
    return False
