# -*- coding: utf-8 -*-
"""
    !!! THIS MODULE IS DEPRECATED, USE `sdk2.helpers.process` INSTEAD !!!
"""

import codecs
import errno
import itertools
import json
import logging
import os
import shlex
import signal
import subprocess
import threading
import time
import traceback

import six

logger = logging.getLogger(__name__)

from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common.types import misc as ctm

from . import paths
from . import errors
from . import channel


# FIXME: remove usage of this from tasks
SandboxSubprocessError = errors.SandboxSubprocessError


class _FakeActionContext(object):
    def __enter__(self):
        pass

    def __exit__(self, *args, **kw):
        pass


def __check_subprocess_timeout(process, timeout, on_timeout=None, timeout_sleep=.5):
    """
        Ждёт завершения процесса timeout секунд, проверяя process каждые sleep секунд
        Если указан параметр on_timeout, то при срабатывании таймаута вызывается on_timeout(process)

        :param process: объект процесса, который нужно ожидать
        :param timeout: сколько ожидать в секундах
        :param on_timeout: объект функции, которую нужно вызвать,
                           если процесс не завершился после истечения срока таймаута
        :param timeout_sleep: сколько ждать между проверками таймаута
        :return: True, если процесс успел завершиться, False, если сработал таймаут

    """
    start_time = time.time()

    context = _FakeActionContext()
    if channel.channel.task:
        context = channel.channel.task.current_action(
            'waiting process "%s" (%s seconds timeout)' % (process.saved_cmd, timeout))

    with context:
        while process.poll() is None:
            if (time.time() - start_time) > timeout:
                try:
                    if on_timeout:
                        on_timeout(process)
                finally:
                    try:
                        process.terminate()
                        if process.poll() is None:
                            time.sleep(timeout_sleep)
                            process.kill()
                    except OSError:
                        pass
                return False
            time.sleep(timeout_sleep)
    return True


def check_binary_arch(binary_name):
    """
        Checks binary specified is suitable to run on the current platform.
    """
    try:
        if not os.path.isfile(binary_name):
            return True

        stat_info = os.stat(binary_name)
        if stat_info.st_size < 2048:
            # real ELF should be bigger
            return True

        with open(binary_name, "rb") as f:
            f.seek(1)
            elf = f.read(3)
            if elf != "ELF":
                return True

            f.seek(7)
            abi_sig = f.read(1)
            os_family = common_config.Registry().this.system.family
            logging.debug("ABI signature: %r, system info: %s", abi_sig, os_family)

            if abi_sig == "\x09" and os_family != ctm.OSFamily.FREEBSD:
                # trying to run FreeBSD binary on something else
                return False

            # abi_sig == "\x03" was excluded as some BSD-Linux compatible binaries
            # have this value (/Berkanavt/skynet/bin/gosky)

            if abi_sig == "\x00" and os_family not in (ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM):
                # trying to run Linux binary on something else
                return False

        return True
    except Exception:
        logging.exception("Problem in check_binary_arch")
        return True


def run_process(
    cmd, log_prefix=None, wait=True, timeout=None, on_timeout=None,
    outs_to_pipe=False, stdin=None, stdout=None, stderr=None, work_dir=None,
    check=True, shell=False, environment=None, outputs_to_one_file=True,
    add_pid_to_task_ctx=True, timeout_sleep=.5, close_fds=False, set_current_action=True,
    exc_class=errors.SandboxSubprocessError, preexec_fn=None,
    time_to_kill=None, unique_file_limit=10000
):
    """
        Запустить подпроцесс. Является обёрткой для subprocess.Popen

        При использовании одновременно wait=True и outs_to_pipe=True возможно зависание процесса, если
            в stderr и stdout пишется много информации и переполняется буфер. Если вы ожидаете, что
            возникнет такая ситуация, используйте запись в файлы.

        :param cmd: список параметров в виде списка. Доступен в возвращаемом объекте через поле saved_cmd.
        :type cmd: list,string
        :param log_prefix: префикс для записи stdout и stderr в лог-файлы.
                           Используется, если не указано outs_to_pipe=True.
                           Если равен None, то запись в файлы не происходит.
                           Пути до файлов сохраняются в полях stdout_path и stderr_path возвращаемого объекта
        :type log_prefix: str
        :param wait: ждать или нет выполнения процесса
        :type wait: bool
        :param timeout: ждать выполнения указанное количество секунд.
                        При срабатывании таймаута кидается иключение SandboxSubprocessTimeoutError.
        :type timeout: int
        :param timeout_sleep: сколько ждать между проверками при указании таймаута
        :type timeout_sleep: int
        :param outs_to_pipe: направить stdout и stderr в subprocess.PIPE,
                             при этом log_prefix, stdout и stderr игнорируются.
                             Если параметр установлен в True и процесс пишет много информации в stderr и stdout,
                             процесс может зависнуть.
        :type outs_to_pipe: bool
        :param stdin: использовать указанный stdin
        :param stdout: использовать указанный stdout
        :param stderr: использовать указанный stderr (не работает, если outputs_to_one_file=True)
        :param work_dir: указать рабочую директорию для процесса, папка должна существовать
        :type work_dir: str
        :param outputs_to_one_file: писать stderr и stdout в один файл
        :type outputs_to_one_file: bool
        :param check: проверять ли код возврата процесса
        :param shell: запускать ли как shell команду
        :type shell: bool
        :param on_timeout: фукнция, вызываемая при срабатывании таймаута
                           (в функцию передаётся один параметр - объект процесса)
        :type on_timeout: function
        :param environment: словарь с переменными окружения для запуска процесса
        :type environment: dict
        :param add_pid_to_task_ctx: add new process pid to task.ctx
        :type add_pid_to_task_ctx: bool
        :param close_fds: flags that stdout and stderr file descriptors should be closed after process exit
        :param preexec_fn: if it is set to a callable object, this object will be called in the child process
                           just before the child is executed
        :param time_to_kill: if not empty, send SIGTERM and wait this many seconds
                             before sending SIGKILL when terminating the process
        :param set_current_action: flags that current action should be reported
        :param exc_class: exception class
        :return: объект запущенного процесса (subprocess.Popen)
    """
    import sandbox.sdk2.helpers

    if not cmd:
        raise common_errors.TaskError("cmd parameter is empty")
    cmd_string = None
    if isinstance(cmd, six.string_types):
        if shell:
            cmd_string = cmd
        cmd = shlex.split(cmd)
    if not isinstance(cmd, (list, tuple)):
        raise common_errors.TaskError("Incorrect cmd parameter type {}".format(type(cmd)))
    if work_dir:
        work_dir = os.path.abspath(work_dir)
        if not os.path.isdir(work_dir):
            raise common_errors.TaskError("Work dir for subprocess is not a folder. Path {}".format(work_dir))
        if not os.path.exists(work_dir):
            raise common_errors.TaskError("Work dir for subprocess does not exist. Path {}".format(work_dir))
    if environment:
        if not isinstance(environment, dict):
            raise common_errors.TaskError("Incorrect env parameters {}".format(environment))
    else:
        environment = os.environ.copy()
        environment["LD_LIBRARY_PATH"] = ""

        task = channel.channel.task
        if task:
            dll_path = os.pathsep.join(
                itertools.chain.from_iterable(env.os_env.dll_path for env in task.environment or [])
            )
            if dll_path:
                arch = common_config.Registry().this.system.family
                if arch == ctm.OSFamily.LINUX:
                    environment['LD_LIBRARY_PATH'] = dll_path
                else:
                    logger.warning('Unknown arch %s. Cant set OS environment', arch)

    if 'LC_ALL' not in environment:
        environment['LC_ALL'] = 'C'
    open_log_files = []
    stdout_path = None
    stderr_path = None
    stdout_path_filename = None
    stderr_path_filename = None
    if outs_to_pipe:
        stdout = subprocess.PIPE
        stderr = subprocess.PIPE
    else:
        if log_prefix:
            log_prefix_name = str(log_prefix)
            logs_folder = paths.get_logs_folder()

            try:
                stdout_path_filename = '{0}.out.txt'.format(log_prefix_name)
                stdout_path = paths.get_unique_file_name(logs_folder, stdout_path_filename, limit=unique_file_limit)
                stdout_path_filename = os.path.basename(stdout_path)
                if stdout is None:
                    stdout = open(stdout_path, 'w')
                    open_log_files.append(stdout)
                if outputs_to_one_file:
                    stderr = subprocess.STDOUT
                else:
                    if stderr is None:
                        stderr_path_filename = '{0}.err.txt'.format(log_prefix_name)
                        stderr_path = paths.get_unique_file_name(
                            logs_folder, stderr_path_filename, limit=unique_file_limit
                        )
                        stderr_path_filename = os.path.basename(stderr_path)
                        stderr = open(stderr_path, 'w')
                        open_log_files.append(stderr)
            except IOError as error:
                for file_object in open_log_files:
                    file_object.close()
                raise common_errors.TaskError(
                    "Cannot open log files. Stdout file: {}. Stderr file: {}. Error: {}".format(
                        stdout_path, stderr_path, error
                    )
                )

    if cmd_string is None:
        cmd_string = ' '.join(map(str, cmd))
    if shell:
        cmd = cmd_string
    # get some uuid for denoting process
    process_uuid = codecs.getencoder('hex')(os.urandom(8))[0]
    logger.info('Run subprocess [%s]: %r', process_uuid, cmd)

    if not check_binary_arch(cmd[0]):
        raise exc_class(
            "System arch mismatch. It seems you're trying to execute Linux binary on FreeBSD system (or vice versa).\n"
            "Command was: \"{0}\"".format(cmd_string)
        )
    # Only strings allowed in env on windows
    for s in environment:
        environment[s] = str(environment[s])

    logging.debug('Trying to execute cmd: %s', cmd)

    process = subprocess.Popen(
        cmd,
        stdout=stdout,
        stderr=stderr,
        stdin=stdin,
        shell=bool(shell),
        env=environment,
        cwd=work_dir,
        close_fds=close_fds,
        preexec_fn=preexec_fn
    )
    process.timeout = timeout
    process.saved_cmd = cmd_string
    process.stdout_path = stdout_path
    process.stderr_path = stderr_path
    process.stderr_path_filename = stderr_path_filename
    process.stdout_path_filename = stdout_path_filename
    process.time_to_kill = time_to_kill
    if stdout_path is not None:
        logger.info("Subprocess [%s] with pid %d outputs to '%s'", process_uuid, process.pid, stdout_path)
    if stderr_path is not None:
        logger.info("Subprocess [%s] with pid %d outputs errors to '%s'", process_uuid, process.pid, stderr_path)

    task = channel.channel.task
    if add_pid_to_task_ctx and task and hasattr(task, "_subproc_list"):
        # добавляем процесс в список _subproc_list чтобы убить его в конце исполнения задачи
        process.rpid = sandbox.sdk2.helpers.ProcessRegistry.register(process.pid, cmd_string)
        task._subproc_list.append(process)

    if timeout or wait:
        try:
            if timeout:
                check_process_timeout(process, timeout, on_timeout, timeout_sleep)
            elif wait:
                context = _FakeActionContext()
                if set_current_action and task:
                    context = sandbox.sdk2.helpers.ProgressMeter(
                        "Executing [{}] \"{}\"".format(process_uuid, process.saved_cmd)
                    )
                with context:
                    process.wait()
        finally:
            for file_object in open_log_files:
                file_object.close()

    if wait and check:
        check_process_return_code(process, exc_class=exc_class)

    return process


def check_process_timeout(process, timeout, on_timeout=None, timeout_sleep=.5):
    """
        Ждёт завершения процесса timeout секунд, проверяя process каждые timeout_sleep секунд
        При срабатывании таймаута кидается иключение SandboxSubprocessTimeoutError

        :param process: объект процесса, который нужно ожидать
        :param timeout: сколько ожидать в секундах
        :param timeout_sleep: сколько ждать между проверками таймаута
    """
    logger.info('Wait process {0} {1} seconds'.format(process.pid, timeout))

    timeout_result = __check_subprocess_timeout(process, timeout, on_timeout, timeout_sleep=timeout_sleep)

    if not timeout_result:
        # сработал таймаут, убиваем процесс и вызываем исключение
        raise errors.SandboxSubprocessTimeoutError(
            message='Process {1} was interrupted by timeout {0}'.format(timeout, process.saved_cmd),
            cmd_string=process.saved_cmd,
            returncode=process.returncode,
            logs_resource=channel.channel.task._log_resource.id,
            stderr_path=process.stderr_path_filename,
            stdout_path=process.stdout_path_filename
        )


def check_process_return_code(process, exc_class=errors.SandboxSubprocessError):
    """
    Checks status code of the process,
    if status is non zero raise exception of the type `exc_class`

    :param process: process object
    :param exc_class: exception class
    """
    if process.returncode:
        raise exc_class(
            message='process "{0}" died with exit code {1}'.format(process.saved_cmd, process.returncode),
            cmd_string=process.saved_cmd,
            returncode=process.returncode,
            logs_resource=getattr(getattr(channel.channel.task, '_log_resource', None), 'id', None),
            stderr_path=process.stderr_path_filename,
            stdout_path=process.stdout_path_filename,
            stderr_full_path=process.stderr_path,
            stdout_full_path=process.stdout_path
        )


def throw_subprocess_error(process):
    """
        Ругаемся и ставим статус таска в FAILED, вызывая исключение SandboxSubprocessError

        :param process: объект процесса
    """
    if hasattr(process, 'saved_cmd'):
        msg = "Process (PID: {0}) '{1}' died with exit code {2}".format(process.pid,
                                                                        process.saved_cmd,
                                                                        process.returncode)
    else:
        msg = "Process (PID: {0}) 'unknown' died with exit code {1}".format(process.pid,
                                                                            process.returncode)
    # check syslog
    data, _ = subprocess.Popen('cat /var/log/messages | grep "pid {0}"'.format(process.pid), shell=True,
                               stdout=subprocess.PIPE).communicate()

    if data:
        msg += "\n=== syslog messages for {0} ===\n{1}".format(process.pid, data)

    raise errors.SandboxSubprocessError(msg, returncode=process.returncode)


def get_process_info(pid, parameters=None, ignore_errors=True):
    """
        .. warning:: **DEPRECATED**, use psutil instead

        Получить информацию о процессе (по команде ps)

        Если указано ignore_errors == True, то при ошибке возвращается пустой словарь;
            в противном случае пробрасывается исключение.

        Нужные параметры для ключа -o можно указать с помощью parameters

             %cpu    percentage cpu usage (alias pcpu)

             %mem    percentage memory usage (alias pmem)

             acflag    accounting flag (alias acflg)

             args    command and arguments

             comm    command

             command    command and arguments

             cpu    short-term cpu usage factor (for scheduling)

             etime    elapsed running time

             flags    the process flags, in hexadecimal (alias f)

             inblk    total blocks read (alias inblock)

             jobc    job control count

             ktrace    tracing flags

             label    MAC label

             lim    memoryuse limit

             logname    login name of user who started the process

             lstart    time started

             majflt    total page faults

             minflt    total page reclaims

             msgrcv    total messages received (reads from pipes/sockets)

             msgsnd    total messages sent (writes on pipes/sockets)

             lockname    lock currently blocked on (as a symbolic name)

             mwchan    wait channel or lock currently blocked on

             nice    nice value (alias ni)

             nivcsw    total involuntary context switches

             nsigs    total signals taken (alias nsignals)

             nswap    total swaps in/out

             nvcsw    total voluntary context switches

             nwchan    wait channel (as an address)

             oublk    total blocks written (alias oublock)

             paddr    swap address

             pagein    pageins (same as majflt)

             pgid    process group number

             pid    process ID

             poip    pageouts in progress

             ppid    parent process ID

             pri    scheduling priority

             re     core residency time (in seconds; 127 = infinity)

             rgid    real group ID

             rgroup    group name (from rgid)

             rlink    reverse link on run queue, or 0

             rss    resident set size

             rtprio    realtime priority (101 = not a realtime process)

             ruid    real user ID

             ruser    user name (from ruid)

             sid    session ID

             sig    pending signals (alias pending)

             sigcatch    caught signals (alias caught)

             sigignore    ignored signals (alias ignored)

             sigmask    blocked signals (alias blocked)

             sl     sleep time (in seconds; 127 = infinity)

             start    time started

             state    symbolic process state (alias stat)

             svgid    saved gid from a setgid executable

             svuid    saved uid from a setuid executable

             tdev    control terminal device number

             time    accumulated cpu time, user + system (alias cputime)

             tpgid    control terminal process group ID

             tsid    control terminal session ID

             tsiz    text size (in Kbytes)

             tt     control terminal name (two letter abbreviation)

             tty    full name of control terminal

             uprocp    process pointer

             ucomm    name to be used for accounting

             uid    effective user ID

             upr    scheduling priority on return from system call (alias usrpri)

             user    user name (from uid)

             vsz    virtual size in Kbytes (alias vsize)

             wchan    wait channel (as a symbolic name)

             xstat    exit or stop status (valid only for stopped or zombie process)

        :param pid: идентификатор (PID) процесса
        :param parameters: параметры для ключа -o команды ps
        :param ignore_errors: игнорировать исплючения при запуске ps
        :return: словарь, ключи которого - переданные параметры, значение - полученные значения
    """
    if parameters is None:
        parameters = ('%cpu', '%mem', 'time', 'command')
    try:
        ps_process = run_process(['ps', 'www', '-p', str(pid), '-o', ' '.join(parameters)], outs_to_pipe=True)
        ps_results = [result for result in ps_process.stdout.readlines()[1][:-1].split(' ') if result]
        ps_process.stdout.close()
        ps_process.stderr.close()
        return dict(zip(parameters, ps_results))
    except (common_errors.TaskError, errors.SandboxSubprocessError) as error:
        if ignore_errors:
            logger.error('ps launch for pid {0} failed. Error: {1}'.format(pid, error))
            return {}
        else:
            raise


class CustomOsEnviron(object):
    """
        Context manager for settings and removing environment variables

        Usage:
            with CustomOsEnviron({'LDFLAGS': "-L/skynet/python/lib"}):
                ...
    """

    def __init__(self, env_vars):
        self.env_vars = env_vars
        self.old_vars = {}

    def __enter__(self):
        for k, v in self.env_vars.items():
            # save old values
            if k in os.environ:
                self.old_vars[k] = os.environ[k]
            os.environ[k] = v

    def __exit__(self, *args, **kw):
        for k in self.env_vars:
            # restore old values
            if k in self.old_vars:
                os.environ[k] = self.old_vars[k]
            elif k in os.environ:
                del os.environ[k]


class _FuncRunner(threading.Thread):
    def __init__(self, func, sleep_time=5):
        threading.Thread.__init__(self)

        self.func = func
        self.sleep_time = sleep_time
        self.__running = True

    def run(self):
        try:
            while self.func():
                if not self.__running:
                    break
                time.sleep(self.sleep_time)
        except:
            logger.error(traceback.format_exc())

    def stop(self):
        self.__running = False


def start_process_profiler(process, columns, out_file_name, sleep_time=5, watchdog_func=None):
    """
        В фоновом потоке сохраняет информацию о процессе

        :param process: объект subprocess.Popen
        :param columns: подробнее - см. функцию get_process_info
        :param out_file_name: куда сохранять информацию
        :param sleep_time: как часто сохранять, пауза в секундах
    """
    def func():
        if process.poll() is not None:
            return False
        info = get_process_info(process.pid, columns, ignore_errors=False)
        with open(out_file_name, "a+") as out_file:
            out_file.write(json.dumps(info) + "\n")

        if watchdog_func:
            return watchdog_func(info)

        return True

    thread = _FuncRunner(func, sleep_time)
    thread.start()
    return thread


def start_open_files_profiler(out_file_name, sleep_time=5):
    """
        В фоновом потоке сохраняет число файлов, открытых текущим пользователем
        пока нет идей куда это правильно поместить

        :param out_file_name: куда сохранять информацию
        :param sleep_time: как часто сохранять, пауза в секундах
    """
    import getpass

    def func():
        last_file_name = out_file_name + ".last"
        with open(last_file_name, "w") as last_file:
            run_process(['lsof', '-u', getpass.getuser()], stdout=last_file)
        count = len(open(last_file_name).read().split("\n"))
        with open(out_file_name, "a+") as out_file:
            out_file.write(str(count) + "\n")
        return True
    thread = _FuncRunner(func, sleep_time)
    thread.start()
    return thread


def kill_process(process, time_to_kill=None):
    """
        Убить процесс - аналог kill -9 {pid}

        :param process: объект процесса или pid в виде строки или числа
        :param time_to_kill: целое число. вместо kill -KILL сначала отправить
                             kill -TERM, затем, если процесс не завершился в
                             течение time_to_kill секунд, отправить kill -KILL
        :return: True, если процесс был убит, False в противном случае
    """
    if isinstance(process, six.string_types) or isinstance(process, six.integer_types):
        process_pid = int(process)
    else:
        process_pid = process.pid
    try:
        if time_to_kill:
            logger.debug('Terminating {0}'.format(process_pid))
            os.kill(process_pid, signal.SIGTERM)
            for i in range(time_to_kill):
                logger.debug('Waiting for {0} ({1}s elapsed)'.format(process_pid, i))
                time.sleep(1)
                pid, status = os.waitpid(process_pid, os.WNOHANG)
                if pid == process_pid and (os.WIFEXITED(status) or
                                           os.WIFSIGNALED(status)):
                    logger.debug('Succesfully terminated {0}'.format(process_pid))
                    return True
        logger.debug('Killing {0}'.format(process_pid))
        os.kill(process_pid, signal.SIGKILL)
        return True
    except OSError as error:
        msg = 'Failed to kill process {0}. Error: {1}'.format(process_pid, error)
        if error.errno != errno.ESRCH:
            logger.exception(msg)
        else:
            logger.warn(msg)
    return False
