"""
    !!! THIS MODULE IS DEPRECATED, USE SDK2 !!!
    Module for work with Sandbox XMLRPC interface.
    Contains wrappers of all sorts (server, tasks, resources, releases) and general constants (mostly statuses)
"""

from __future__ import absolute_import, unicode_literals

import os
import json
import time
import calendar
import aniso8601

import six
from six.moves import http_client as httplib

from sandbox.common import rest as common_rest
from sandbox.common import proxy as common_proxy
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import platform as common_platform
from sandbox.common.types import task as ctt
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr

# !!! OLD TASK STATUSES ARE DEPRECATED, USE `task.new_status` AND STATUSES FROM `sandbox.common.types.task` !!!

TASK_NOT_READY = 'NOT_READY'
TASK_ENQUEUED = 'ENQUEUED'
TASK_EXECUTING = 'EXECUTING'
TASK_FINISHED = 'FINISHED'
TASK_STOPPING = 'STOPPING'
TASK_STOPPED = 'STOPPED'
TASK_FAILURE = 'FAILURE'
TASK_WAIT_CHILD = 'WAIT_CHILD'
TASK_UNKNOWN = 'UNKNOWN'
TASK_DELETED = 'DELETED'


TASK_STATUSES = (
    TASK_NOT_READY,
    TASK_EXECUTING,
    TASK_FAILURE,
    TASK_FINISHED,
    TASK_ENQUEUED,
    TASK_WAIT_CHILD,
    TASK_STOPPING,
    TASK_STOPPED,
    TASK_UNKNOWN,
    TASK_DELETED,
)


TASK_OK_STATUSES = (
    TASK_FINISHED,
    TASK_ENQUEUED,
    TASK_EXECUTING,
    TASK_WAIT_CHILD,
    TASK_STOPPING,
    TASK_STOPPED,
)


TASK_DONE_STATUSES = (
    TASK_FINISHED,
    TASK_FAILURE,
    TASK_DELETED,
)


TASK_FAILED_STATUSES = (
    TASK_FAILURE, )


RESOURCE_NOT_READY = 'NOT_READY'
RESOURCE_READY = 'READY'
RESOURCE_BROKEN = 'BROKEN'
RESOURCE_DELETED = 'DELETED'


RESOURCE_STATUSES = (
    RESOURCE_NOT_READY,
    RESOURCE_READY,
    RESOURCE_BROKEN,
    RESOURCE_DELETED,
)


RESOURCE_OK_STATUSES = (
    RESOURCE_READY,
    RESOURCE_NOT_READY,
)


RELEASE_STABLE = 'stable'
RELEASE_PRESTABLE = 'prestable'
RELEASE_TESTING = 'testing'
RELEASE_UNSTABLE = 'unstable'
RELEASE_CANCELLED = 'cancelled'


RELEASE_STATUSES = (
    RELEASE_STABLE,
    RELEASE_PRESTABLE,
    RELEASE_TESTING,
    RELEASE_UNSTABLE,
    RELEASE_CANCELLED,
)


ARCH_ANY = 'any'
ARCH_OSX = 'osx'
ARCH_LINUX = 'linux'
ARCH_FREEBSD = 'freebsd'

ARCH_VALUES = list(ctm.OSFamily)


class SandboxAPIException(Exception):
    pass


class SandboxAPIValidateException(Exception):
    pass


class SandboxValidator(object):
    """
        API parameters and fields validators.
        All validators raise SandboxAPIValidateException on invalid values
    """

    @staticmethod
    def check_arch(arch):
        if arch not in ARCH_VALUES:
            raise SandboxAPIValidateException('Incorrect "arch" parameter value: {0}'.format(arch))
        return arch

    @staticmethod
    def check_task_context(context):
        if not isinstance(context, dict):
            raise SandboxAPIValidateException('Incorrect "ctx" parameter (its type is not dict): {0}'.format(context))
        return context

    @staticmethod
    def __check_id(id_value, id_type=''):
        error_message = 'Incorrect {0} parameter: {1}'.format(id_type, id_value)
        if isinstance(id_value, six.string_types) and len(id_value) == 24:
            return id_value
        try:
            id_value = int(id_value)
        except ValueError:
            raise SandboxAPIValidateException(error_message)
        if not id_value > 0:
            raise SandboxAPIValidateException(error_message)
        return id_value

    @classmethod
    def check_task_id(cls, task_id):
        """
        .. warning:: **DEPRECATED**
        """
        return cls.__check_id(task_id, 'task id')

    @classmethod
    def check_resource_id(cls, resource_id):
        """
        .. warning:: **DEPRECATED**
        """
        return cls.__check_id(resource_id, 'resource id')

    @staticmethod
    def check_release_status(status):
        if status not in RELEASE_STATUSES:
            raise SandboxAPIValidateException('Incorrect release status type parameter: {0}'.format(status))
        else:
            return status

    @staticmethod
    def check_resource_status(status):
        if status not in RESOURCE_STATUSES:
            raise SandboxAPIValidateException('Incorrect release status type parameter: {0}'.format(status))
        else:
            return status

    @staticmethod
    def check_task_status(status):
        if status in TASK_STATUSES:
            return status
        else:
            raise SandboxAPIValidateException('Incorrect task status parameter: {0}'.format(status))


class SandboxResource(object):
    def __init__(self, parameters):
        self.skynet_id = parameters['skynet_id']
        self.description = parameters['name']
        self.url = parameters['url']
        self.proxy_url = parameters['proxy_url']
        self.file_name = parameters['file_name']
        self.task_id = parameters['task_id']
        self.type = parameters['type_name']
        self.host = parameters['host']
        self.file_md5 = parameters['file_md5']
        self.status = parameters['state']
        self.timestamp = parameters['timestamp']
        self.owner = parameters['owner']
        self.size = parameters['size']
        self.arch = SandboxValidator.check_arch(parameters['arch'])
        self.complete = parameters['complete']
        self.id = parameters['id']
        self.attributes = parameters['attrs']
        self.rsync = parameters['rsync']
        self.path = self.__remote_path()

    @classmethod
    def from_rest(cls, data):
        from sandbox import sdk2

        parameters = data.copy()

        parameters["name"] = data["description"]
        parameters["proxy_url"] = data["http"]["proxy"]
        parameters["url"] = data["http"]["links"][0] if data["http"]["links"] else ""
        parameters["task_id"] = data["task"]["id"]
        try:
            resource_type = sdk2.Resource[data["type"]]
        except common_errors.UnknownResourceType:
            resource_type = sdk2.Resource["UNKNOWN_RESOURCE"]

        parameters["type_name"] = str(resource_type)
        parameters["host"] = ""
        parameters["file_md5"] = data["md5"]
        parameters["timestamp"] = calendar.timegm(
            aniso8601.parse_datetime(data["time"]["created"]).timetuple()
        )
        parameters["size"] = data["size"] >> 10
        arch = data["arch"]
        parameters["arch"] = (
            ctm.OSFamily.ANY
            if resource_type.any_arch or not arch else
            common_platform.get_arch_from_platform(arch)
        )

        parameters["complete"] = 0 if data["state"] == ctr.State.NOT_READY else 1
        parameters["attrs"] = parameters["attributes"]
        parameters["rsync"] = ""

        # for compatibility with XMLRPC None values should be replaced with empty string
        for k, v in parameters.iteritems():
            if v is None:
                parameters[k] = ""

        return cls(parameters)

    def __remote_path(self):
        try:
            local_path = os.path.join(*(ctt.relpath(self.task_id) + [self.file_name]))
            return os.path.join(common_config.Registry().client.tasks.data_dir, local_path)
        except:
            return None

    def is_ready(self):
        return self.status == RESOURCE_READY

    def is_broken(self):
        return self.status == RESOURCE_BROKEN

    def is_deleted(self):
        return self.status == RESOURCE_DELETED

    def is_ok(self):
        """
            Whether the resource is in correct state (may become or is "READY")
        """
        return self.status in RESOURCE_OK_STATUSES

    def __repr__(self):
        return 'Sandbox resource #{0}, type {1}, from task {2}'.format(self.id, self.type, self.task_id)


class SandboxTask(object):
    id = None

    def __init__(self, task_info):
        self.id = task_info['id']
        self.owner = task_info['owner']
        self.priority = task_info['priority']
        self.timestamp_finish = task_info['timestamp_finish']
        self.description = task_info['descr']
        self.type = task_info['type_name']
        self.parent_id = task_info['parent_id']
        self.hidden = task_info['hidden']
        self.status = ctt.Status.old_status(task_info['status'])
        self.new_status = task_info['status']
        self.timestamp = task_info['timestamp']
        self.timestamp_start = task_info['timestamp_start']
        self.important = task_info['important']
        self.arch = task_info['arch']
        self.info = task_info['info']
        self.url = task_info['url']
        self.execution_space = task_info['execution_space']
        self.host = task_info['host']
        self.model = task_info['model']
        self.ctx = task_info['ctx']
        if isinstance(self.ctx, six.string_types):
            self.ctx = json.loads(self.ctx)

    def is_not_ready(self):
        return self.status == TASK_NOT_READY

    def is_executing(self):
        return self.status == TASK_EXECUTING

    def is_failure(self):
        return self.status == TASK_FAILURE

    def is_finished(self):
        return self.status == TASK_FINISHED

    def is_enqueued(self):
        return self.status == TASK_ENQUEUED

    def is_wait_child(self):
        return self.status == TASK_WAIT_CHILD

    def is_stopped(self):
        return self.status == TASK_STOPPED

    def is_unknown(self):
        return self.status == TASK_UNKNOWN

    def is_deleted(self):
        return self.status == TASK_DELETED

    def is_ok(self):
        return self.status in TASK_OK_STATUSES

    def is_done(self):
        """
            Check whether the task is in a final state
        """
        return self.status in TASK_DONE_STATUSES

    def is_active(self):
        return self.status in [TASK_ENQUEUED, TASK_EXECUTING]

    def __repr__(self):
        return 'Sandbox task #{0}, type {1}, created by {2}'.format(self.id, self.type, self.owner)


class SandboxRelease(object):
    def __init__(self, release_info):
        self.status = SandboxValidator.check_release_status(release_info['status'])
        self.task_id = release_info['task_id']
        self.owner = release_info['author']
        self.timestamp = release_info['timestamp']
        self.id = release_info['id']
        self.subject = release_info['subject']
        self.resources = [SandboxResource(item) for item in release_info['resources']]

    def __repr__(self):
        return 'Sandbox release #{0} for task #{1}, released by {2}'.format(self.id, self.task_id, self.owner)


class Sandbox(object):
    """
        Sandbox XMLRPC API wrapper.
        If server URL is not specified on creation, default one, sandbox.yandex-team.ru, is used
    """

    def __init__(self, host=None, port=None, **kwargs):
        """
            :param host: Sandbox server fully qualified domain name (FQDN)
            :param port: XMLRPC API port
            :param kwargs: additional parameters to pass to an XMLRPC wrapper
        """
        url = kwargs.pop('url', None)
        if not url:
            if not host:
                host = 'sandbox.yandex-team.ru'
            if host.startswith('http'):
                url = host
            else:
                url = "http{0}://{1}{2}/sandbox/xmlrpc".format(
                    ("s" if not port else ""), host, ("" if not port or int(port) == 80 else ":" + str(port))
                )
        self.xmlrpc_url = url
        self.server = common_proxy.ThreadLocalCachableServerProxy(
            url=url, ua=common_config.Registry().this.id, **kwargs
        )
        self._sdk_server = common_rest.Client(auth=kwargs.get("auth"))

    def __str__(self):
        return '<SandboxAPI object {}>'.format(self.server.url)

    @staticmethod
    def _to_id(obj):
        if isinstance(obj, six.integer_types):
            return obj
        elif isinstance(obj, six.string_types):
            try:
                return int(obj)
            except ValueError:
                raise SandboxAPIValidateException("Can't convert '{}' to valid ID".format(obj))
        try:
            return obj.id
        except AttributeError:
            raise SandboxAPIValidateException("Can't convert '{}' to valid ID".format(obj))

    def ping(self):
        responce = self.server.ping()
        if responce == 'OK':
            return True
        else:
            return False

    ########################################
    # task
    ########################################

    def list_task_types(self):
        """
            List task types available on the server
        """
        return self.server.list_task_types()

    def list_tasks(
        self, task_type=None, parent_id=None, hidden=True,
        owner=None, limit=None, offset=None, completed_only=None, status=None,
        children=True, descr_mask=None, important_only=None, get_id_only=False, host=None
    ):
        """
            Find tasks by parameters

            :param hidden: include hidden tasks
            :param limit: maximum tasks to return
            :param offset: results offset.
                For example, if tasks with IDs 3, 4, 5 and 6 match the criteria, offset=2 returns all but first two
            :param completed_only: only include tasks that are finished (overrides parameter `status`)
            :param children: include tasks that are yielded by other tasks
            :param descr_mask: description mask that matches MySQL "like" format
            :param host: task execution host
            :return: list of SandboxTask objects, list of tasks IDs if get_id_only=True, [] if nothing is found.
                SandboxAPIValidateException is raised if one of parameters is invalid
        """
        parameters = {}
        if task_type:
            parameters['task_type'] = str(task_type)

        if parent_id:
            parameters['parent_id'] = parent_id

        if hidden:
            parameters['hidden'] = True
        else:
            parameters['hidden'] = False

        if children:
            parameters['show_childs'] = True
        else:
            parameters['show_childs'] = False

        if owner:
            owner = str(owner)
            parameters['owner'] = owner

        if status:
            parameters['status'] = SandboxValidator.check_task_status(status)

        if completed_only:
            parameters['completed_only'] = True
        else:
            parameters['completed_only'] = False

        if limit:
            limit = int(limit)
            parameters['limit'] = limit

        if offset:
            offset = int(offset)
            parameters['offset'] = offset

        if descr_mask:
            descr_mask = str(descr_mask)
            parameters['descr_mask'] = descr_mask

        if important_only:
            parameters['important_only'] = True
        else:
            parameters['important_only'] = False

        if get_id_only:
            parameters['load'] = False
        else:
            parameters['load'] = True

        if host:
            parameters['host'] = host

        results = self.server.list_tasks_legacy(parameters)
        if results:
            if get_id_only:
                return results
            else:
                return [SandboxTask(task_info) for task_info in results]
        else:
            return []

    def get_task(self, task_id):
        task_info = self.server.get_task(self._to_id(task_id))
        if task_info:
            return SandboxTask(task_info)
        else:
            return None

    def create_task(
        self,
        task_type,
        owner,
        description,
        parent_task_id=None,
        context=None,
        host=None,
        model=None,
        arch=None,
        priority=None,
        important=False,
        execution_space=None,
        parameters=None,
        enqueue=True
    ):
        """
            Create a task

            :param parent_task_id: parent task identifier
            :param context: task context, which contains input parameters
            :param host: host to execute task on
            :param model: CPU model of hosts the task may execute on
            :param arch: architecture of hosts the task may execute on
            :param priority: task priority, a tuple of priority class and subclass names
            :param important: mark the task as important
            :param execution_space: execution space requirement (disk, in MiB)
            :param parameters: a dictionary with additional parameters (for ex., kill_timeout)
            :param enqueue: enqueue task after creation
            :return: SandboxTask
        """
        owner = str(owner)
        if not context:
            context = {}
        context = SandboxValidator.check_task_context(context)
        context = {k: v for k, v in context.iteritems() if not k.startswith('__')}

        description = str(description)
        task_parameters = {
            "type": task_type,
            "description": description,
            "owner": owner,
            "context": context,
        }

        task_parameters["requirements"] = {}
        if parent_task_id is not None:
            task_parameters["children"] = True
        if arch:
            task_parameters["requirements"]["platform"] = str(arch)
        if host:
            task_parameters["requirements"]["host"] = str(host)
        if model:
            task_parameters["requirements"]["cpu_model"] = str(model)
        if priority:
            task_parameters["priority"] = priority
        if important:
            task_parameters["important"] = True
        if execution_space is not None:
            task_parameters["requirements"]["disk_space"] = int(execution_space) << 20
        if parameters is not None:
            if "notifications" in parameters:
                task_parameters["notifications"] = parameters["notifications"]
            if "tags" in parameters:
                task_parameters["tags"] = parameters["tags"]
            if "ram" in parameters and parameters["ram"] is not None:
                task_parameters["requirements"]["ram"] = parameters["ram"] << 20
            if "fail_on_any_error" in parameters and parameters["fail_on_any_error"] is not None:
                task_parameters["fail_on_any_error"] = parameters["fail_on_any_error"]
            if "tasks_archive_resource" in parameters and parameters["tasks_archive_resource"] is not None:
                task_parameters["tasks_archive_resource"] = parameters["tasks_archive_resource"]
            if parameters.get("max_restarts") is not None:
                task_parameters["max_restarts"] = parameters["max_restarts"]

        task_id = self._sdk_server.task(**task_parameters)["id"]
        if enqueue:
            result = self._sdk_server.batch.tasks.start.update([task_id])
            if result[0]["status"] == ctm.BatchResultStatus.ERROR:
                raise common_errors.TaskNotEnqueued("Task #{} is not enqueued: {}".format(task_id, result[0]))

        return self.get_task(task_id)

    def wait_task(self, task_id, timeout=300, checking_interval=10):
        """
            Wait until the task is done executing.
            SandboxAPIException raised

            :param task_id: identifier of a task to wait
            :param timeout: amount of time to wait for, in seconds
            :param checking_interval: how often task status should be checked, in seconds
            :return: True if the task has successfully finished, False otherwise
        """
        def check_status(task_id):
            """
                Check task status.

                :param tid: identifier of a task to check
                :return: None if the task isn't finished yet, True if finished successfully, False if failed
            """
            task_state = ctt.Status.old_status(self.server.get_task_status(task_id))
            if task_state in TASK_DONE_STATUSES:
                if task_state in TASK_OK_STATUSES:
                    return True
                elif task_state in TASK_FAILED_STATUSES:
                    return False
                else:
                    raise SandboxAPIException('Incorrect task state {0}'.format(task_state))
            else:
                return None

        task_id = self._to_id(task_id)
        if not timeout:
            result = check_status(task_id)
            if result is not None:
                return result
        failure_time = time.time() + int(timeout)
        while time.time() < failure_time:
            result = check_status(task_id)
            if result is None:
                time.sleep(checking_interval)
            else:
                return result
        raise SandboxAPIException("Task is not finished. Terminated by timeout.")

    @staticmethod
    def set_task_context_value(task_id, name, value):
        from sandbox.sandboxsdk.channel import channel
        return channel.rest.set_task_context_value(task_id, name, value)

    def set_task_priority(self, task_id, priority):
        """
            Set task priority according to task and owner limits

            :param task_id: task identifier
            :param priority: new priority

            :return: True on success, False otherwise
        """
        return self.server.set_task_priority(self._to_id(task_id), priority)

    ########################################
    # resource
    ########################################

    def list_resource_types(self):
        """
            List resource types available on the server
        """
        return self.server.list_resource_types()

    @staticmethod
    def list_resources(
        resource_type=None, omit_failed=None, status=None, hidden=None, owner=None, task_id=None,
        attribute_name=None, attribute_value=None, all_attrs=None, any_attrs=None,
        limit=10, offset=None, arch=None, host=None,
        order_by=None
    ):
        """
            Find resources by parameters

            :param omit_failed: do not include broken resources. Warning: takes precedence over "status" parameter
            :param hidden: include hidden resources
            :param attribute_name: filter by a single attribute
            :param attribute_value: filter by a single attribute
            :param all_attrs: dict with ALL attributes to filter by, {attr_name: attr_value}
            :param any_attrs: dict with ANY attribute to filter by, {attr_name: attr_value}
            :param limit: maximum resources to return
            :param offset: results offset.
                For example, for resources 3, 5, 42 offset=2 returns only the last one
            :param arch: resource architecture
            :param host: host on which resource is stored
            :param order_by: field to sort by (default is "-id"). Examples: "id", "-size"
            :return: list of SandboxResource objects, None if nothing is found
        """
        from sandbox.sandboxsdk.channel import channel
        return channel.rest.list_resources(
            resource_type, omit_failed, status, hidden, owner, task_id,
            attribute_name, attribute_value, all_attrs, any_attrs,
            limit, offset, arch, host, order_by
        )

    @staticmethod
    def get_resource(resource_id):
        from sandbox.sandboxsdk.channel import channel
        return channel.rest.get_resource(resource_id)

    def delete_resource(self, resource_id, ignore_last_usage_time=False):
        from sandbox.sdk2 import legacy

        cur_task = legacy.current_task
        if ignore_last_usage_time and cur_task and cur_task.sdk1_agentr_enabled:
            res = cur_task.agentr.resource_delete(resource_id)
            if res["status"] == ctm.BatchResultStatus.SUCCESS:
                return None
            else:
                return res["message"]
        else:
            # this only seems to be used by mistake (doesn't delete resources unless their ttl expired)
            return self.server.delete_resource(str(self._to_id(resource_id)), ignore_last_usage_time)

    def get_resource_attribute(self, resource_id, attribute_name):
        resource = self.get_resource(self._to_id(resource_id))
        return resource.attributes.get(attribute_name) if resource else None

    def set_resource_attribute(self, resource_id, attribute_name, attribute_value=''):
        """
            Set value of a resource's attribute (create one if it doesn't exist)

            :param resource_id: resource identifier
            :param attribute_name: attribute name
            :param attribute_value: attribute value
            :return: True on success, False otherwise
        """
        return self.server.set_resource_attr(str(self._to_id(resource_id)), attribute_name, attribute_value)

    def drop_resource_attribute(self, resource_id, attribute_name):
        """
            Remove resource attribute

            :param resource_id: resource identifier
            :param attribute_name: attribute name to remove
            :return: True on success, False otherwise
        """
        return self.server.drop_resource_attr(str(self._to_id(resource_id)), attribute_name)

    def get_resource_http_links(self, resource_id):
        return self.server.get_resource_http_links(str(self._to_id(resource_id)))

    def get_resource_rsync_links(self, resource_id):
        return self.server.get_resource_rsync_links(str(self._to_id(resource_id)))

    def get_resource_hosts(self, resource_id, get_all_hosts=False):
        """
            Get a list of Sandbox hosts where resource copies are located

            :param resource_id: resource identifier
            :param get_all_hosts: include hosts where the said resource is marked for deletion
            :return: list of host names
        """
        return self.server.get_resource_hosts(str(self._to_id(resource_id)), get_all_hosts)

    ########################################
    # release
    ########################################

    @staticmethod
    def list_releases(
        resource_type=None, arch=None, release_status='stable', include_broken=False, task_id=None,
        order_by=None, limit=None, offset=None
    ):
        """
            Find releases by parameters; default behaviour is return all releases in "stable" status

            :param arch: resource architecture
            :param release_status: release status (see RELEASE_STATUSES constant for reference)
            :param include_broken: include resources in BROKEN state
            :param order_by: field to sort by (default is "-id"). Examples: "id", "-release__creation_time"
            :param limit: maximum releases to return
            :param offset: results offset.
                For example, if releases 1, 2, .., 10 are found, offset=9 returns a list with only the last release
            :return: list of SandboxRelease objects, or [] if nothing is found.
                SandboxAPIValidateException is raised if one of parameters is invalid
        """
        from sandbox.sandboxsdk.channel import channel
        return channel.rest.list_releases(
            resource_type, arch, release_status, include_broken, task_id, order_by, limit, offset
        )

    ########################################
    # client
    ########################################

    def list_clients(self):
        """
            List all sandbox clients

            :return: list of dicts
        """
        return self.server.list_clients()

    ########################################
    # notifications
    ########################################

    def send_email(
        self, mail_to, mail_cc, mail_subject, mail_body, content_type='text/plain', charset='utf-8', extra_headers=None,
        urgent=False, owner='sandbox'
    ):
        """
            Send a message from sandbox-noreply@ via email.
            If its size exceeds 10'000KB, the message body is replaced with a warning.

            :param content_type: message type, either of "text/plain", "text/html"
            :param owner: notification author to be displayed in web interface
            :param urgent: send message from sandbox-urgent@
            :return: notification identifier
        """
        return self.server.send_email(
            mail_to, mail_cc, mail_subject, mail_body, content_type, charset, extra_headers, urgent, owner
        )


class SandboxRest(object):
    """ Backwards compatible implementation for Sandbox XMLRPC API wrapper, using REST API internally """

    def __init__(self, *args, **kwargs):
        self.server = common_rest.DispatchedClient(*args, **kwargs)

    def get_resource(self, resource_id):
        try:
            resource_info = self.server.resource[Sandbox._to_id(resource_id)][:]
        except self.server.HTTPError as ex:
            if ex.status == httplib.NOT_FOUND:
                return None
            raise
        return SandboxResource.from_rest(resource_info)

    def list_resources(
        self, resource_type=None, omit_failed=None, status=None, hidden=None, owner=None, task_id=None,
        attribute_name=None, attribute_value=None, all_attrs=None, any_attrs=None,
        limit=10, offset=None, arch=None, host=None,
        order_by=None
    ):
        fltr = {"limit": limit if limit else 3000}
        if resource_type:
            fltr["type"] = str(resource_type)
        if order_by:
            fltr["order"] = order_by
        if offset:
            fltr["offset"] = offset
        if task_id:
            fltr["task_id"] = Sandbox._to_id(task_id)
        if omit_failed:
            fltr["state"] = [ctr.State.READY, ctr.State.NOT_READY]
        elif status is not None:
            fltr["state"] = SandboxValidator.check_resource_status(status)
        if owner:
            fltr["owner"] = owner
        if arch:
            fltr["arch"] = str(arch)

        attrs = {}
        if attribute_name or attribute_value:
            attrs[attribute_name] = attribute_value
        if all_attrs:
            attrs.update(all_attrs)
        elif any_attrs:
            attrs.update(any_attrs)
            fltr["any_attr"] = True
        if attrs:
            fltr["attrs"] = attrs

        resources = self.server.resource.read(**fltr)
        return [SandboxResource.from_rest(r) for r in resources["items"]]

    def list_releases(
        self, resource_type=None, arch=None, release_status='stable', include_broken=False, task_id=None,
        order_by=None, limit=10, offset=None
    ):
        releases_filter = {"limit": limit if limit else 3000}
        if resource_type:
            releases_filter["resource_type"] = str(resource_type)
        if arch:
            releases_filter["arch"] = arch
        if release_status:
            releases_filter["release_status"] = SandboxValidator.check_release_status(release_status)
        if task_id:
            releases_filter["task_id"] = Sandbox._to_id(task_id)
        if order_by:
            releases_filter["order_by"] = order_by
        if offset:
            releases_filter["offset"] = int(offset)
        if include_broken:
            releases_filter["include_broken"] = True

        releases = self.server.release.list.read(**releases_filter)
        return [SandboxRelease(release_info) for release_info in releases]

    def set_task_context_value(self, task_id, name, value):
        from sandbox.sandboxsdk.channel import channel

        task_id = Sandbox._to_id(task_id)
        cur_task = channel.task
        assert cur_task and task_id == cur_task.id, "Task can only change self context"
        self.server.task.current.context.value = {"key": str(name), "value": value}
