# -*- coding: utf-8 -*-
"""
    !!! THIS MODULE IS DEPRECATED, USE SDK2 !!!
    Модуль для описания общего поведения задачи Sandbox на клиенте.
"""

from __future__ import absolute_import, unicode_literals

import os
import sys
import time
import shutil
import socket
import inspect
import logging
import datetime as dt
import itertools as it

import requests

from six.moves import xmlrpc_client as xmlrpclib

from sandbox.common import fs as common_fs
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import format as common_format
from sandbox.common import system as common_system
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import threading as common_threading

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt

from sandbox.sandboxsdk import util
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import environments

from sandbox import sdk2
from sandbox.sdk2.helpers import misc

if common_system.inside_the_binary():
    from sandbox.yasandbox.proxy import base as proxy_task
else:
    from sandbox.yasandbox.proxy import task as proxy_task


logger = logging.getLogger(__name__)


class MemoizeStage(object):
    """
    Pure magic, for the organization stages to be executed only once. Use in method `on_execute`.

    Examples:

        .. code-block:: python

            with self.memoize_stage.important_action:
                # <code will be executed only once>

            with self.memoize_stage.important_action(3):
                # <code will be executed a maximum of three times>

        Where `important_action` is stage name and can be any.

        By default section marks as executed on entrance. You may change this behavior:

        .. code-block:: python

            with self.memoize_stage.interrupted_section(commit_on_entrance=False):
                # <if code raise exception then section will not be marked as executed>

    """

    SkipStageException = Exception()

    def __init__(self, task, stage_name):
        self.task = task
        self.stage_name = stage_name
        # keys his name start with "__" can't be saved throught set_task_context_value
        self._exec_key = '_a_stage_{}_exec_count__'.format(self.stage_name)
        self._skip_key = '_a_stage_{}_skip_count__'.format(self.stage_name)
        self._max_runs = 1
        self.executed = None
        self.commit_on_entrance = True
        self.__commit_on_wait = True
        self._logger = logging.getLogger(self.__class__.__name__)

    @property
    def runs(self):
        return self.task.ctx.get(self._exec_key, 0)

    @property
    def passes(self):
        return self.task.ctx.get(self._skip_key, 0)

    def __inc_key(self, key):
        value = self.task.ctx.get(key, 0) + 1
        self.task.ctx[key] = value
        channel.channel.sandbox.set_task_context_value(self.task.id, key, value)

    def __call__(self, max_runs=None, commit_on_entrance=None, commit_on_wait=None):
        if max_runs is not None:
            self._max_runs = int(max_runs)
        if commit_on_entrance is not None:
            self.commit_on_entrance = bool(commit_on_entrance)
        if commit_on_wait is not None:
            self.__commit_on_wait = bool(commit_on_wait)
        return self

    def __enter__(self):
        skip = self.runs >= self._max_runs
        self.executed = not skip
        if skip:
            self._logger.info("Skipping stage '%s'", self.stage_name)
            self.__inc_key(self._skip_key)
            # OMG! Do some magic
            sys.settrace(lambda *args, **keys: None)
            frame = inspect.currentframe(1)
            frame.f_trace = self.trace
        else:
            self._logger.info("Entering stage '%s'", self.stage_name)
            if self.commit_on_entrance:
                self.__inc_key(self._exec_key)

        return self

    def trace(self, frame, event, arg):
        raise self.SkipStageException

    def __exit__(self, exc_type, exc_value, traceback):
        self._logger.info("Exiting stage '%s'", self.stage_name)
        if exc_value is None or isinstance(exc_value, common_errors.Wait):
            if not self.commit_on_entrance and (exc_value is None or self.__commit_on_wait):
                self.__inc_key(self._exec_key)
        elif exc_value is self.SkipStageException:
            # suppress the SkipStageException
            return True


class MemoizeCreator(object):
    def __init__(self, task):
        self.task = task

    def __getattr__(self, name):
        return MemoizeStage(self.task, name)

    def __getitem__(self, name):
        return MemoizeStage(self.task, name)


class SandboxTask(proxy_task.Task):
    """
        Общий класс задач. От него должны быть унаследованы все задачи Sandbox
    """
    agentr = None
    # Current task's platform (as `platform.platform()`). Will be provided on task execution.
    platform = None
    # Current task's container metadata if any. Will be provided on task execution with the instance of
    # `sandbox.common.types.client.Container` class.
    container = None
    # сохранять или нет файлы корок, которые образовались в результате работы задачи, в ресурсах
    save_gdb_coredump_files = True

    @common_patterns.singleton_classproperty
    def coredump_dir(cls):
        return common_config.Registry().client.tasks.coredumps_dir

    @common_patterns.singleton_classproperty
    def owners(cls):
        module_path = cls.__module__.replace(".", os.sep)
        owners_file_path = os.path.join(
            common_config.Registry().client.tasks.code_dir,
            "projects",
            ctm.EnvironmentFiles.OWNERS_STORAGE
        )
        owners_storage = misc.SingleFileStorage(owners_file_path, autoload=True)
        return owners_storage[module_path]

    class ReleaseTemplate(common_patterns.Abstract):
        """
        An instance of this class is required to describe task's release template data
        (see :py:meth:`release_template`).
        """
        __slots__ = ("cc", "subject", "message", "types")
        __defs__ = (None, None, None, list(ctt.ReleaseStatus))

    environment = ()

    archs_for_bundle = ('freebsd', 'linux', )

    def __init__(self, task_id=0):
        proxy_task.Task.__init__(self, task_id)
        self.__client_info = None
        self.__main_thread = None

    @property
    def client_info(self):
        """
        Информация о текущем хосте:
        * arch - вид ОС системы (например, freebsd или linux)
        * os_version - версия ОС
        * cpu_model - модель процессора
        * ncpu - количество ядер CPU
        * physmem - количество памяти (в байтах, в виде строки).
            Опытным путём установлено, что количество байт обычно
            кратно 1024 * 1024, однако всегда чуть меньше "круглого" числа
            в гигабайтах. Так, для машины с памятью 64 GB будет выдано
            64399 * 1024 * 1024 (для "честных" 64 GB должно было бы
            получиться 65536 * 1024 * 1024)
        :return: словарь с информацией о текущем клиенте
        :rtype: dict
        """
        if self.__client_info is None:
            self.__client_info = common_system.get_sysparams()
        return self.__client_info

    def initCtx(self):
        """
            Временная заглушка для дефолтных параметров
            :return:
        """
        return {}

    def on_enqueue(self):
        resources_to_register = (
            self.ctx.get(cls.name)
            for cls in self.input_parameters
            if issubclass(cls, parameters.ResourceSelector) and cls.register_dependency
        )
        for resource_to_register in common_itertools.chain(filter(None, resources_to_register)):
            self._register_dep_resource(resource_to_register)

    def get_default_parameters(self):
        """
            Получить значения по умолчанию для задачи.
            Пока что также вызывает метод initCtx и добавляет его результат к возращаемому словарю
            :return: словарь с значениями по умолчанию
        """
        result = proxy_task.Task.get_default_parameters(self)
        if self.input_parameters:
            for parameter in self.input_parameters:
                self.ctx[parameter.name] = parameter().default_value
        init_result = self.initCtx()
        if init_result and isinstance(init_result, dict):
            result.update(init_result)
        return result

    @property
    def footer(self):
        """
        Return custom task's footer data for new UI. The return value should be JSON serializable.
        In case of the return value is :class:`list`, it will be returned as is,
        in case of :class:`tuple`, the first value will be used as helper name, the second - as context,
        in other cases, it will be placed as context value.
        """
        return None

    @property
    def release_template(self):
        """
        Return custom task's release template for new UI.
        The return value should be an instance of :class:`ReleaseTemplate`. The default implementation
        will use legacy :py:meth:`arcadia_info` to fill the template.
        """
        return self.ReleaseTemplate(subject=self.arcadia_info()[1])

    def path(self, path=''):
        """
            Получить абсолютный путь по переданному относительному от директории таска

            :param path: относительный путь
            :type path: str
            :return: абсолютный путь
            :rtype: str
        """
        path = str(path)
        if os.path.isabs(path):
            raise common_errors.TaskError('Parameter "path" is absolute, value: {}'.format(path))
        elif path:
            return os.path.join(self.abs_path(), path)
        else:
            return self.abs_path()

    def prepare_environment(self):
        """
            Подготовить окружение для выполнения таска - скачать нужные бинарники, распаковать и так далее
            Если указано IS_TEST_SERVER в настройках
        """
        logger.debug("Prepare environment for task.")
        if not self.privileged:
            os.environ["YA_CACHE_DIR"] = os.path.join(environments.SandboxEnvironment.build_cache_dir, "ya")
        if self.environment:
            paths.make_folder(common_config.Registry().client.tasks.env_dir)
            for environment in self.environment:
                environment.prepare()
                environment.touch()

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе
            Может быть переопределён в наследниках для уточнения возвращаемых данных

            :return список из трёх значений revision, tag, branch
        """
        return None, None, None

    def check_release_permissions(self, user):
        """
            Проверить права на релиз таска пользователя.
            Проходит по всем ресурсам, для тех, которые помечены как releasable проверяет,
                есть ли пользователь в списке тех, кто может делать релиз
            Если список релизеров не указан в типе ресурса, смотрится на поле releasers задачи

            :param user: объект пользователя
            :return: True, если пользователь имеет право и возможность релизить, False в противном случае
        """
        # разрешаем релизы, если отключена аутентификация
        if not common_config.Registry().server.auth.enabled:
            return True
        releasable = [r for r in self.list_resources() if r.type.releasable]
        return all(user.login in (r.type.releasers or []) for r in releasable) if releasable else False

    def __kill_registered_subprocesses(self, processes_list):
        """
            Посылает всем процессам, которые были сохранены в списке запущенных процессов, kill -9, если они запущены

            :param processes_list: список объектов процессов, которые нужно проверить
            :return: список убитых процессов
        """
        logger.debug('Kill registered subprocesses.')
        killed_processes = []
        for p in processes_list:
            process_pid = p.pid
            if p.poll() is None:
                was_killed = process.kill_process(process_pid, getattr(p, 'time_to_kill', None))
                if was_killed:
                    killed_processes.append(process_pid)
                    logger.debug('Process with pid {0} was killed.'.format(process_pid))
            else:
                logger.debug('Process with pid {0} already completed.'.format(process_pid))
        return killed_processes

    def __get_coredumps(self, processes_list):
        """
            Сохраняет и обрабатывает созданные за время работы задачи корки

            :param processes_list: список объектов процессов, которые нужно проверить
            :return:
        """
        logger.debug("Try to find core files for registered processes %r", [_.pid for _ in processes_list])
        result = []
        if os.path.exists(self.coredump_dir):
            logger.debug('Coredumps dir "{0}":\n{1}'.format(self.coredump_dir, os.listdir(self.coredump_dir)))
        else:
            logger.error('Coredumps folder {0} does not exist.'.format(self.coredump_dir))
        coredumps = self.agentr.coredumps
        gdb_env_installed = False
        for p in processes_list:
            binary_path = p.saved_cmd.split(' ')[0]
            # save coredump and traces
            binary_name = os.path.split(binary_path)[-1]
            core_file = coredumps.get(p.rpid or p.pid)
            if not core_file:
                core_file = sdk2.helpers.gdb.get_core_path(self.coredump_dir, binary_name, p.pid)
            if core_file:
                if not gdb_env_installed:
                    gdb_version = common_config.Registry().client.tasks.coredumps_gdb_version
                    environments.GDBEnvironment(gdb_version).prepare()
                    gdb_env_installed = True
                self.set_info('Coredump file {} ({}) for process {} was found'.format(
                    core_file, common_format.size2str(os.path.getsize(core_file)), p.pid
                ))
                self.__save_gdb_traces(binary_path, core_file)
                core_resource = None
                if self.save_gdb_coredump_files:
                    core_resource = self.__save_coredump(core_file)
                if core_resource:
                    core_resource_id = core_resource.id
                else:
                    core_resource_id = None
                result.append({
                    'cmd': p.saved_cmd,
                    'pid': p.pid,
                    'core_file': core_file,
                    'core_resource': core_resource_id,
                })
        return result

    def postprocess(self):
        """
            Действия, выполняемые каждый раз после завершения исполнения задачи.

            :return:
        """
        logger.debug("\n\n==== Run postprocess for task {0} ====\n\n".format(self.id))
        self.__kill_registered_subprocesses(self._subproc_list)
        try:
            coredumps_info = self.__get_coredumps(self._subproc_list)
            if coredumps_info:
                self.ctx['__coredumps_info'] = coredumps_info
        except Exception as error:
            logger.error('Cannot check coredumps. Error: {0}'.format(error))
        logger.debug("\n\n==== Postprocess for task {0} is completed ====\n\n".format(self.id))

    def _get_core_path(self, binary_name, process_pid):
        """
            Получить путь до корки указанного процесса с указанным pid-ом, если она существует

            Названия корок должны быть в формате {process_name}.{process_pid}[.signal]
            При этом {process_name} может быть неполным. Примеры:
            # basesearch.27892.11
            # ranking_middlese.14692 (без указания сигнала)

            :param binary_name: название бинарника
            :param process_pid: pid процесса
            :return: путь до корки, если она существует; в противном случае возвращается None
        """
        if not os.path.exists(self.coredump_dir):
            return None
        coredump_path = None
        for core_name in os.listdir(self.coredump_dir):
            core_name_info = core_name.split('.')
            core_process_name = core_name_info[0]
            process_pid_match_template = '.{}.' if len(core_name_info) == 3 else '.{}'
            process_pid_match = process_pid_match_template.format(process_pid)
            # проверяем, что в названии бинарника есть {process_name} и pid-ы совпадают
            if (core_process_name in binary_name) and process_pid_match in core_name:
                coredump_path = os.path.join(self.coredump_dir, core_name)
        if coredump_path:
            # ждём, пока корка допишется (не будет происходить изменение размера в течение пары секунд
            coredump_is_done = False
            while not coredump_is_done:
                current_size = os.path.getsize(coredump_path)
                time.sleep(2)
                new_size = os.path.getsize(coredump_path)
                coredump_is_done = current_size == new_size
            return coredump_path
        else:
            return None

    def __save_coredump(self, coredump_path):
        """
            Сохранить корку

            :param coredump_path: путь до файла корки
            :return: объект сохранённого ресурса; None, если ресурс сохранить не получилось
        """
        coredump_path = os.path.abspath(coredump_path)
        logger.debug('Save coredump {0}'.format(coredump_path))
        coredump_filename = os.path.basename(coredump_path)
        saved_coredump_path = self.abs_path(coredump_filename)
        gzipped_coredump_path = saved_coredump_path + '.gz'
        try:
            if coredump_path != saved_coredump_path:
                shutil.move(coredump_path, saved_coredump_path)
            process.run_process(['gzip', saved_coredump_path])
        except OSError as e:
            logger.exception(e)
            self.set_info("Cannot copy coredump {0}".format(coredump_filename))
            return None
        paths.add_read_permissions_for_path(gzipped_coredump_path)
        coredump_resource = self.create_resource(
            description='{0} coredump'.format(coredump_filename),
            resource_path=gzipped_coredump_path,
            resource_type=sdk2.service_resources.CoreDump,
        )
        self.mark_resource_ready(coredump_resource)
        self.set_info("COREDUMP was saved as resource:{0}".format(coredump_resource.id))
        self.set_info('<hr/>', do_escape=False)
        return coredump_resource

    def _send_core_to_server(self, url, core_data, binary_path):
        """
            Send coredump to cores aggregator.

            :param url: coredump aggregator url
            :param core_data: trace body
            :param binary_path: coredumped binary full path
        """
        response = requests.post(
            url=url + "/submit_core",
            json={"parsed_traces": core_data, "dump_json": {}},
            params={
                "time": int(time.time()),
                # treat binary basename as itype
                "service": os.path.split(binary_path)[-1],
                "ctype": "sandbox-task",
                "server": common_config.Registry().this.fqdn,
                "prj": self.type,
                "task_id": self.id,
            },
            timeout=5,
        )
        response.raise_for_status()
        logger.debug("Core sent to %s\nResponse %s", url, response.text)

    def __save_gdb_traces(self, binary_path, coredump_path):
        """
            Сохранить трейсы gdb в папке логов

            :param binary_path: путь до бинарника
            :param coredump_path: путь до корки
        """

        cmd = sdk2.helpers.gdb.gdb_trace_command(binary_path, coredump_path)
        core_name = os.path.split(coredump_path)[-1]
        get_gdb_traces = process.run_process(cmd, check=False, shell=False, log_prefix='{0}.gdb'.format(core_name))
        html_traceback_log = sdk2.helpers.gdb.get_html_view_for_logs_file(
            'gdb_traceback',
            os.path.basename(get_gdb_traces.stdout_path),
            self._log_resource.id,
        )
        self.set_info("GDB traceback is available in task logs:<br />{0}".format(html_traceback_log), do_escape=False)

        try:
            filtered_output_name = os.path.join(
                self.abs_path(),
                '{0}.gdb_traceback.filtered.html'.format(core_name)
            )
            util.filter_traceback(get_gdb_traces.stdout_path, filtered_output_name)

            traceback_resource = self.create_resource(
                description='Filtered traceback',
                resource_path=filtered_output_name,
                resource_type=sdk2.service_resources.FilteredGdbTraceback,
                attributes={
                    # these resources are small, but can be useful for similar
                    # coredumps investigation.
                    'ttl': 120,
                },
            )
            self.mark_resource_ready(traceback_resource)

            html_filtered_traceback = sdk2.helpers.gdb.get_html_view_for_logs_file(
                'filtered_traceback',
                filtered_output_name,
                traceback_resource.id,
                is_dir=False)

            self.set_info(
                "GDB filtered traceback is also available:<br />{0}".format(html_filtered_traceback),
                do_escape=False)

        except Exception:
            logger.exception('Cannot filter traceback:')
        else:
            try:
                with open(get_gdb_traces.stdout_path, "r") as core_trace:
                    core_data = core_trace.read()
                    self._send_core_to_server("https://coredumps.yandex-team.ru", core_data, binary_path)
                    self._send_core_to_server("https://coredumps-testing.yandex-team.ru", core_data, binary_path)
            except Exception:
                logger.exception("Cannot send traceback")

    def on_before_timeout(self, seconds):
        """
        Called before task executor is killed by timeout.

        :param seconds: seconds left until timeout
        """
        logging.warning("[TC] %s seconds left until task timeout", seconds)
        if seconds == min(self.timeout_checkpoints()):
            logging.warning("[TC] Dumping threads tracebacks for debug purpose:")
            common_threading.dump_threads()

    def timeout_checkpoints(self):
        """
        This method returns a list of intervals (in seconds) before timeout
        when on_before_timeout method is called.

        :rtype: list(int)
        """
        return [10, 30, 60, 60 * 3, 60 * 5]

    ########################################
    # task
    ########################################

    def create_subtask(
        self,
        task_type,
        description,
        input_parameters=None,
        host=None,
        model=None,
        arch=None,
        priority=None,
        important=False,
        execution_space=None,
        inherit_notifications=False,
        tags=None,
        se_tag=None,
        enqueue=True,
        ram=None,
        max_restarts=None,
    ):
        """
        Create sub task. To be used from task code only.

        :param task_type: task type
        :param description: description of the task
        :param input_parameters: task's input parameters, task context by default
        :param host: execute task on selected host only
        :param model: execute task on hosts with selected CPU model only, by default to be taken from the task if set
        :param arch: execute task on hosts with selected OS only, by default to be taken from the task if set
        :param priority: priority for new task, by default equal to priority of the task
        :param important: mask new task as important
        :param execution_space: required disk space in MiB
        :param inherit_notifications: flags to inherit notification settings from the parent task (self)
        :param tags: task tags to be set (list of strings)
        :param se_tag: tag to limit number of simultaneously executing tasks
        :param enqueue: put task to the queue
        :param ram: required ram size in MiB
        :param max_restarts: specifies number of maximum restarts for the subtask in case of temporary error status
        :return: new task object
        """
        context = self.ctx if input_parameters is None else input_parameters
        context.setdefault('GSID', self.ctx.get('__GSID', ''))
        context.setdefault('tasks_archive_resource', self.tasks_archive_resource)
        if arch is None:
            arch = self.arch
        if model is None:
            model = self.model
        if host is None:
            host = self.required_host
        if isinstance(priority, (tuple, list)):
            priority = self.Priority().__setstate__(priority)
        if not isinstance(priority, self.Priority) or priority > self.priority:
            priority = self.priority

        params = {}
        if inherit_notifications:
            notifications = getattr(self, "notifications", None)
            if notifications is not None:
                params['notifications'] = notifications
        if tags is not None:
            params['tags'] = tags
        if se_tag is not None:
            params['se_tag'] = se_tag
        if ram is not None:
            params['ram'] = ram
        if max_restarts is not None:
            params['max_restarts'] = max_restarts

        params = params or None
        task = channel.channel.sandbox.create_task(
            task_type=task_type, description=description, owner=self.owner,
            context=context, parent_task_id=self.id, model=model,
            host=host, arch=arch, priority=priority.__getstate__(), important=important,
            execution_space=execution_space, parameters=params, enqueue=enqueue
        )
        logger.info(
            "Sub-task #%d (type: '%s', host: '%s', model: '%s', arch: '%s', priority: %s) created.",
            task.id, task_type, host, model, arch, priority
        )
        return task

    def wait_time(self, timeout, state=None):
        """
            Перевести задачу в состояние WAIT_CHILD, она будет ожидать timeout секунд, потом перейдёт снова в ENQUEUED

            :param timeout: сколько с секундах находиться в состянии WAIT_CHILD
            :param state: сохраняемое в WAIT_STATE состояние
        """
        timeout = int(timeout)
        if state:
            self.ctx['WAIT_STATE'] = state
        if self.status == self.Status.DRAFT:
            # server-side call from within on_enqueue
            from sandbox.yasandbox.controller import trigger as trigger_controller
            from sandbox.yasandbox.database import mapping
            if trigger_controller.TimeTrigger.create(mapping.TimeTrigger(
                source=self.id,
                time=dt.datetime.utcnow() + dt.timedelta(seconds=timeout),
                activated=True,
            )):
                raise common_errors.WaitTime
        else:
            channel.channel.sandbox.server.wait_time(self.id, timeout)
            raise common_errors.WaitTime

    def wait_task_completed(self, task, state=None):
        """
            Подождать завершения указанной задачи (переход в состояние FINISHED, FAILURE, DELETED)

            Текущая задача переходит в состояние WAIT_CHILD

            :param task: идентификатор или объект задачи
            :param state: сохраняемое в WAIT_STATE состояние

            .. deprecated:: r1567514
                Use :func:`wait_tasks` instead.
        """
        return self.wait_tasks(
            [task],
            (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED),
            True,
            state
        )

    def wait_all_tasks_completed(self, tasks, state=None):
        """
            Подождать завершения всех задач из списка (переход в состояние FINISHED, FAILURE, DELETED)

            Текущая задача переходит в состояние WAIT_CHILD

            :param tasks: список идентификаторов или объектов задач
            :param state: сохраняемое в WAIT_STATE состояние

            .. deprecated:: r1567514
                Use :func:`wait_tasks` instead.
        """
        return self.wait_tasks(
            tasks,
            (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED),
            True,
            state
        )

    def wait_all_tasks_stop_executing(self, tasks, state=None):
        """
            Подождать завершения всех задач из списка (переход в состояние FINISHED, FAILURE, DELETED, UNKNOWN)

            Текущая задача переходит в состояние WAIT_CHILD

            :param tasks: список идентификаторов или объектов задач
            :param state: сохраняемое в WAIT_STATE состояние

            .. deprecated:: r1567514
                Use :func:`wait_tasks` instead.
        """
        return self.wait_tasks(tasks, tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK), True, state)

    def wait_all_tasks_completed_restart_once(self, tasks, state=None):
        """
            Подождать завершения всех задач из списка (переход в состояние FINISHED, FAILURE, DELETED, UNKNOWN)

            Текущая задача переходит в состояние WAIT_CHILD. После того, как все задачи завершились, метод
            должен быть вызван повторно. Если список переданных задач пустой, он наполняется подзадачами таска.
            Если какие-то задач перешли в состояние UNKNOWN, они будут перезапущены. Текущая задача снова переходит
            в состояние WAIT_CHILD и ожидает завершения всех задач из списка. После повторного ожидания, никакие
            задачи перезапущены не будут, независимо от их финального статуса.

            :param tasks: список идентификаторов или объектов задач
            :param state: сохраняемое в WAIT_STATE состояние
        """
        restarter_ctx_key = '__unknown_restarter_for_%s' % self.id
        task_ids = [sandboxapi.Sandbox._to_id(task) for task in tasks]
        if tasks and not (
            self.ctx.get(restarter_ctx_key) and
            self.ctx[restarter_ctx_key] in (task_ids, 'finished')
        ):
            self.ctx[restarter_ctx_key] = task_ids
            self.wait_all_tasks_stop_executing(tasks, state)
        else:
            self.ctx[restarter_ctx_key] = 'finished'
            need_wait = False
            if not tasks:
                tasks = channel.channel.sandbox.list_tasks(parent_id=self.id)
            for task in tasks:
                task_object = channel.channel.sandbox.get_task(task)
                if task_object.new_status in (self.Status.EXCEPTION, self.Status.NO_RES, self.Status.TIMEOUT):
                    need_wait = True
                    channel.channel.sandbox.server.restart_task(task_object.id)
            if need_wait:
                self.wait_all_tasks_completed(tasks)
            else:
                self.ctx[restarter_ctx_key] = None

    def wait_any_task_completed(self, tasks, state=None, wait_for_unknown=False):
        """
            Подождать завершения любой задачи из списка (переход в состояние FINISHED, FAILURE, DELETED, [UNKNOWN])

            Текущая задача переходит в состояние WAIT_CHILD

            :param tasks: список объектов задач или их идентификаторов
            :param state: сохраняемое в WAIT_STATE состояние
            :param wait_for_unknown: ожидать ли перехода в состояние UNKNOWN

            .. deprecated:: r1567514
                Use :func:`wait_tasks` instead.
        """
        statuses = list(self.Status.Group.FINISH)
        if wait_for_unknown:
            statuses.extend(self.Status.Group.BREAK)
        return self.wait_tasks(tasks, statuses, False, state)

    def wait_tasks(self, tasks, statuses, wait_all, state=None):
        """
        Wait for specified task(s) switched into one of statuses specified.

        :param tasks:       Task or list of task objects or identifiers the task will wait for.
        :param statuses:    Status or list of statuses or list of groups to wait for.
        :param wait_all:    Wait for all or any of tasks specified.
        :param state:       Data to be stored in `WAIT_STATE` context key.
        """
        statuses = tuple(sorted(set(it.chain.from_iterable(
            tuple(st) if hasattr(st, "__iter__") else (st,)
            for st in (
                statuses if hasattr(statuses, "__iter__") else (statuses,)
            )
        )) - set(ctt.Status.Group.NONWAITABLE)))
        if not statuses:
            raise ValueError("Empty statuses list to wait for")

        ids = [
            sandboxapi.Sandbox._to_id(task)
            for task in (tasks if hasattr(tasks, "__iter__") else [tasks])
        ]
        logger.info("Wait %s tasks %r for %r.", "all" if wait_all else "any", ids, statuses)
        if state:
            self.ctx['WAIT_STATE'] = state

        if self.status == self.Status.DRAFT:
            # server-side call from within on_enqueue
            from sandbox.yasandbox.controller import trigger as trigger_controller
            from sandbox.yasandbox.database import mapping

            targets = trigger_controller.TaskStatusTrigger.get_not_ready_targets(
                targets=ids, statuses=statuses
            )
            if not targets or (not wait_all and len(ids) != len(targets)):
                raise common_errors.NothingToWait

            if trigger_controller.TaskStatusTrigger.create(mapping.TaskStatusTrigger(
                source=self.id,
                targets=targets,
                statuses=statuses,
                wait_all=wait_all,
                activated=True,
            )):
                raise common_errors.WaitTask
        else:
            try:
                channel.channel.sandbox.server.wait_tasks(self.id, ids, statuses, wait_all)
            except xmlrpclib.Fault as ex:
                if common_errors.NothingToWait.__name__ in ex.faultString:
                    raise common_errors.NothingToWait
                raise
            raise common_errors.WaitTask

    ########################################
    # resource
    ########################################

    def _validate_resource_path(self, resource_path):
        """ Check that resource is placed in the current task directory """
        if not os.path.isabs(resource_path):
            return resource_path

        current_task_dir = self.abs_path()
        real_current_task_dir = os.path.join(os.path.realpath(current_task_dir), '')
        real_resource_path = os.path.realpath(resource_path)
        if not real_resource_path.startswith(real_current_task_dir):
            raise common_errors.TaskError(
                'Cannot create a resource with path {}. '
                'It is not in current task directory {}.'.format(resource_path, current_task_dir)
            )
        return os.path.relpath(real_resource_path, real_current_task_dir)

    def create_resource(self, description, resource_path, resource_type, arch=None, attributes=None, owner=None):
        """
        Create resource

        :param description: description of the created resource
        :param resource_path: path to the resource content, must be relative to the task working directory
        :param resource_type: type of resource as string or resource object
        :param arch: architecture of the resource (linux, freebsd, any)
        :param attributes: dictionary with resource attributes
        :param owner: resource owner
        :return: resource object
        """
        if arch is None:
            arch = self.arch
        resource_type = str(resource_type)
        resource_path = self._validate_resource_path(os.path.normpath(resource_path))
        logger.info('Create resource with type: %s, path: %s,  arch: %s', resource_type, resource_path, arch)
        try:
            resource = self._create_resource(
                description,
                resource_path,
                resource_type,
                arch=arch,
                attrs=attributes,
                owner=owner
            )
        except xmlrpclib.Fault as exc:
            if exc.faultString.startswith('SandboxException: '):
                raise common_errors.TaskFailure(exc.faultString.replace('SandboxException: ', ''))
            else:
                raise

        return sandboxapi.SandboxResource(resource.to_dict())

    def sync_resource(self, resource_id):
        """
            Download resource to current client if it was not already downloaded
            (and touch it in both cases).

            :param resource_id: resource identifier or resource object
            :return: local path to synchronized resource;
                raise SandboxTaskUnknownError exception when sync failed.
        """
        return self.agentr.resource_sync(sandboxapi.Sandbox._to_id(resource_id))

    def _sync_resource_via_synchrophazotron(self, resource):
        """ The method is required to avoid pickling problems after patching of :py:meth:`_sync_resource`. """
        return self.synchrophazotron.sync_resource(resource.id)

    def mark_resource_ready(self, resource_id):
        """
        | Mark resource as **READY**
        | After this operation you cannot modify it

        :param resource_id: resource id

        :return: None
        """
        resource_id = sandboxapi.Sandbox._to_id(resource_id)
        if self.sdk1_agentr_enabled:
            from sandbox.yasandbox.proxy import resource as proxy_resource
            if proxy_resource.Resource.register(resource_id):
                self.agentr.resource_complete(resource_id)
        else:
            from sandbox.yasandbox.manager import resource_manager
            resource = resource_manager.load(resource_id)
            if not resource:
                raise common_errors.TaskFailure('Invalid resource_id: {!r}'.format(resource_id))
            resource.mark_ready()

    def save_parent_task_resource(self, path, resource, save_basename=False, move=False):
        """
            Save `path` as parent task's resource data. The resource should be in `NOT_READY` state.

            :param path:            resource data path
            :param resource:        resource object or ID to be updated
            :param save_basename:   flags to keep resource basename unchanged
            :param move:            flags the resource data should be moved instead of copying
            :return:                path to resource's data
        """
        source_path = os.path.abspath(path)
        resource = channel.channel.rest.get_resource(resource)
        logger.info("%s %r to parent task resource %r", "Move" if move else "Copy", source_path, resource.id)

        if resource.is_ready():
            raise common_errors.TaskError(
                'Cannot save {0} as parent task resource {1}: it is in ready state'.format(source_path, resource.id))

        if resource.task_id != self.parent_id:
            raise common_errors.TaskError(
                'Cannot save {0} as parent task resource {1}: resource task {2} is not parent task for {3}'.format(
                    source_path, resource.id, resource.task_id, self.parent_id))

        if self.sdk1_agentr_enabled:
            resource_path = self.path(resource.file_name)

            if save_basename:
                new_basename = os.path.basename(source_path)
                new_resource_path = os.path.join(os.path.dirname(resource.file_name), new_basename)
                new_resource_path = self._validate_resource_path(new_resource_path)
                channel.channel.rest.server.resource[resource.id] = {"file_name": new_resource_path}
                resource_path = os.path.join(os.path.dirname(resource_path), new_basename)

            if source_path != resource_path:
                (shutil.move if move else paths.copy_path)(source_path, resource_path)

            from sandbox.yasandbox.proxy import resource as proxy_resource
            proxy_resource.Resource.register(resource.id)
            self.agentr.resource_complete(resource.id)

            if source_path == resource_path and not move:
                # agentr moves files and replaces them with a symlink
                # if move=False, we need to copy files back
                new_path = os.path.realpath(resource_path)
                os.unlink(resource_path)
                paths.copy_path(new_path, resource_path)
        else:
            resource_path = resource.path
            if save_basename:
                new_basename = os.path.basename(source_path)
                self.change_resource_basename(resource.id, new_basename)
                # обновляем объект ресурса
                resource_path = os.path.join(os.path.dirname(resource_path), new_basename)

            task_resource_folder = self.get_task_abs_path(resource.task_id)
            if os.path.exists(task_resource_folder):
                logger.info('Parent task folder {0} exists, add +w permissions.'.format(task_resource_folder))
                paths.add_write_permissions_for_path(task_resource_folder)
            else:
                logger.info('Parent task folder {0} does not exist, create it.'.format(task_resource_folder))
                common_fs.create_task_dir(resource.task_id)

            (shutil.move if move else paths.copy_path)(path, resource_path)
            paths.remove_write_permissions_for_path(task_resource_folder)
            self.mark_resource_ready(resource)
            return resource_path

    @staticmethod
    def remove_resource_files(resource_id):
        """
            Locally remove resource's files

            :param resource_id: resource id
        """
        from sandbox.yasandbox.manager import resource_manager
        resource = resource_manager.load(resource_id)
        resource.remove_resource_files()

    def change_resource_basename(self, resource_id, new_basename):
        """
            Assign new basename to existing resources in state NOT_READY.
            Path to folder of resource is saved.

            :param resource_id: resource id
            :param new_basename: new name of resource, must be relative path
            :return: id of the update resource
        """
        from sandbox.yasandbox.manager import resource_manager
        resource = resource_manager.load(resource_id)
        if resource.is_not_ready():
            new_resource_path = os.path.join(os.path.dirname(resource.file_name), new_basename)
            resource.file_name = self._validate_resource_path(new_resource_path)
            resource_manager.update(resource)
        else:
            raise common_errors.TaskFailure(
                'Cannot change file_name for resource {}. Resource must be in NOT_READY state not in {}.'.format(
                    resource_id, resource.state
                )
            )
        return resource_id

    ########################################
    # release
    ########################################

    def create_release(
            self,
            task_id, status='stable',
            subject=None, comments=None, changelog=None,
            addresses_to=None, addresses_cc=None,
            additional_parameters=None
    ):
        """
        Release task with given id.

        :param task_id: id of task to release
        :param status: release status (cancelled, stable, prestable, testing, unstable)
        :param subject: release notification subject
        :param comments: release notification body
        :param changelog: release changelog
        :param addresses_to: list of recipient logins
        :param addresses_cc: list of CC-recipient logins
        :param additional_parameters: dictionary with params to pass to `on_release` method
        :return: release id or None if task releasing failed
        :rtype: int
        """
        task_id = int(task_id)
        additional_parameters = additional_parameters or {}
        additional_parameters.update(author=self.author)
        return channel.channel.sandbox.server.create_release(
            task_id, status,
            subject, comments, changelog, addresses_to, addresses_cc,
            additional_parameters)

    ########################################
    # service
    ########################################

    def is_arch_compatible(self, arch):
        """
            | check for selected arch to compatible with current task arch
            | **any** arch compatible with all

            :param arch: arch to compare with current

            :return: True if compatible or False
        """
        return util.is_arch_compatible(arch, self.arch)

    @staticmethod
    def current_action(action):
        """
        Return a context manager, which will set task's current action on enter and nullify it on exit

        :param action: string with action description
        """

        return sdk2.helpers.ProgressMeter(action)

    @staticmethod
    def get_vault_data(owner_or_name, name=None):
        """
        Get vault item data by owner and name

        :param owner_or_name: owner or name of vault item
        :param name: name of vault item
        :return str: vault item data
        """
        return sdk2.Vault.data(owner_or_name, name)  # vault_key is set in executor on all possible stages

    @property
    def memoize_stage(self):
        return MemoizeCreator(self)

    def suspend(self):
        """ Suspend the current task. """
        logging.info("Suspending current task.")
        rest = common_rest.Client()
        res = rest.batch.tasks.suspend.update([self.id])
        if res[0]["status"] != "SUCCESS":
            raise common_errors.TaskError("Unable to suspend the task: " + res[0]["message"])
        try:
            logging.debug("Waking up client via socket at port %s", common_config.Registry().client.port)
            socket.create_connection(("127.0.0.1", common_config.Registry().client.port)).close()
        except Exception as ex:
            logging.warning("Error waking up client instance: %s", ex)

        logging.info("Waiting for suspend")
        common_itertools.progressive_waiter(
            .1, 30, int(self.ctx["kill_timeout"]),
            lambda: rest.task[self.id][:]["status"] not in (ctt.Status.SUSPENDING, ctt.Status.SUSPENDED)
        )


class BuildForAllMode(SandboxTask):
    """
    Base class for all build task types that need to run on all platforms.
    They will be created as child tasks of BuildMeForAll task on each platform.
    Parent task will prepare resources according to the info provided by `required_resources`
    method and pass this resources to child tasks context. So child tasks after
    building this resources should save them to parent using `save_parent_task_resource` method
    """

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self.ctx["_prepared_resources"] = []

    @classmethod
    def required_resources(cls):
        """
        Information about resources that will be build by child tasks.

        :return: list of tuples with resource type and description.
        """
        raise NotImplementedError

    def prepared_resources(self):
        return self.ctx["_prepared_resources"]


class ForcedBackupTask(SandboxTask):
    """
    Base class for tasks that require forced backup functionality.

    Use mark_resource_ready with force_backup=True to make the task wait until the backup is finished.
    """
    FORCE_BKUP_CTX = "__force_backup_tasks"

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self._on_execute = self.on_execute
        self.on_execute = self._real_on_execute

    def mark_resource_ready(self, resource_id, force_backup=False):
        """ force_backup=True makes the task wait until the resource is backed up """
        SandboxTask.mark_resource_ready(self, resource_id)
        if force_backup:
            rest = common_rest.Client()
            backup_id = rest.resource[resource_id].backup().get('id')
            if backup_id:
                self.ctx[self.FORCE_BKUP_CTX] = self.ctx.get(self.FORCE_BKUP_CTX, []) + [backup_id]

    def on_backup_completed(self):
        """ to be called after all backup tasks have succeeded """

    def _process_force_backup_tasks(self):
        finish_statuses = tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        force_backup_tasks = self.ctx.get(self.FORCE_BKUP_CTX)
        logger.debug("Force backup tasks: %s", force_backup_tasks)
        rest = common_rest.Client()
        for task_id in force_backup_tasks:
            task = rest.task[task_id][:]
            if task["status"] not in finish_statuses:
                # oops, we didn't actually wait for all backup tasks to finish
                # this might happen when task's post-processing fails and it goes to TEMPORARY
                self.wait_tasks(
                    tasks=force_backup_tasks,
                    statuses=finish_statuses,
                    wait_all=True,
                )
            if task["status"] not in ctt.Status.Group.SUCCEED:
                logging.warning("Backup failed: task #%s has status %r", task["id"], task["status"])
                # raise common_errors.TaskFailure(
                #     "Backup failed: task #{} has status {}".format(task["id"], task["status"])
                # )  # FIXME: SANDBOX-5167: Temporary muted error on backup failure
        del self.ctx[self.FORCE_BKUP_CTX]
        self.on_backup_completed()

    def _real_on_execute(self):
        if self.ctx.get(self.FORCE_BKUP_CTX):
            self._process_force_backup_tasks()
        else:
            self._on_execute()
            force_bkup_tasks = self.ctx.get(self.FORCE_BKUP_CTX)
            if force_bkup_tasks:
                self.wait_tasks(
                    tasks=force_bkup_tasks,
                    statuses=common_itertools.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK),
                    wait_all=True,
                )
