from __future__ import print_function, unicode_literals

import argparse
import contextlib
import datetime as dt
import hashlib
import json
import os
import shutil
import sys
import tempfile
import time

from library.python import resource
from library.python.sfx import extract


def dir_hash(path):
    h = hashlib.sha1()

    file_paths = [
        os.path.join(dir_path, filename)
        for dir_path, _, files in os.walk(path)
        for filename in files
    ]
    file_paths.sort()

    for file_path in file_paths:
        with open(file_path, "rb") as f:
            h.update(f.read())

    return h.hexdigest()


@contextlib.contextmanager
def mkdtemp():
    tmp_path = tempfile.mkdtemp()
    yield tmp_path
    shutil.rmtree(tmp_path)


class Result(object):
    def __init__(self, path):
        self.path = path

    def save(self, obj):
        with open(os.path.join(self.path, "result.json"), "w") as f:
            json.dump(obj, f)


def main():
    parser = argparse.ArgumentParser(prog="make-pip")
    parser.add_argument("--last-hash", dest="last_hash", type=str, help="Hash of the previous release", required=False)
    parser.add_argument("--post", dest="post_version", type=str, help="Custom post-version", required=False)
    args = parser.parse_args()

    destination = os.path.abspath(os.path.curdir)
    result = Result(destination)

    with mkdtemp() as tmp_path:
        extract.extract_sources(tmp_path)

        package_path = os.path.join(tmp_path, "__pkg__")
        os.makedirs(os.path.join(package_path, "sandbox"))

        shutil.move(os.path.join(tmp_path, "sandbox/common"), os.path.join(package_path, "sandbox/common"))

        # Avoid extra dependencies. `common.config` is not needed for external distribution.
        shutil.rmtree(os.path.join(package_path, "sandbox/common/config"))

        dh = dir_hash(package_path)

        if dh == args.last_hash:
            result.save({"status": "unchanged"})
            return

        # List immediate subdirectories of `sandbox_common`
        modules = next(os.walk(os.path.join(package_path, "sandbox/common")))[1]
        modules.sort()

        with open(os.path.join(package_path, "sandbox/__init__.py"), "wb"):
            pass

        with open(os.path.join(package_path, "sandbox/common/__init__.py"), "w") as f:
            f.write("from . import (\n{}\n)".format(
                ",\n".join("    " + m for m in modules))
            )

        if args.post_version:
            post_version = args.post_version
        else:
            post_version = str(int(time.time()))

        version = "{}.post{}".format(
            dt.date.today().strftime('%Y.%-m.%-d'),  # month and day withon leading zeroes
            post_version
        )

        with open(os.path.join(package_path, "setup.py"), "wb") as f:
            f.write(resource.find("setup.py").replace("{version}", version))

        os.chdir(package_path)

        from distutils.core import run_setup
        from wheel import bdist_wheel
        sys.modules["distutils.command.bdist_wheel"] = bdist_wheel

        run_setup(script_name="./setup.py", script_args=["sdist"])
        run_setup(script_name="./setup.py", script_args=["bdist_wheel", "--universal"])

        dist_dir = os.path.join(package_path, "dist")
        packages = next(os.walk(dist_dir))[2]
        for package in packages:
            shutil.copy2(os.path.join(dist_dir, package), os.path.join(destination, package))

        result.save({
            "status": "ok",
            "hash": dh,
            "version": version,
            "files": packages,
        })
